/**
 * TextRad Dictation Extension - Direct Transcription Service
 * 
 * This service bypasses the Firebase client libraries and directly
 * calls the TextRad server-side transcription endpoint.
 */

export class DirectTranscriptionService {
  constructor() {
    // TextRad transcription endpoint - HTTP endpoint for CORS-enabled function
    this.transcriptionEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/directTranscribeHttp';

    // Event callbacks
    this.onTranscriptionStart = null;
    this.onTranscriptionComplete = null;
    this.onTranscriptionError = null;
    this.onTranscriptionProgress = null;

    // Store current transcription for appending when using 'Record More'
    this.currentTranscription = '';

    // Service warmup tracking
    this.lastWarmupTime = 0;
    this.warmupInterval = 5 * 60 * 1000; // 5 minutes
    this.isWarmingUp = false;

    // Start periodic warmup
    this.startPeriodicWarmup();
  }
  
  /**
   * Transcribe audio using direct API call
   * @param {Blob} audioBlob - The recorded audio blob
   */
  async transcribeAudio(audioBlob) {
    console.log('Starting direct transcription service');

    // Call start callback
    if (typeof this.onTranscriptionStart === 'function') {
      this.onTranscriptionStart();
    }

    try {
      // Ensure service is warmed up before proceeding
      await this.ensureWarmService();

      // Convert audio to base64
      const base64Audio = await this.blobToBase64(audioBlob);
      console.log("Audio converted to base64, size:", Math.round(base64Audio.length / 1024), "KB");

      if (typeof this.onTranscriptionProgress === 'function') {
        this.onTranscriptionProgress(0.3);
      }

      // Call the transcription API directly
      const response = await this.callTranscriptionAPI(base64Audio);

      if (typeof this.onTranscriptionProgress === 'function') {
        this.onTranscriptionProgress(0.9);
      }

      // Process the response
      if (!response || !response.text) {
        throw new Error('Invalid response from transcription service');
      }

      const transcription = response.text;

      // We'll let the popup.js handle appending to avoid duplication
      // Just store the current transcription property for reference

      // Call completion callback with ONLY the new transcription
      // The parent component will handle appending if needed
      if (typeof this.onTranscriptionComplete === 'function') {
        this.onTranscriptionComplete(transcription);
      }

      return transcription;
    } catch (error) {
      console.error('Transcription error:', error);

      // Call error callback
      if (typeof this.onTranscriptionError === 'function') {
        this.onTranscriptionError(error);
      }

      throw error;
    }
  }
  
  /**
   * Call the TextRad transcription API directly with retry logic for cold starts
   * @param {string} base64Audio - Base64-encoded audio data
   * @returns {Promise<Object>} - Transcription response
   */
  async callTranscriptionAPI(base64Audio) {
    // Prepare the request - match the exact format expected by your function
    const requestData = {
      data: {
        audioData: base64Audio,
        userId: 'extension-user', // Anonymous user ID
        source: 'extension' // Identify source as extension
      }
    };

    console.log('Calling API with data structure:', Object.keys(requestData));

    const maxRetries = 3;
    const baseDelay = 2000; // Start with 2 seconds

    for (let attempt = 0; attempt <= maxRetries; attempt++) {
      try {
        // Calculate timeout - longer for first attempt to handle cold starts
        const timeout = attempt === 0 ? 45000 : 30000; // 45s for first attempt, 30s for retries

        console.log(`Transcription attempt ${attempt + 1}/${maxRetries + 1}, timeout: ${timeout}ms`);

        // Create abort controller for timeout
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), timeout);

        try {
          // Make the request - Firebase Functions expect a specific format
          const response = await fetch(this.transcriptionEndpoint, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-TextRad-Extension': 'true',
              'Origin': 'chrome-extension://textrad-extension'
            },
            body: JSON.stringify(requestData),
            signal: controller.signal
          });

          clearTimeout(timeoutId);

          // Check for HTTP errors
          if (!response.ok) {
            // 500/502/503/504 errors are often cold start related - retry them
            if (response.status >= 500 && response.status <= 504 && attempt < maxRetries) {
              console.warn(`Server error ${response.status} on attempt ${attempt + 1}, retrying...`);
              throw new Error(`HTTP error ${response.status}: ${response.statusText} (retryable)`);
            }
            throw new Error(`HTTP error ${response.status}: ${response.statusText}`);
          }

          // Parse the response
          const data = await response.json();
          console.log(`Transcription successful on attempt ${attempt + 1}`);
          return data.result || data; // Handle response format from HTTP function

        } catch (fetchError) {
          clearTimeout(timeoutId);
          throw fetchError;
        }

      } catch (error) {
        console.error(`API call attempt ${attempt + 1} failed:`, error);

        // Check if this is a retryable error
        const isRetryable =
          error.name === 'AbortError' || // Timeout
          error.message.includes('Failed to fetch') || // Network error
          error.message.includes('500') || // Server error
          error.message.includes('502') || // Bad gateway
          error.message.includes('503') || // Service unavailable
          error.message.includes('504') || // Gateway timeout
          error.message.includes('retryable');

        // If this is the last attempt or error is not retryable, throw
        if (attempt === maxRetries || !isRetryable) {
          if (error.message.includes('Failed to fetch') || error.name === 'AbortError') {
            console.warn('Network error or timeout accessing transcription API');
            throw new Error('Network error or timeout occurred. This often happens after periods of inactivity. Please check your connection and try again.');
          }
          throw error;
        }

        // Calculate exponential backoff delay
        const delay = baseDelay * Math.pow(2, attempt);
        console.log(`Retrying in ${delay}ms...`);

        // Update progress to show retry
        if (typeof this.onTranscriptionProgress === 'function') {
          this.onTranscriptionProgress(0.1 + (attempt * 0.1));
        }

        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }
  
  /**
   * Convert Blob to Base64 string
   * @param {Blob} blob - The blob to convert
   * @returns {Promise<string>} Base64 string
   */
  blobToBase64(blob) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onloadend = () => {
        // Get the base64 string (remove the data URL prefix)
        const base64String = reader.result.split(',')[1];
        resolve(base64String);
      };
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    });
  }
  
  /**
   * Set the current transcription (used for appending with 'Record More')
   * @param {string} text - The transcription text to store
   */
  setCurrentTranscription(text) {
    this.currentTranscription = text || '';
    console.log('Stored current transcription for appending, length:', this.currentTranscription.length);
  }
  
  /**
   * Reset the current transcription
   */
  resetTranscription() {
    this.currentTranscription = '';
    console.log('Reset current transcription');
  }

  /**
   * Start periodic service warmup to prevent cold starts
   */
  startPeriodicWarmup() {
    // Warmup immediately if it's been a while
    const now = Date.now();
    if (now - this.lastWarmupTime > this.warmupInterval) {
      this.warmupService();
    }

    // Set up periodic warmup
    setInterval(() => {
      const timeSinceLastWarmup = Date.now() - this.lastWarmupTime;
      if (timeSinceLastWarmup >= this.warmupInterval) {
        this.warmupService();
      }
    }, this.warmupInterval);
  }

  /**
   * Warm up the service to prevent cold starts
   */
  async warmupService() {
    if (this.isWarmingUp) return;

    try {
      this.isWarmingUp = true;
      console.log('Warming up transcription service...');

      // Send a minimal ping request to keep the function warm
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout for warmup

      try {
        const response = await fetch(this.transcriptionEndpoint, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-TextRad-Extension': 'true',
            'X-TextRad-Warmup': 'true', // Special header to indicate warmup
            'Origin': 'chrome-extension://textrad-extension'
          },
          body: JSON.stringify({
            data: {
              audioData: '', // Empty data for warmup
              userId: 'warmup-user',
              source: 'extension-warmup'
            }
          }),
          signal: controller.signal
        });

        clearTimeout(timeoutId);
        this.lastWarmupTime = Date.now();
        console.log(`Service warmup completed, status: ${response.status}`);

      } catch (warmupError) {
        clearTimeout(timeoutId);
        // Warmup failures are non-critical, just log them
        console.log('Service warmup failed (non-critical):', warmupError.message);
        this.lastWarmupTime = Date.now(); // Still update time to avoid rapid retries
      }

    } catch (error) {
      console.log('Warmup error (non-critical):', error);
    } finally {
      this.isWarmingUp = false;
    }
  }

  /**
   * Ensure service is warmed up before critical operations
   */
  async ensureWarmService() {
    const timeSinceLastWarmup = Date.now() - this.lastWarmupTime;
    if (timeSinceLastWarmup > this.warmupInterval && !this.isWarmingUp) {
      await this.warmupService();
    }
  }
}