/**
 * TextRad Dictation Extension - Storage Manager
 * 
 * Handles storage of extension state, settings, and temporary data
 */

export class StorageManager {
  constructor() {
    // Default settings
    this.defaultSettings = {
      autoSend: false,
      visualizerEnabled: true,
      silenceTimeout: 40000,
      lastUpdated: Date.now()
    };
  }
  
  /**
   * Initialize storage
   */
  async initialize() {
    // Check if settings exist, create defaults if not
    const settings = await this.getSettings();
    if (!settings || Object.keys(settings).length === 0) {
      await this.setSettings(this.defaultSettings);
    }
  }
  
  /**
   * Get settings from storage
   */
  async getSettings() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['settings'], (result) => {
        resolve(result.settings || this.defaultSettings);
      });
    });
  }
  
  /**
   * Save settings to storage
   */
  async setSettings(settings) {
    return new Promise((resolve) => {
      // Add timestamp
      settings.lastUpdated = Date.now();
      
      chrome.storage.local.set({ settings }, () => {
        resolve(true);
      });
    });
  }
  
  /**
   * Get recording in progress flag
   */
  async getRecordingInProgress() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['recordingInProgress'], (result) => {
        resolve(result.recordingInProgress || false);
      });
    });
  }
  
  /**
   * Set recording in progress flag
   */
  async setRecordingInProgress(inProgress) {
    return new Promise((resolve) => {
      chrome.storage.local.set({ recordingInProgress: inProgress }, () => {
        resolve(true);
      });
    });
  }
  
  /**
   * Get last audio blob
   * Note: This actually returns a reference, not the actual blob
   * The actual blob is stored in IndexedDB for better performance
   */
  async getLastAudioBlob() {
    // Audio blobs are too large for chrome.storage
    // We store a reference in storage and use IndexedDB for the actual blob
    // This is a placeholder for the actual implementation
    return null;
  }
  
  /**
   * Set last audio blob
   * Note: This actually stores a reference, not the actual blob
   * The actual blob is stored in IndexedDB for better performance
   */
  async setLastAudioBlob(audioBlob) {
    // Audio blobs are too large for chrome.storage
    // We store a reference in storage and use IndexedDB for the actual blob
    // This is a placeholder for the actual implementation
    return new Promise((resolve) => {
      // Store metadata about the blob
      const metadata = {
        size: audioBlob.size,
        type: audioBlob.type,
        timestamp: Date.now()
      };
      
      chrome.storage.local.set({ lastAudioBlobMetadata: metadata }, () => {
        resolve(true);
      });
    });
  }
  
  /**
   * Get last transcription
   */
  async getLastTranscription() {
    return new Promise((resolve) => {
      chrome.storage.local.get(['lastTranscription'], (result) => {
        resolve(result.lastTranscription || null);
      });
    });
  }
  
  /**
   * Set last transcription
   */
  async setLastTranscription(transcription) {
    return new Promise((resolve) => {
      chrome.storage.local.set({
        lastTranscription: transcription,
        lastTranscriptionTimestamp: Date.now()
      }, () => {
        resolve(true);
      });
    });
  }
  
  /**
   * Clear temporary data
   */
  async clearTemporaryData() {
    return new Promise((resolve) => {
      chrome.storage.local.remove([
        'recordingInProgress',
        'lastAudioBlobMetadata',
        'lastTranscription',
        'lastTranscriptionTimestamp'
      ], () => {
        resolve(true);
      });
    });
  }
}