/**
 * TextRad Dictation Extension - Direct Services
 *
 * This service bypasses the Firebase client libraries and directly
 * calls the TextRad server-side endpoints for transcription and report generation.
 */

export class DirectTranscriptionService {
  constructor() {
    // TextRad transcription endpoint - HTTP endpoint for CORS-enabled function
    this.transcriptionEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/directTranscribeHttp';

    // TextRad report generation endpoint - HTTP endpoint for CORS-enabled function
    this.reportGenerationEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/directGenerateReportHttp';

    // Event callbacks for transcription
    this.onTranscriptionStart = null;
    this.onTranscriptionComplete = null;
    this.onTranscriptionError = null;
    this.onTranscriptionProgress = null;

    // Event callbacks for report generation
    this.onReportGenerationStart = null;
    this.onReportGenerationComplete = null;
    this.onReportGenerationError = null;
    this.onReportGenerationProgress = null;

    // Store current transcription for appending when using 'Record More'
    this.currentTranscription = '';

    // Service warmup tracking
    this.lastWarmupTime = 0;
    this.warmupInterval = 5 * 60 * 1000; // 5 minutes
    this.isWarmingUp = false;

    // Start periodic warmup
    this.startPeriodicWarmup();
  }

  /**
   * Transcribe audio using direct API call
   * @param {Blob} audioBlob - The recorded audio blob
   */
  async transcribeAudio(audioBlob) {
    console.log('Starting direct transcription service');

    // Call start callback
    if (typeof this.onTranscriptionStart === 'function') {
      this.onTranscriptionStart();
    }

    try {
      // Ensure service is warmed up before proceeding
      await this.ensureWarmService();

      // Convert audio to base64
      const base64Audio = await this.blobToBase64(audioBlob);
      console.log("Audio converted to base64, size:", Math.round(base64Audio.length / 1024), "KB");

      if (typeof this.onTranscriptionProgress === 'function') {
        this.onTranscriptionProgress(0.3);
      }

      // Call the transcription API directly
      const response = await this.callTranscriptionAPI(base64Audio);

      if (typeof this.onTranscriptionProgress === 'function') {
        this.onTranscriptionProgress(0.9);
      }

      // Process the response
      if (!response || !response.text) {
        throw new Error('Invalid response from transcription service');
      }

      const transcription = response.text;

      // We'll let the popup.js handle appending to avoid duplication
      // Just store the current transcription property for reference

      // Call completion callback with ONLY the new transcription
      // The parent component will handle appending if needed
      if (typeof this.onTranscriptionComplete === 'function') {
        this.onTranscriptionComplete(transcription);
      }

      return transcription;
    } catch (error) {
      console.error('Transcription error:', error);

      // Call error callback
      if (typeof this.onTranscriptionError === 'function') {
        this.onTranscriptionError(error);
      }

      throw error;
    }
  }

  /**
   * Call the TextRad transcription API directly with retry logic for cold starts
   * @param {string} base64Audio - Base64-encoded audio data
   * @returns {Promise<Object>} - Transcription response
   */
  async callTranscriptionAPI(base64Audio) {
    // Get user's transcription provider preference
    const settings = await chrome.storage.sync.get(['transcriptionProvider']);
    const transcriptionProvider = settings.transcriptionProvider || 'auto';
    console.log('📋 Using transcription provider setting:', transcriptionProvider);

    // Prepare the request - match the exact format expected by your function
    const requestData = {
      data: {
        audioData: base64Audio,
        userId: 'extension-user', // Anonymous user ID
        source: 'extension' // Identify source as extension
      }
    };

    console.log('Calling API with data structure:', Object.keys(requestData));

    // Endpoints
    const openAIEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/directTranscribeHttp';
    const elevenLabsEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/elevenlabsTranscribeBackup';

    // CASE 1: Explicit ElevenLabs preference
    if (transcriptionProvider === 'elevenlabs') {
      console.log('🎙️ Using ElevenLabs Scribe v1 (user preference only)');
      this.transcriptionEndpoint = elevenLabsEndpoint;
      return this._makeRequestWithRetries(elevenLabsEndpoint, requestData, 'elevenlabs');
    }

    // CASE 2: Auto / Default (OpenAI with Failover)
    console.log('🤖 Using OpenAI gpt-4o-mini-transcribe (primary)');
    this.transcriptionEndpoint = openAIEndpoint;

    try {
      // Try Primary (OpenAI)
      const result = await this._makeRequestWithRetries(openAIEndpoint, requestData, 'openai');
      return result;
    } catch (primaryError) {
      console.log('⚠️ Primary OpenAI failed, attempting automatic failover to ElevenLabs...', primaryError.message);

      // Try Backup (ElevenLabs)
      try {
        console.log('🔄 Automatic Failover: Trying ElevenLabs Scribe v1...');
        const backupResult = await this._makeRequestWithRetries(elevenLabsEndpoint, requestData, 'elevenlabs-backup');

        // Log success of failover
        console.log('✅ Automatic Failover successful!');
        return backupResult;
      } catch (backupError) {
        console.error('❌ Automatic Failover failed:', backupError);
        // Throw combined error
        throw new Error(`Transcription failed. Primary: ${primaryError.message}. Backup: ${backupError.message}`);
      }
    }
  }

  /**
   * Helper method to make API requests with retry logic
   * @param {string} endpoint - The API endpoint URL
   * @param {Object} requestData - The request payload
   * @param {string} providerName - Name of the provider for logging
   * @returns {Promise<Object>} - The API response
   */
  async _makeRequestWithRetries(endpoint, requestData, providerName) {
    const maxRetries = 2; // Reduced slightly as we now have failover (2 retries = 3 total attempts per provider)
    const baseDelay = 2000;

    for (let attempt = 0; attempt <= maxRetries; attempt++) {
      try {
        // Calculate timeout - longer for first attempt to handle cold starts
        // Increased to 120s to match Web App and prevent premature failover to backup service
        const timeout = attempt === 0 ? 120000 : 90000;

        console.log(`[${providerName}] Attempt ${attempt + 1}/${maxRetries + 1}, timeout: ${timeout}ms`);

        // Create abort controller for timeout
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), timeout);

        try {
          const response = await fetch(endpoint, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-TextRad-Extension': 'true',
              'Origin': 'chrome-extension://textrad-extension'
            },
            body: JSON.stringify(requestData),
            signal: controller.signal
          });

          clearTimeout(timeoutId);

          if (!response.ok) {
            // Retry on server errors
            if (response.status >= 500 && response.status <= 504 && attempt < maxRetries) {
              console.warn(`[${providerName}] Server error ${response.status} on attempt ${attempt + 1}, retrying...`);
              throw new Error(`HTTP error ${response.status}: ${response.statusText} (retryable)`);
            }
            throw new Error(`HTTP error ${response.status}: ${response.statusText}`);
          }

          const data = await response.json();
          console.log(`[${providerName}] Transcription successful on attempt ${attempt + 1}`);

          const result = data.result || data;

          // Tag the result with the actual provider used, if not already present
          if (result && !result.provider) {
            result.provider = providerName;
          }

          return result;

        } catch (fetchError) {
          clearTimeout(timeoutId);
          throw fetchError;
        }

      } catch (error) {
        console.error(`[${providerName}] Attempt ${attempt + 1} failed:`, error.message);

        const isRetryable =
          error.name === 'AbortError' ||
          error.message.includes('Failed to fetch') ||
          error.message.includes('500') ||
          error.message.includes('502') ||
          error.message.includes('503') ||
          error.message.includes('504') ||
          error.message.includes('retryable');

        if (attempt === maxRetries || !isRetryable) {
          throw error;
        }

        const delay = baseDelay * Math.pow(2, attempt);
        console.log(`[${providerName}] Retrying in ${delay}ms...`);

        if (typeof this.onTranscriptionProgress === 'function') {
          // Just a small visual bump to show liveliness
          // Don't modify main progress too much or it looks jerky
          try { this.onTranscriptionProgress(0.5); } catch (e) { }
        }

        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }

  /**
   * Convert Blob to Base64 string
   * @param {Blob} blob - The blob to convert
   * @returns {Promise<string>} Base64 string
   */
  blobToBase64(blob) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onloadend = () => {
        // Get the base64 string (remove the data URL prefix)
        const base64String = reader.result.split(',')[1];
        resolve(base64String);
      };
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    });
  }

  /**
   * Set the current transcription (used for appending with 'Record More')
   * @param {string} text - The transcription text to store
   */
  setCurrentTranscription(text) {
    this.currentTranscription = text || '';
    console.log('Stored current transcription for appending, length:', this.currentTranscription.length);
  }

  /**
   * Reset the current transcription
   */
  resetTranscription() {
    this.currentTranscription = '';
    console.log('Reset current transcription');
  }

  /**
   * Start periodic service warmup to prevent cold starts
   */
  startPeriodicWarmup() {
    // Warmup immediately if it's been a while
    const now = Date.now();
    if (now - this.lastWarmupTime > this.warmupInterval) {
      this.warmupService();
    }

    // Set up periodic warmup
    setInterval(() => {
      const timeSinceLastWarmup = Date.now() - this.lastWarmupTime;
      if (timeSinceLastWarmup >= this.warmupInterval) {
        this.warmupService();
      }
    }, this.warmupInterval);
  }

  /**
   * Warm up the service to prevent cold starts
   */
  async warmupService() {
    if (this.isWarmingUp) return;

    try {
      this.isWarmingUp = true;
      console.log('Warming up transcription service...');

      // Send a minimal ping request to keep the function warm
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout for warmup

      try {
        const response = await fetch(this.transcriptionEndpoint, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-TextRad-Extension': 'true',
            'X-TextRad-Warmup': 'true', // Special header to indicate warmup
            'Origin': 'chrome-extension://textrad-extension'
          },
          body: JSON.stringify({
            data: {
              audioData: '', // Empty data for warmup
              userId: 'warmup-user',
              source: 'extension-warmup'
            }
          }),
          signal: controller.signal
        });

        clearTimeout(timeoutId);
        this.lastWarmupTime = Date.now();
        console.log(`Service warmup completed, status: ${response.status}`);

      } catch (warmupError) {
        clearTimeout(timeoutId);
        // Warmup failures are non-critical, just log them
        console.log('Service warmup failed (non-critical):', warmupError.message);
        this.lastWarmupTime = Date.now(); // Still update time to avoid rapid retries
      }

    } catch (error) {
      console.log('Warmup error (non-critical):', error);
    } finally {
      this.isWarmingUp = false;
    }
  }

  /**
   * Ensure service is warmed up before critical operations
   */
  async ensureWarmService() {
    const timeSinceLastWarmup = Date.now() - this.lastWarmupTime;
    if (timeSinceLastWarmup > this.warmupInterval && !this.isWarmingUp) {
      await this.warmupService();
    }
  }

  /**
   * Generate a quick report without format template - AI determines structure
   * This matches exactly how the web app handles quick reports
   * @param {string} transcription - The transcribed text
   * @param {string} userId - The user ID (optional, defaults to 'extension-user')
   */
  async generateQuickReport(transcription, userId = 'extension-user') {
    console.log('🚀 QUICK REPORT: Starting quick report generation');
    console.log('🚀 QUICK REPORT: Transcription length:', transcription?.length || 'undefined');
    console.log('🚀 QUICK REPORT: UserId:', userId);

    // Call the same generateReport method but with null format parameters
    // This exactly matches how the web app handles quick reports
    const result = await this.generateReport(
      transcription,
      null, // formatName: null (AI determines structure)
      null, // formatTemplate: null
      userId
    );

    console.log('🚀 QUICK REPORT: Generation completed');
    return result;
  }

  /**
   * Generate report using direct API call
   * @param {string} transcription - The transcribed text
   * @param {string} formatName - The name of the format
   * @param {string} formatTemplate - The template content
   * @param {string} userId - The user ID (optional, defaults to 'extension-user')
   */
  async generateReport(transcription, formatName, formatTemplate, userId = 'extension-user') {
    console.log('📊 GENERATE REPORT: Starting direct report generation service');
    console.log('📊 GENERATE REPORT: Input parameters:', {
      transcriptionLength: transcription?.length || 'undefined',
      formatName: formatName,
      formatTemplate: formatTemplate?.length ? `${formatTemplate.length} chars` : formatTemplate,
      userId: userId
    });

    // Call start callback
    if (typeof this.onReportGenerationStart === 'function') {
      this.onReportGenerationStart();
    }

    try {
      // Ensure service is warmed up before proceeding
      await this.ensureWarmService();

      console.log("Report generation data:", {
        transcriptionLength: transcription?.length || 0,
        formatName,
        templateLength: formatTemplate?.length || 0,
        userId
      });

      if (typeof this.onReportGenerationProgress === 'function') {
        this.onReportGenerationProgress(0.3);
      }

      // Call the report generation API directly
      const response = await this.callReportGenerationAPI(transcription, formatName, formatTemplate, userId);

      if (typeof this.onReportGenerationProgress === 'function') {
        this.onReportGenerationProgress(0.9);
      }

      // Process the response
      if (!response || !response.report) {
        throw new Error('Invalid response from report generation service');
      }

      const reportData = {
        report: response.report,
        tokenUsage: response.tokenUsage,
        reportId: response.reportId,
        provider: response.provider || 'openai-gpt5-direct',
        model: response.model || 'gpt-5-chat-latest'
      };

      // Call completion callback with the report data
      if (typeof this.onReportGenerationComplete === 'function') {
        this.onReportGenerationComplete(reportData);
      }

      return reportData;
    } catch (error) {
      console.error('Report generation error:', error);

      // Call error callback
      if (typeof this.onReportGenerationError === 'function') {
        this.onReportGenerationError(error);
      }

      throw error;
    }
  }

  /**
   * Call the TextRad report generation API directly with retry logic
   * @param {string} transcription - The transcribed text
   * @param {string} formatName - The name of the format
   * @param {string} formatTemplate - The template content
   * @param {string} userId - The user ID
   * @returns {Promise<Object>} - Report generation response
   */
  async callReportGenerationAPI(transcription, formatName, formatTemplate, userId) {
    // Prepare the request - match exactly how the web app calls the cloud function
    // Always include formatName and formatTemplate fields (now supported by HTTP endpoint)
    const requestData = {
      data: {
        transcription: transcription || '',
        formatName: formatName,        // Always include (null for quick reports)
        formatTemplate: formatTemplate, // Always include (null for quick reports)
        userId: userId || 'extension-user',
        source: 'extension' // Identify source as extension
      }
    };

    // Log the actual values being sent for debugging
    console.log('Request parameters:', {
      transcription: typeof transcription,
      formatName: formatName,
      formatTemplate: formatTemplate,
      userId: userId,
      transcriptionLength: transcription?.length || 0
    });

    console.log('Calling report generation API with data structure:', Object.keys(requestData));

    const maxRetries = 3;
    const baseDelay = 2000; // Start with 2 seconds

    for (let attempt = 0; attempt <= maxRetries; attempt++) {
      try {
        // Calculate timeout - longer for report generation
        const timeout = attempt === 0 ? 300000 : 240000; // 5 minutes for first attempt, 4 minutes for retries

        console.log(`Report generation attempt ${attempt + 1}/${maxRetries + 1}, timeout: ${timeout}ms`);

        // Create abort controller for timeout
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), timeout);

        try {
          // Log the request we're about to make for debugging
          console.log('Making request to:', this.reportGenerationEndpoint);
          console.log('Request data:', JSON.stringify(requestData, null, 2));

          // Make the request - Firebase Functions expect a specific format
          const response = await fetch(this.reportGenerationEndpoint, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-TextRad-Extension': 'true',
              'Origin': 'chrome-extension://textrad-extension'
            },
            body: JSON.stringify(requestData),
            signal: controller.signal
          });

          clearTimeout(timeoutId);

          // Check for HTTP errors
          if (!response.ok) {
            // Get detailed error information for debugging
            let errorBody = '';
            try {
              errorBody = await response.text();
              console.error(`HTTP ${response.status} Response Body:`, errorBody);
            } catch (e) {
              console.error('Could not read response body');
            }

            // 500/502/503/504 errors are often cold start related - retry them
            if (response.status >= 500 && response.status <= 504 && attempt < maxRetries) {
              console.warn(`Server error ${response.status} on attempt ${attempt + 1}, retrying...`);
              throw new Error(`HTTP error ${response.status}: ${response.statusText} (retryable)`);
            }
            throw new Error(`HTTP error ${response.status}: ${response.statusText}. Response: ${errorBody}`);
          }

          // Parse the response
          const data = await response.json();
          console.log(`Report generation successful on attempt ${attempt + 1}`);
          return data.result || data; // Handle response format from HTTP function

        } catch (fetchError) {
          clearTimeout(timeoutId);
          throw fetchError;
        }

      } catch (error) {
        console.error(`Report generation API call attempt ${attempt + 1} failed:`, error);

        // Check if this is a retryable error
        const isRetryable =
          error.name === 'AbortError' || // Timeout
          error.message.includes('Failed to fetch') || // Network error
          error.message.includes('500') || // Server error
          error.message.includes('502') || // Bad gateway
          error.message.includes('503') || // Service unavailable
          error.message.includes('504') || // Gateway timeout
          error.message.includes('retryable');

        // If this is the last attempt or error is not retryable, throw
        if (attempt === maxRetries || !isRetryable) {
          if (error.message.includes('Failed to fetch') || error.name === 'AbortError') {
            console.warn('Network error or timeout accessing report generation API');
            throw new Error('Network error or timeout occurred. This often happens after periods of inactivity. Please check your connection and try again.');
          }
          throw error;
        }

        // Calculate exponential backoff delay
        const delay = baseDelay * Math.pow(2, attempt);
        console.log(`Retrying in ${delay}ms...`);

        // Update progress to show retry
        if (typeof this.onReportGenerationProgress === 'function') {
          this.onReportGenerationProgress(0.1 + (attempt * 0.1));
        }

        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }
}