/**
 * HTML Formatter for TextRad Extension
 * Converts markdown reports to HTML for better copy-paste formatting
 * Based on the web app's formatting utilities
 */

export class HTMLFormatter {
  /**
   * Convert markdown to HTML with professional medical report styling
   * @param {string} markdownText - The markdown formatted text
   * @param {string} formatName - Optional format name to use as report title
   * @returns {string} HTML formatted text with styling
   */
  static convertMarkdownToHTML(markdownText, formatName = null) {
    if (!markdownText) {
      return "<p>Error: No report content available</p>";
    }

    try {
      // Add format name as main heading if provided
      let html = markdownText;
      if (formatName && formatName.trim() !== '' && formatName !== 'Extension Report') {
        html = `# ${formatName}\n\n${html}`;
      }

      // Split into lines for impression section processing
      const lines = html.split('\n');
      let inImpressionSection = false;
      let processedLines = [];

      for (let i = 0; i < lines.length; i++) {
        let line = lines[i];
        const trimmedLine = line.trim();

        // Check if we're entering the impression section
        if (trimmedLine.toLowerCase().includes('impression:') ||
            trimmedLine.toLowerCase().includes('**impression**') ||
            (trimmedLine.toLowerCase().startsWith('impression') && trimmedLine.includes(':'))) {
          inImpressionSection = true;
          processedLines.push(line);
          continue;
        }

        // Check if we're leaving the impression section
        if (inImpressionSection &&
            (trimmedLine.includes('**End of Report**') ||
             trimmedLine.toLowerCase().includes('end of report') ||
             (trimmedLine.match(/^[A-Z][A-Z\s]+:$/) && !trimmedLine.toLowerCase().includes('impression')))) {
          inImpressionSection = false;
          processedLines.push(line);
          continue;
        }

        // If we're in the impression section, ensure all content is bold
        if (inImpressionSection && trimmedLine !== '') {
          // Skip if it's already a heading or contains markdown formatting
          if (!trimmedLine.startsWith('#') &&
              !trimmedLine.includes('**') &&
              !trimmedLine.toLowerCase().includes('impression:')) {

            // Handle bullet points specially
            if (trimmedLine.startsWith('•') || trimmedLine.startsWith('-')) {
              const bulletContent = trimmedLine.replace(/^[•-]\s*/, '');
              line = line.replace(trimmedLine, `• **${bulletContent}**`);
            } else {
              // Make regular text bold
              line = line.replace(trimmedLine, `**${trimmedLine}**`);
            }
          }
        }

        processedLines.push(line);
      }

      html = processedLines.join('\n');

      // Clean up any markdown headings
      html = html.replace(/^(#{1,3})\s+(.+)$/gm, '<h1 class="report-heading">$2</h1>');

      // Convert bold text - make title text larger if it's the first line
      const finalLines = html.split('\n');
      if (finalLines.length > 0 && finalLines[0].includes('**')) {
        // This is likely a title - make it larger
        finalLines[0] = finalLines[0].replace(/\*\*(.*?)\*\*/g, '<strong class="report-title">$1</strong>');
        html = finalLines.join('\n');
      }

      // Regular bold text conversion for remaining content
      html = html.replace(/\*\*(.*?)\*\*/g, '<strong class="report-bold">$1</strong>');

      // Convert bullet points
      html = html.replace(/^•(.*)/gm, '<li class="report-list-item">$1</li>');

      // Convert line breaks to proper HTML
      html = html.replace(/\n\n/g, '</p><p class="report-paragraph">');
      html = html.replace(/\n/g, '<br>');

      // Wrap in paragraphs if not already wrapped
      if (!html.startsWith('<')) {
        html = '<p class="report-paragraph">' + html + '</p>';
      }

      // Wrap bullet points in ul tags
      html = html.replace(/(<li class="report-list-item">.*?<\/li>)/g, '<ul class="report-list">$1</ul>');

      return html;
    } catch (error) {
      console.error('Error converting markdown to HTML:', error);
      return `<p class="report-paragraph">${markdownText}</p>`;
    }
  }

  /**
   * Generate HTML with complete styling for copy-paste
   * @param {string} markdownText - The markdown formatted text
   * @param {string} formatName - Optional format name to use as report title
   * @returns {string} Complete HTML with embedded styles
   */
  static getStyledHTMLForCopy(markdownText, formatName = null) {
    const htmlContent = this.convertMarkdownToHTML(markdownText, formatName);

    // Add comprehensive styling that will be preserved when copied
    const styledHTML = `
<div style="
  line-height: 1.6;
  font-family: Calibri, 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
  font-size: 12pt;
  color: #000000;
  background-color: transparent;
  margin: 0;
  padding: 0;
">
  <style>
    .report-content {
      line-height: 1.6;
      font-family: Calibri, 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
      font-size: 12pt;
      color: #000000;
      background-color: transparent;
    }
    .report-paragraph {
      margin: 0.7em 0;
      line-height: 1.6;
      white-space: pre-wrap;
    }
    .report-paragraph:first-of-type {
      text-align: center;
    }
    .report-title {
      font-size: 14pt;
      font-weight: bold;
      color: #000000;
      text-align: center;
    }
    .report-heading {
      font-size: 13pt;
      font-weight: bold;
      margin: 1em 0 0.5em 0;
      color: #000000;
      text-align: center;
    }
    .report-bold {
      font-weight: bold;
      color: #000000;
    }
    .report-list {
      list-style-type: disc;
      margin-left: 20px;
      padding-left: 0;
      margin-top: 0.5em;
      margin-bottom: 0.5em;
    }
    .report-list-item {
      line-height: 1.5;
      margin: 0.3em 0;
      white-space: pre-wrap;
    }
  </style>
  <div class="report-content">
    ${htmlContent}
  </div>
</div>`;

    return styledHTML;
  }

  /**
   * Convert HTML to plain text for display in extension popup
   * @param {string} htmlText - The HTML formatted text
   * @returns {string} Plain text version
   */
  static htmlToPlainText(htmlText) {
    try {
      // Create a temporary div to parse HTML
      const tempDiv = document.createElement('div');
      tempDiv.innerHTML = htmlText;

      // Extract text content while preserving some formatting
      let plainText = tempDiv.textContent || tempDiv.innerText || '';

      // Clean up extra whitespace
      plainText = plainText.replace(/\s+/g, ' ').trim();

      return plainText;
    } catch (error) {
      console.error('Error converting HTML to plain text:', error);
      return htmlText;
    }
  }

  /**
   * Check if text contains markdown formatting
   * @param {string} text - Text to check
   * @returns {boolean} True if contains markdown
   */
  static isMarkdownFormatted(text) {
    const markdownPatterns = [
      /\*\*.*?\*\*/,  // Bold text
      /#{1,6}\s+/,       // Headers
      /^•/m,              // Bullet points
      /\n\n/            // Double line breaks
    ];

    return markdownPatterns.some(pattern => pattern.test(text));
  }

  /**
   * Get appropriate copy format based on content type
   * @param {string} content - Content to format
   * @param {boolean} isReport - Whether this is a generated report
   * @param {string} formatName - Optional format name to use as report title
   * @returns {string} Formatted content for copying
   */
  static getOptimalCopyFormat(content, isReport = false, formatName = null) {
    if (isReport && this.isMarkdownFormatted(content)) {
      // For reports with markdown, return styled HTML
      return this.getStyledHTMLForCopy(content, formatName);
    } else {
      // For regular transcriptions, return plain text
      return content;
    }
  }
}