// Debug Commands for TextRad Extension
// Run these in the browser console to test functionality

// 1. Test TextRad objects availability (run in TextRad page console)
function debugTextRadObjects() {
  console.log('=== TextRad Object Debug ===');
  console.log('window.textradApp:', !!window.textradApp);

  if (window.textradApp) {
    console.log('textradApp.isInitialized:', window.textradApp.isInitialized);
    console.log('textradApp.isAuthenticated:', window.textradApp.isAuthenticated);
    console.log('textradApp.formatManager:', !!window.textradApp.formatManager);
    console.log('textradApp.reportGenerator:', !!window.textradApp.reportGenerator);
    console.log('textradApp.currentUser:', !!window.textradApp.currentUser);

    if (window.textradApp.formatManager) {
      try {
        const formats = window.textradApp.formatManager.getAllFormats();
        console.log('Available formats:', formats?.length || 0, formats);
      } catch (error) {
        console.error('Error getting formats:', error);
      }
    }
  }

  // Check DOM elements
  console.log('#format-selector exists:', !!document.querySelector('#format-selector'));
  console.log('#generate-button exists:', !!document.querySelector('#generate-button'));
}

// 2. Test content script communication (run in extension popup console)
function testContentScriptCommunication() {
  chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
    const tabId = tabs[0].id;
    console.log('Testing communication with tab:', tabId);

    // Test basic ping
    chrome.tabs.sendMessage(tabId, {action: 'PING'}, (response) => {
      if (chrome.runtime.lastError) {
        console.error('Ping failed:', chrome.runtime.lastError);
        return;
      }
      console.log('Ping response:', response);

      // Test format loading
      chrome.tabs.sendMessage(tabId, {action: 'GET_FORMATS'}, (response) => {
        if (chrome.runtime.lastError) {
          console.error('GET_FORMATS failed:', chrome.runtime.lastError);
          return;
        }
        console.log('GET_FORMATS response:', response);
      });
    });
  });
}

// 3. Force inject content script (run in extension popup console)
function forceInjectContentScript() {
  chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
    const tabId = tabs[0].id;
    console.log('Force injecting content script into tab:', tabId);

    chrome.scripting.executeScript({
      target: { tabId: tabId },
      files: ['content/content-script.js']
    }).then(() => {
      console.log('Content script injected successfully');

      // Wait a moment then test communication
      setTimeout(() => {
        testContentScriptCommunication();
      }, 1000);
    }).catch(error => {
      console.error('Failed to inject content script:', error);
    });
  });
}

// 4. Test template loading manually (run in extension popup console)
function testTemplateLoading() {
  console.log('Testing template loading...');

  chrome.runtime.sendMessage({
    action: 'LOAD_FORMATS'
  }, (response) => {
    console.log('LOAD_FORMATS response:', response);
  });
}

// 5. Check if content script is loaded (run in TextRad page console)
function checkContentScript() {
  console.log('=== Content Script Check ===');

  // Check for content script specific functions
  const contentScriptMarkers = [
    'textrad-webapp-message',
    'textrad-extension-bridge-ready'
  ];

  contentScriptMarkers.forEach(event => {
    document.addEventListener(event, (e) => {
      console.log('Content script event detected:', event, e.detail);
    });
  });

  // Try to trigger content script detection
  document.dispatchEvent(new CustomEvent('test-content-script'));
}

console.log(`
=== TextRad Extension Debug Commands ===

Run in TextRad page console:
- debugTextRadObjects()
- checkContentScript()

Run in Extension popup console:
- testContentScriptCommunication()
- forceInjectContentScript()
- testTemplateLoading()

These commands will help identify where the template integration is failing.
`);