/**
 * Extension ReportStorage for TextRad Extension
 * Handles saving extension-generated reports to Firestore and updating usage counters
 */

import { firebaseConfig } from './firebase-config.js';

export class ExtensionReportStorage {
  constructor() {
    this.db = null;
    this.auth = null;
    this.isInitialized = false;
    this.initializationPromise = null;
    this.scriptsLoaded = false;
  }


  /**
   * Initialize Firebase services
   */
  async initialize() {
    if (this.isInitialized) {
      return true;
    }

    if (this.initializationPromise) {
      return this.initializationPromise;
    }

    this.initializationPromise = this._initializeFirebase();
    return this.initializationPromise;
  }

  async _initializeFirebase() {
    try {
      // Use content script proxy instead of direct Firebase access
      console.log('Using content script Firebase proxy for report storage');
      this.isInitialized = true;
      console.log('Extension ReportStorage initialized successfully (proxy mode)');
      return true;
    } catch (error) {
      console.error('Failed to initialize Extension ReportStorage:', error);
      this.isInitialized = false;
      throw error;
    }
  }

  /**
   * Check if user is authenticated via content script
   */
  async isUserAuthenticated() {
    try {
      const response = await chrome.runtime.sendMessage({
        action: 'CHECK_AUTH_STATUS'
      });
      return response && response.authenticated;
    } catch (error) {
      console.warn('Failed to check auth status:', error);
      return false;
    }
  }

  /**
   * Save a report to Firestore (same structure as web app)
   * @param {Object} reportData - The report data
   * @returns {string|null} The ID of the saved report or null if failed
   */
  async saveReport(reportData) {
    try {
      await this.initialize();

      const isAuthenticated = await this.isUserAuthenticated();
      if (!isAuthenticated) {
        console.warn('User not authenticated, cannot save report');
        return null;
      }

      // Validate report content
      if (!reportData.content || reportData.content.trim() === '') {
        throw new Error('Cannot save report: Empty content');
      }

      // Send save request to content script
      const response = await chrome.runtime.sendMessage({
        action: 'SAVE_REPORT_VIA_CONTENT_SCRIPT',
        reportData: {
          content: reportData.content,
          formatName: reportData.formatName || 'Extension Report',
          source: 'extension'
        }
      });

      if (response && response.success) {
        console.log(`Extension report ${response.reportId} saved via content script`);
        return response.reportId;
      } else {
        console.error('Failed to save report via content script:', response?.error);
        return null;
      }
    } catch (error) {
      console.error('Error saving extension report:', error);
      return null;
    }
  }

  /**
   * Update report usage counter (same as web app)
   * @param {number} count - Number of reports to increment (default: 1)
   */
  async updateReportUsage(count = 1) {
    try {
      await this.initialize();

      const isAuthenticated = await this.isUserAuthenticated();
      if (!isAuthenticated) {
        console.warn('User not authenticated, cannot update report usage');
        return false;
      }

      // Send usage update request to content script
      const response = await chrome.runtime.sendMessage({
        action: 'UPDATE_USAGE_VIA_CONTENT_SCRIPT',
        count: count
      });

      if (response && response.success) {
        console.log(`Updated report usage (+${count}) via content script`);
        return true;
      } else {
        console.error('Failed to update report usage via content script:', response?.error);
        return false;
      }
    } catch (error) {
      console.error('Error updating report usage from extension:', error);
      return false;
    }
  }

  /**
   * Check user's subscription status and report limits
   * @returns {Object} Subscription info
   */
  async checkSubscriptionStatus() {
    try {
      await this.initialize();

      const isAuthenticated = await this.isUserAuthenticated();
      if (!isAuthenticated) {
        return { valid: false, reason: 'not_authenticated' };
      }

      // Send subscription check request to content script
      const response = await chrome.runtime.sendMessage({
        action: 'CHECK_SUBSCRIPTION_VIA_CONTENT_SCRIPT'
      });

      if (response && response.success) {
        return {
          valid: response.valid,
          subscriptionStatus: response.subscriptionStatus,
          reportsUsed: response.reportsUsed,
          maxReports: response.maxReports,
          reason: response.reason
        };
      } else {
        console.error('Failed to check subscription via content script:', response?.error);
        return { valid: false, reason: 'error', error: response?.error };
      }
    } catch (error) {
      console.error('Error checking subscription status:', error);
      return { valid: false, reason: 'error', error };
    }
  }

  /**
   * Save report and update usage counter in one operation
   * @param {Object} reportData - The report data
   * @returns {Object} Result with success status and details
   */
  async saveReportAndUpdateUsage(reportData) {
    try {
      // First check subscription status
      const subscriptionStatus = await this.checkSubscriptionStatus();
      if (!subscriptionStatus.valid) {
        return {
          success: false,
          reason: subscriptionStatus.reason,
          subscriptionStatus
        };
      }

      // Save the report
      const reportId = await this.saveReport(reportData);
      if (!reportId) {
        return {
          success: false,
          reason: 'save_failed'
        };
      }

      // Update usage counter
      const usageUpdated = await this.updateReportUsage(1);
      if (!usageUpdated) {
        console.warn('Report saved but usage counter update failed');
      }

      return {
        success: true,
        reportId,
        usageUpdated
      };
    } catch (error) {
      console.error('Error in saveReportAndUpdateUsage:', error);
      return {
        success: false,
        reason: 'error',
        error
      };
    }
  }
}