/**
 * Page context injector for TextRad Extension
 * This script runs in the page context (not content script isolated context)
 * and can access window.textRadApp directly
 */

// Listen for messages from content script
window.addEventListener('message', (event) => {
  // Only accept messages from same origin
  if (event.origin !== window.location.origin) {
    return;
  }

  // Handle user ID request
  if (event.data && event.data.type === 'TEXTRAD_EXTENSION_GET_USER_ID') {
    try {
      let userId = null;
      if (window.textRadApp && window.textRadApp.auth && window.textRadApp.auth.currentUser) {
        userId = window.textRadApp.auth.currentUser.uid;
      }

      window.postMessage({
        type: 'TEXTRAD_EXTENSION_USER_ID',
        userId: userId
      }, window.location.origin);
    } catch (error) {
      console.error('❌ Error getting user ID:', error);
      window.postMessage({
        type: 'TEXTRAD_EXTENSION_USER_ID',
        userId: null
      }, window.location.origin);
    }
    return;
  }

  // Check if this is a TextRad extension message
  if (event.data && event.data.type === 'TEXTRAD_EXTENSION_SET_REPORT') {
    console.log('📨 Page injector received report data:', {
      formatName: event.data.formatName,
      contentLength: event.data.content?.length
    });

    try {
      // Check if textRadApp is available
      if (!window.textRadApp) {
        console.error('❌ window.textRadApp not found in page context');
        window.postMessage({
          type: 'TEXTRAD_EXTENSION_REPORT_SET',
          success: false,
          error: 'textRadApp not found'
        }, window.location.origin);
        return;
      }

      // Set currentReport in the page context
      window.textRadApp.currentReport = {
        content: event.data.content,
        formatName: event.data.formatName,
        source: event.data.source || 'extension'
      };

      console.log('✅ Set currentReport in page context:', window.textRadApp.currentReport.formatName);

      // Use the web app's displayInlineReport method to properly show the report
      // This ensures all event listeners are attached correctly
      if (typeof window.textRadApp.displayInlineReport === 'function') {
        console.log('📄 Calling displayInlineReport to show report with proper event listeners');
        window.textRadApp.displayInlineReport(event.data.content);
      } else {
        console.warn('⚠️ displayInlineReport not available, report might not have event listeners');
      }

      // Send success response back to content script
      window.postMessage({
        type: 'TEXTRAD_EXTENSION_REPORT_SET',
        success: true
      }, window.location.origin);

    } catch (error) {
      console.error('❌ Error setting currentReport:', error);
      window.postMessage({
        type: 'TEXTRAD_EXTENSION_REPORT_SET',
        success: false,
        error: error.message
      }, window.location.origin);
    }
  }
});

console.log('✅ TextRad extension page injector loaded');