/**
 * TextRad Dictation Extension - Service Worker (Background Script)
 * 
 * Responsibilities:
 * - Managing extension lifecycle
 * - Coordinating communication between components
 * - Tab management and detection
 * - State persistence
 */

// Track TextRad tabs
let textradTabs = {};
let activeTabId = null;

// Track service health across tabs
let tabServiceHealth = {};

// Track current state
let currentState = {
  recording: false,
  transcribing: false,
  lastTranscription: null,
  lastTranscriptionTimestamp: null,
  appendMode: true // true = append, false = replace (set to false by "New Dictation")
};

// Background recording has been disabled - no offscreen document needed

/**
 * Initialize the service worker
 */
function initialize() {
  console.log('TextRad Dictation Extension service worker initialized');

  // Generate a new browser session ID for tracking session-based mini extension deployments
  // This ID changes every time the service worker restarts (browser restart/extension reload)
  const browserSessionId = `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  chrome.storage.local.set({ currentBrowserSessionId: browserSessionId });
  console.log('Generated new browser session ID:', browserSessionId);

  // Load state from storage
  chrome.storage.local.get(['textradTabs', 'currentState', 'tabServiceHealth'], (data) => {
    if (data.textradTabs) {
      textradTabs = data.textradTabs;
    }

    if (data.currentState) {
      currentState = data.currentState;
    }

    if (data.tabServiceHealth) {
      tabServiceHealth = data.tabServiceHealth;
    }

    // Verify tabs still exist and are TextRad tabs
    verifyStoredTabs();
  });

  // Setup listeners
  setupListeners();
}

/**
 * Verify that stored tabs still exist
 */
async function verifyStoredTabs() {
  const validTabs = {};

  for (const tabId in textradTabs) {
    try {
      // Check if tab still exists
      const tab = await chrome.tabs.get(parseInt(tabId));
      if (tab && isTextRadUrl(tab.url)) {
        validTabs[tabId] = textradTabs[tabId];
      }
    } catch (error) {
      // Tab no longer exists, ignore error
    }
  }

  // Update stored tabs
  textradTabs = validTabs;
  saveState();
}

/**
 * Check if URL is a TextRad URL
 * Returns true for TextRad domains (both landing page and app page)
 * Content script injection is controlled by manifest.json (restricted to /app/* only)
 * This function is used for login detection and background script management
 */
function isTextRadUrl(url) {
  if (!url) return false;

  // Hospital integration domains - allow all paths
  if (url.includes('augmento.vijayadiagnostic.com') ||
    url.includes('mhe-risweb05.manipalhospitals.com') ||
    url.includes('mhmkaengsrv')) {
    return true;
  }

  // TextRad main domains - accept both landing page and app page
  // This allows login detection to work on landing page
  // Content script injection is restricted by manifest.json to /app/* only
  if (url.includes('textrad.com') || url.includes('textrad.in') || url.includes('localhost')) {
    return true;
  }

  return false;
}

/**
 * Set up message and event listeners
 */
function setupListeners() {
  // Listen for messages from popup and content scripts
  chrome.runtime.onMessage.addListener(handleMessage);

  // Listen for tab updates
  chrome.tabs.onUpdated.addListener(handleTabUpdated);

  // Listen for tab removal
  chrome.tabs.onRemoved.addListener(handleTabRemoved);

  // Listen for active tab changes
  chrome.tabs.onActivated.addListener(handleTabActivated);
}

/**
 * Handle messages from popup and content scripts
 */
function handleMessage(message, sender, sendResponse) {
  console.log('Received message:', message);

  // Handle different message types
  switch (message.action) {
    case 'START_RECORDING':
      handleStartRecording(sendResponse);
      break;

    case 'STOP_RECORDING':
      handleStopRecording(sendResponse);
      break;

    // Mini extension independent recording
    case 'TRANSCRIBE_MINI_AUDIO':
      handleTranscribeMiniAudio(message.audioData, message.mimeType, sendResponse);
      return true; // Keep the message channel open for async response

    case 'ADD_TRANSCRIPTION_FROM_MINI':
      handleAddTranscriptionFromMini(message.transcription, sendResponse);
      break;

    case 'TRIGGER_AUTO_SEND':
      handleTriggerAutoSend(sendResponse);
      return true; // Keep message channel open for async response

    case 'SEND_TRANSCRIPTION_TO_TEXTRAD':
      handleSendTranscriptionToTextRad(sendResponse);
      return true; // Keep message channel open for async response

    // Open popup in full mode
    case 'OPEN_POPUP_FULL_MODE':
      chrome.action.openPopup();
      sendResponse({ success: true });
      break;

    case 'MAINTAIN_RECORDING':
      // Handler to maintain recording when popup is closed
      handleMaintainRecording(message.inProgress, sendResponse);
      break;

    case 'SET_VIEW_MODE':
      // Save view mode in storage
      chrome.storage.local.set({ viewMode: message.mode });
      sendResponse({ success: true });
      break;

    case 'SET_TRANSCRIPTION':
      handleSetTranscription(message.transcription, sendResponse);
      break;

    case 'NEW_DICTATION':
      handleNewDictation(sendResponse);
      break;

    case 'SEND_TRANSCRIPTION_TO_TAB':
      handleSendTranscriptionToTab(message.tabId, message.transcription, sendResponse);
      return true; // Keep the message channel open for async response

    case 'FIND_TEXTRAD_TABS':
      findTextRadTabs(sendResponse);
      return true; // Keep the message channel open for async response

    case 'CONNECTED_TO_TEXTRAD':
      handleTextRadTabConnected(sender.tab.id, sendResponse);
      break;

    case 'GET_STATE':
      sendResponse({
        state: currentState,
        tabs: textradTabs,
        recordingInBackground: currentState.recording
      });
      break;

    case 'GET_FIREBASE_CONFIG':
      // If we have stored Firebase config, return it
      chrome.storage.local.get(['firebaseConfig'], (data) => {
        sendResponse({ config: data.firebaseConfig || null });
      });
      return true; // Keep the message channel open for async response

    case 'LOAD_FORMATS':
      handleLoadFormats(message.tabId, sendResponse);
      return true; // Keep the message channel open for async response

    case 'GENERATE_REPORT':
      handleGenerateReport(message.tabId, message.transcription, message.formatId, sendResponse);
      return true; // Keep the message channel open for async response

    case 'ACTIVATE_TEXTRAD_TABS':
      handleActivateTextRadTabs(sendResponse);
      return true; // Keep the message channel open for async response

    case 'SERVICE_HEALTH_UPDATE':
      handleServiceHealthUpdate(sender.tab.id, message, sendResponse);
      break;
  }

  // Return false to close the message channel
  return false;
}

/**
 * Handle tab updates
 */
async function handleTabUpdated(tabId, changeInfo, tab) {
  // Check if this is a TextRad tab that has finished loading
  if (changeInfo.status === 'complete') {
    // 1. Check for standard TextRad URLs (Manifest-based)
    if (isTextRadUrl(tab.url)) {
      // Add to tracked tabs if not already tracked
      if (!textradTabs[tabId]) {
        textradTabs[tabId] = {
          url: tab.url,
          title: tab.title,
          lastConnected: Date.now()
        };
        saveState();
      }

      // Automatically inject content script when TextRad pages load
      try {
        console.log('TextRad page loaded, ensuring content script is available for tab', tabId);

        // Wait a moment for page to fully initialize
        setTimeout(async () => {
          try {
            await ensureContentScriptLoaded(tabId);
            console.log('Content script ready for TextRad tab', tabId);
          } catch (error) {
            console.log('Could not auto-inject content script for tab', tabId, ':', error.message);
          }
        }, 2000); // Wait 2 seconds for page to settle
      } catch (error) {
        console.log('Error setting up auto-injection for tab', tabId, ':', error);
      }
    }
    // 2. Check for User Preference URLs (Dynamic Auto-Deploy)
    else if (tab.url && !tab.url.startsWith('chrome://') && !tab.url.startsWith('edge://') && !tab.url.startsWith('about:')) {
      try {
        const urlObj = new URL(tab.url);
        const hostname = urlObj.hostname;

        // Check preferences
        chrome.storage.local.get(['miniExtensionDomainPreferences'], async (result) => {
          const preferences = result.miniExtensionDomainPreferences || {};

          if (preferences[hostname] === 'always') {
            console.log(`Dynamic Auto-Deploy: Domain ${hostname} has "always" preference. Injecting...`);

            try {
              // Inject content script
              await chrome.scripting.executeScript({
                target: { tabId: tabId },
                files: ['content/content-script.js']
              });
              console.log('Dynamic Auto-Deploy: Content script injected for', hostname);

              // Wait slightly for script to init
              setTimeout(async () => {
                try {
                  // Trigger deployment
                  await chrome.tabs.sendMessage(tabId, {
                    action: 'DEPLOY_MINI_EXTENSION',
                    preference: 'always'
                  });
                  console.log('Dynamic Auto-Deploy: Deployment message sent to', hostname);
                } catch (msgError) {
                  console.error('Dynamic Auto-Deploy: Failed to send deployment message:', msgError);
                }
              }, 500);
            } catch (injectError) {
              console.error('Dynamic Auto-Deploy: Failed to inject script:', injectError);
            }
          }
        });
      } catch (e) {
        // Invalid URL, ignore
      }
    }
  }
}

/**
 * Handle tab removal
 */
function handleTabRemoved(tabId) {
  // Remove from tracked tabs
  if (textradTabs[tabId]) {
    delete textradTabs[tabId];
    saveState();
  }

  // Reset active tab if it was this one
  if (activeTabId === tabId) {
    activeTabId = null;
  }
}

/**
 * Handle tab activation
 */
function handleTabActivated(activeInfo) {
  // Check if this is a TextRad tab
  if (textradTabs[activeInfo.tabId]) {
    activeTabId = activeInfo.tabId;
  }
}

/**
 * Handle TextRad tab connection
 */
function handleTextRadTabConnected(tabId, sendResponse) {
  // Update tracked tabs
  chrome.tabs.get(tabId, (tab) => {
    if (chrome.runtime.lastError) {
      console.error('Error getting tab:', chrome.runtime.lastError);
      return;
    }

    textradTabs[tabId] = {
      url: tab.url,
      title: tab.title,
      lastConnected: Date.now(),
      connected: true
    };

    saveState();
    sendResponse({ success: true });
  });
}

/**
 * Handle starting recording
 */
function handleStartRecording(sendResponse) {
  currentState.recording = true;
  currentState.transcribing = false;
  saveState();

  // Update extension badge
  chrome.action.setBadgeText({ text: 'REC' });
  chrome.action.setBadgeBackgroundColor({ color: '#FF0000' });

  sendResponse({ success: true });
}

/**
 * Handle stopping recording
 */
function handleStopRecording(sendResponse) {
  currentState.recording = false;
  currentState.transcribing = true;
  saveState();

  // Update extension badge
  chrome.action.setBadgeText({ text: 'TRS' });
  chrome.action.setBadgeBackgroundColor({ color: '#0000FF' });

  sendResponse({ success: true });
}

/**
 * Handle maintaining recording in background when popup closes
 * Now we create a detached window to keep the recording alive
 */
function handleMaintainRecording(inProgress, sendResponse) {
  if (inProgress) {
    console.log('Maintaining recording in progress - keeping recording active');

    // Update state to indicate we're recording
    currentState.recording = true;
    saveState();

    // Update badge to show recording is in progress
    chrome.action.setBadgeText({ text: 'REC' });
    chrome.action.setBadgeBackgroundColor({ color: '#FF0000' });

    // Make badge blink to indicate background recording
    startBadgeBlinking();

    // Open a detached mini window to continue recording
    openRecordingWindow();
  } else {
    // Reset recording state
    currentState.recording = false;
    saveState();

    // Clear any recording badge
    chrome.action.setBadgeText({ text: '' });

    // Stop any blinking badge
    stopBadgeBlinking();
  }

  // Response is optional for this handler
  if (sendResponse) {
    sendResponse({ success: true });
  }
}

/**
 * Handle transcribing audio from mini extension
 */
async function handleTranscribeMiniAudio(audioData, mimeType, sendResponse) {
  console.log('Transcribing mini extension audio...');

  try {
    // Get user's transcription provider preference
    const result = await chrome.storage.sync.get(['transcriptionProvider']);
    const transcriptionProvider = result.transcriptionProvider || 'auto';
    console.log('📋 Using transcription provider setting:', transcriptionProvider);

    // Create a blob from the base64 audio data
    const binaryString = atob(audioData);
    const bytes = new Uint8Array(binaryString.length);
    for (let i = 0; i < binaryString.length; i++) {
      bytes[i] = binaryString.charCodeAt(i);
    }
    const audioBlob = new Blob([bytes], { type: mimeType });

    console.log('Mini audio blob for transcription:', audioBlob.size, 'bytes');

    // Convert blob to base64 for API call
    const reader = new FileReader();
    reader.onload = async () => {
      try {
        const base64Audio = reader.result.split(',')[1]; // Remove data:audio/webm;base64, prefix
        console.log('Mini audio converted to base64, size:', Math.round(base64Audio.length / 1024), 'KB');

        const requestData = {
          data: {
            audioData: base64Audio,
            userId: 'extension-user',
            source: 'extension'
          }
        };

        let transcriptionResult = null;
        let finalError = null;

        // Try transcription based on user setting
        if (transcriptionProvider === 'openai') {
          // Use OpenAI only (manual selection)
          console.log('🤖 Using OpenAI GPT-4o-Transcribe only (user preference)');
          transcriptionResult = await tryOpenAITranscription(requestData);
          if (!transcriptionResult.success) {
            finalError = transcriptionResult.error;
          }
        } else if (transcriptionProvider === 'elevenlabs') {
          // Use ElevenLabs only (manual selection)
          console.log('🎙️ Using ElevenLabs only (user preference)');
          transcriptionResult = await tryElevenLabsTranscription(requestData);
          if (!transcriptionResult.success) {
            finalError = transcriptionResult.error;
          }
        } else {
          // Auto mode: Try OpenAI first, automatic fallback to ElevenLabs on failure
          console.log('🔄 Auto mode: Trying OpenAI first, will auto-fallback to ElevenLabs on errors');

          // Try OpenAI first
          transcriptionResult = await tryOpenAITranscription(requestData);

          if (!transcriptionResult.success) {
            console.log('⚠️ OpenAI failed, automatically trying ElevenLabs backup:', transcriptionResult.error);

            // Automatic failover to ElevenLabs backup
            const backupResult = await tryElevenLabsTranscription(requestData);
            if (backupResult.success) {
              console.log('✅ ElevenLabs backup successful (automatic failover)!');
              transcriptionResult = backupResult;
            } else {
              console.log('❌ Both providers failed');
              finalError = `OpenAI: ${transcriptionResult.error}; ElevenLabs: ${backupResult.error}`;
            }
          } else {
            console.log('✅ OpenAI successful!');
          }
        }

        // Send final response
        if (transcriptionResult && transcriptionResult.success) {
          console.log('🎉 Transcription completed successfully:', transcriptionResult.transcription.substring(0, 100) + '...');
          sendResponse({
            success: true,
            transcription: transcriptionResult.transcription,
            provider: transcriptionResult.provider
          });
        } else {
          console.error('💥 All transcription attempts failed:', finalError);
          sendResponse({
            success: false,
            error: 'Transcription failed: ' + finalError
          });
        }

      } catch (apiError) {
        console.error('Mini transcription API error:', apiError);
        sendResponse({
          success: false,
          error: 'Transcription API error: ' + apiError.message
        });
      }
    };

    reader.onerror = () => {
      console.error('Error reading mini audio blob');
      sendResponse({
        success: false,
        error: 'Error reading audio data'
      });
    };

    reader.readAsDataURL(audioBlob);

  } catch (error) {
    console.error('Error transcribing mini audio:', error);
    sendResponse({
      success: false,
      error: error.message
    });
  }
}

/**
 * Try transcription with ElevenLabs API
 */
async function tryElevenLabsTranscription(requestData) {
  try {
    const transcriptionEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/elevenlabsTranscribeBackup';

    const response = await fetch(transcriptionEndpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-TextRad-Extension': 'true',
        'Origin': 'chrome-extension://textrad-extension'
      },
      body: JSON.stringify(requestData)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const result = await response.json();
    const transcriptionData = result.result || result;

    if (transcriptionData && transcriptionData.text) {
      return {
        success: true,
        transcription: transcriptionData.text,
        provider: 'elevenlabs'
      };
    } else {
      throw new Error('No transcription text in response');
    }

  } catch (error) {
    return {
      success: false,
      error: error.message,
      provider: 'elevenlabs'
    };
  }
}

/**
 * Try transcription with OpenAI (primary)
 */
async function tryOpenAITranscription(requestData) {
  try {
    const transcriptionEndpoint = 'https://asia-south1-textrad-dc865.cloudfunctions.net/directTranscribeHttp';

    const response = await fetch(transcriptionEndpoint, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-TextRad-Extension': 'true',
        'Origin': 'chrome-extension://textrad-extension'
      },
      body: JSON.stringify(requestData)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const result = await response.json();
    const transcriptionData = result.result || result;

    if (transcriptionData && transcriptionData.text) {
      return {
        success: true,
        transcription: transcriptionData.text,
        provider: 'elevenlabs-scribe-v1-backup'
      };
    } else {
      throw new Error('No transcription text in response');
    }

  } catch (error) {
    return {
      success: false,
      error: error.message,
      provider: 'openai-gpt4o-transcribe'
    };
  }
}

/**
 * Handle triggering auto-send workflow for mini extension transcriptions
 */
async function handleTriggerAutoSend(sendResponse) {
  console.log('Triggering auto-send workflow for mini transcription...');

  try {
    // Check if auto-send is enabled in settings
    const storage = await chrome.storage.local.get(['settings']);
    const settings = storage.settings || {};

    if (settings.autoSend !== false) { // Default to enabled if not set
      console.log('Auto-send enabled, finding TextRad tabs...');

      // Find TextRad tabs (this will trigger template loading)
      findTextRadTabs((tabResponse) => {
        if (tabResponse && tabResponse.success && tabResponse.tabs && tabResponse.tabs.length > 0) {
          console.log('Found TextRad tabs, auto-send successful');
          sendResponse({ success: true, sent: true });
        } else {
          console.log('No TextRad tabs found or user not logged in');
          sendResponse({ success: true, sent: false, reason: 'No tabs or not logged in' });
        }
      });
    } else {
      console.log('Auto-send disabled');
      sendResponse({ success: true, sent: false, reason: 'Auto-send disabled' });
    }
  } catch (error) {
    console.error('Error in trigger auto-send:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Handle sending transcription to TextRad tabs (inject for format selection)
 */
async function handleSendTranscriptionToTextRad(sendResponse) {
  console.log('Injecting transcription to TextRad for format selection...');

  try {
    // Check if we have a transcription to send
    if (!currentState.lastTranscription) {
      sendResponse({ success: false, error: 'No transcription to send' });
      return;
    }

    // Find TextRad tabs
    findTextRadTabs(async (tabResponse) => {
      if (tabResponse && tabResponse.success && tabResponse.tabs && tabResponse.tabs.length > 0) {
        console.log('Found TextRad tabs, injecting transcription...');

        // Use the first available tab
        const targetTab = tabResponse.tabs[0];

        // Inject transcription to prepare for format selection
        chrome.tabs.sendMessage(targetTab.id, {
          action: 'INJECT_TRANSCRIPTION_FOR_FORMAT_SELECTION',
          transcription: currentState.lastTranscription
        }, (injectResult) => {
          if (chrome.runtime.lastError) {
            console.error('Error injecting transcription:', chrome.runtime.lastError);
            sendResponse({ success: false, error: chrome.runtime.lastError.message });
            return;
          }

          if (injectResult && injectResult.success) {
            console.log('Transcription injected successfully, ready for format selection');
            sendResponse({ success: true, injected: true, tabId: targetTab.id });
          } else {
            console.error('Failed to inject transcription:', injectResult?.error);
            sendResponse({ success: false, error: injectResult?.error || 'Failed to inject' });
          }
        });
      } else {
        console.log('No TextRad tabs found or user not logged in');
        sendResponse({ success: false, error: 'No TextRad tabs found or user not logged in' });
      }
    });
  } catch (error) {
    console.error('Error injecting transcription to TextRad:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Handle adding transcription from mini extension to the extension state
 */
function handleAddTranscriptionFromMini(transcription, sendResponse) {
  console.log('Adding transcription from mini extension:', transcription);

  try {
    // Add to current state (append if there's existing text)
    const existingText = currentState.lastTranscription || '';
    const newText = existingText ? existingText + ' ' + transcription : transcription;

    currentState.lastTranscription = newText;
    currentState.lastTranscriptionTimestamp = Date.now();
    currentState.transcribing = false;
    saveState();

    console.log('Mini transcription added to state, total length:', newText.length);

    sendResponse({
      success: true,
      message: 'Transcription added successfully'
    });

  } catch (error) {
    console.error('Error adding mini transcription:', error);
    sendResponse({
      success: false,
      error: error.message
    });
  }
}

/**
 * Update mini extension state in content script
 */
function updateMiniExtensionState(tabId, state) {
  if (tabId) {
    chrome.tabs.sendMessage(tabId, {
      action: 'UPDATE_MINI_EXTENSION_STATE',
      state: state
    }).catch(err => {
      console.log('Could not update mini extension state (tab may not have content script):', err.message);
    });
  }
}

/**
 * Notify popup that transcription is complete
 */
function notifyTranscriptionComplete() {
  // This would be called by the actual transcription process
  // For now, it's a placeholder to show the workflow
  console.log('Transcription complete, dictated text available in popup');
}

/**
 * Open a detached recording window to continue recording
 */
function openRecordingWindow() {
  // Get the extension's popup URL
  const popupURL = chrome.runtime.getURL('popup/popup.html');

  // Check if we already have a recording window open
  if (recordingWindowId) {
    // Focus the existing window
    chrome.windows.update(recordingWindowId, { focused: true });
    return;
  }

  // Create a new window with the popup content that's sized perfectly for the mini mode
  chrome.windows.create({
    url: popupURL + '?viewMode=mini',
    type: 'popup',
    width: 150,  // Mini width with margin
    height: 40,  // Mini height
    left: Math.round(screen.width - 100),
    top: 50,
    focused: false
  }, (window) => {
    if (window) {
      recordingWindowId = window.id;

      // Add a listener for when the window is closed
      chrome.windows.onRemoved.addListener((windowId) => {
        if (windowId === recordingWindowId) {
          recordingWindowId = null;

          // If window was closed, stop the recording
          currentState.recording = false;
          saveState();

          // Clear badge
          chrome.action.setBadgeText({ text: '' });
          stopBadgeBlinking();
        }
      });
    }
  });
}

// Track recording window
let recordingWindowId = null;

// Badge blinking for background recording
let badgeBlinkInterval = null;

/**
 * Start badge blinking animation to indicate background recording
 */
function startBadgeBlinking() {
  // Clear any existing interval
  if (badgeBlinkInterval) {
    clearInterval(badgeBlinkInterval);
  }

  // Set up blinking every 1 second
  let visible = true;
  badgeBlinkInterval = setInterval(() => {
    if (visible) {
      chrome.action.setBadgeText({ text: '' });
    } else {
      chrome.action.setBadgeText({ text: 'REC' });
    }
    visible = !visible;
  }, 1000);
}

/**
 * Stop badge blinking
 */
function stopBadgeBlinking() {
  if (badgeBlinkInterval) {
    clearInterval(badgeBlinkInterval);
    badgeBlinkInterval = null;
  }

  // Restore badge based on current state
  if (currentState.recording) {
    chrome.action.setBadgeText({ text: 'REC' });
  } else if (currentState.transcribing) {
    chrome.action.setBadgeText({ text: 'TRS' });
  } else {
    chrome.action.setBadgeText({ text: '' });
  }
}

/**
 * Handle setting transcription
 */
function handleSetTranscription(transcription, sendResponse) {
  currentState.transcribing = false;
  currentState.recording = false; // Reset recording state

  // Check if we should append to existing transcription based on appendMode flag
  if (currentState.appendMode && currentState.lastTranscription && currentState.lastTranscription.trim() !== '') {
    // Append with a space separator
    console.log('Appending to existing transcription (appendMode=true)');
    currentState.lastTranscription = currentState.lastTranscription.trim() + ' ' + transcription;
  } else {
    // Set new transcription (either appendMode=false or no existing transcription)
    console.log('Setting new transcription (appendMode=' + currentState.appendMode + ')');
    currentState.lastTranscription = transcription;
    // Reset append mode to true after a fresh start
    currentState.appendMode = true;
  }

  currentState.lastTranscriptionTimestamp = Date.now();
  saveState();

  // Update extension badge
  chrome.action.setBadgeText({ text: '' });

  // Stop badge blinking if it was active
  stopBadgeBlinking();

  sendResponse({ success: true });
}

/**
 * Handle new dictation (clear transcription and disable append mode)
 */
function handleNewDictation(sendResponse) {
  console.log('New dictation started - clearing transcription and disabling append mode');

  // Clear transcription and set append mode to false
  currentState.lastTranscription = '';
  currentState.lastTranscriptionTimestamp = Date.now();
  currentState.appendMode = false; // Next transcription will replace, not append
  currentState.recording = false;
  currentState.transcribing = false;

  saveState();

  // Update extension badge
  chrome.action.setBadgeText({ text: '' });

  // Stop badge blinking if it was active
  stopBadgeBlinking();

  sendResponse({ success: true });
}

/**
 * Handle sending transcription to tab
 */
async function handleSendTranscriptionToTab(tabId, transcription, sendResponse) {
  try {
    // If no tabId provided, use active TextRad tab
    if (!tabId) {
      const tabs = await findTextRadTabs();
      if (tabs.length === 0) {
        throw new Error('No logged-in TextRad tabs found. Please log in to TextRad in your browser.');
      }
      tabId = tabs[0].id;
    }

    // Verify the tab is still valid
    try {
      const tab = await chrome.tabs.get(tabId);
      if (!tab || !isTextRadUrl(tab.url)) {
        throw new Error('The selected TextRad tab is no longer valid');
      }
    } catch (error) {
      throw new Error('Tab no longer exists or is not accessible');
    }

    // Ensure content script is loaded
    await ensureContentScriptLoaded(tabId);

    // Check if user is logged in
    try {
      const loginCheckResponse = await chrome.tabs.sendMessage(tabId, { action: 'CHECK_LOGIN' });
      if (loginCheckResponse && loginCheckResponse.loggedIn === false) {
        throw new Error('User is not logged in to TextRad. Please log in and go to TextRad Web App to use the plugin.');
      }
    } catch (error) {
      if (error.message.includes('not logged in')) {
        throw error;
      }
      console.warn('Could not verify login state:', error);
    }

    // Send message to content script
    const response = await chrome.tabs.sendMessage(tabId, {
      action: 'INJECT_TRANSCRIPTION',
      transcription: transcription
    });

    if (response && response.success) {
      sendResponse({ success: true, tabId: tabId });
    } else {
      throw new Error('Failed to inject transcription. Ensure you are logged in to TextRad Web App.');
    }
  } catch (error) {
    console.error('Error sending transcription to tab:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Find all TextRad tabs
 */
async function findTextRadTabs(sendResponse) {
  try {
    // First get all tabs that match TextRad URLs
    const tabs = await chrome.tabs.query({ url: ['*://*.textrad.com/*', '*://*.textrad.in/*', '*://localhost/*', '*://augmento.vijayadiagnostic.com/*', '*://mhe-risweb05.manipalhospitals.com/*'] });

    // Check if we found any tabs at all
    if (tabs.length === 0) {
      if (sendResponse) {
        sendResponse({
          success: true,
          tabs: [],
          hasTextRadTabs: false,
          notLoggedInCount: 0
        });
      }
      return [];
    }

    // Now filter to only those where user is logged in
    const loggedInTabs = [];
    const potentialTabs = {};
    const loggedInTabsMap = {};
    let notLoggedInCount = 0;

    for (const tab of tabs) {
      try {
        // Try to check login state for each tab
        const loginCheckResult = await checkTabLoginState(tab.id);

        if (loginCheckResult) {
          // User is logged in, add to our list
          loggedInTabs.push(tab);
          loggedInTabsMap[tab.id] = true;

          // Track this tab
          potentialTabs[tab.id] = {
            url: tab.url,
            title: tab.title,
            lastConnected: Date.now(),
            connected: true,
            loggedIn: true
          };
        } else {
          console.log(`Tab ${tab.id} is TextRad but user not logged in, skipping`);
          notLoggedInCount++;

          // Track this tab but mark as not logged in
          potentialTabs[tab.id] = {
            url: tab.url,
            title: tab.title,
            lastConnected: Date.now(),
            connected: false,
            loggedIn: false
          };
        }
      } catch (error) {
        console.log(`Error checking login state for tab ${tab.id}:`, error);
        notLoggedInCount++;
      }
    }

    // Update tracked tabs
    textradTabs = potentialTabs;
    saveState();

    if (sendResponse) {
      sendResponse({
        success: true,
        tabs: loggedInTabs,
        loggedInTabs: loggedInTabsMap,
        hasTextRadTabs: true,
        notLoggedInCount: notLoggedInCount,
        error: loggedInTabs.length === 0 && notLoggedInCount > 0 ?
          'TextRad tabs found but user not logged in' : null
      });
    }

    return loggedInTabs;
  } catch (error) {
    console.error('Error finding TextRad tabs:', error);
    if (sendResponse) {
      sendResponse({ success: false, error: error.message });
    }
    return [];
  }
}

/**
 * Check if the user is logged in for a specific tab
 */
async function checkTabLoginState(tabId) {
  try {
    // First, check if tab is on the TextRad domain
    let tab;
    try {
      tab = await chrome.tabs.get(tabId);
      if (!tab || !isTextRadUrl(tab.url)) {
        console.log(`Tab ${tabId} is not a TextRad URL`);
        return false;
      }

      // Don't make assumptions based on URL path - both main page and app page can be 
      // valid if the user is logged in. Let the content script determine login state.
    } catch (error) {
      console.log(`Error getting tab info for ${tabId}:`, error);
      return false;
    }

    // Determine if we're on the main site or app page
    const isMainSite = tab.url.includes('textrad.in/') === false ||
      tab.url.endsWith('textrad.in/') ||
      tab.url.includes('textrad.in/index') ||
      tab.url.includes('textrad.in/?');

    const isAppPage = tab.url.includes('textrad.in/app');

    console.log(`Tab ${tabId} is on: ${isMainSite ? 'Main site' : (isAppPage ? 'App page' : 'Other TextRad page')}`);

    // Look for cookies first - most reliable across both sites
    try {
      const cookieCheckResults = await chrome.scripting.executeScript({
        target: { tabId: tabId },
        func: () => {
          try {
            // Check for cookies that indicate login
            const cookies = document.cookie.split('; ');
            for (const cookie of cookies) {
              const [name, value] = cookie.split('=');
              if ((name.includes('auth') || name.includes('session') || name.includes('token') || name.includes('user')) &&
                value && value.length > 10) {
                return { loggedIn: true, reason: `Found auth cookie: ${name}` };
              }
            }
            return null; // No conclusive result from cookies
          } catch (e) {
            console.error('Error checking cookies:', e);
            return null;
          }
        }
      });

      if (cookieCheckResults && cookieCheckResults[0] && cookieCheckResults[0].result) {
        console.log(`Cookie check for tab ${tabId}: ${cookieCheckResults[0].result.loggedIn} (${cookieCheckResults[0].result.reason})`);
        return cookieCheckResults[0].result.loggedIn;
      }
    } catch (error) {
      console.log(`Error checking cookies in tab ${tabId}:`, error);
    }

    // Main site specific login check with more aggressive DOM element checks
    if (isMainSite) {
      try {
        const mainSiteCheckResults = await chrome.scripting.executeScript({
          target: { tabId: tabId },
          func: () => {
            // FIRST: Check Firebase auth in localStorage (MOST RELIABLE)
            // This should be checked BEFORE DOM elements to avoid false negatives
            try {
              // Check for any Firebase auth key (API key may vary)
              const firebaseKeys = Object.keys(localStorage).filter(key =>
                key.startsWith('firebase:authUser:')
              );

              console.log('Main site - Found Firebase keys:', firebaseKeys.length);

              for (const key of firebaseKeys) {
                const firebaseAuth = localStorage.getItem(key);
                console.log('Main site - Checking key:', key, 'Length:', firebaseAuth?.length);
                if (firebaseAuth && firebaseAuth.length > 20) {
                  // Parse the auth data to make sure it's valid
                  try {
                    const authData = JSON.parse(firebaseAuth);
                    console.log('Main site - Parsed Firebase auth, has UID:', !!authData?.uid);
                    if (authData && authData.uid) {
                      return { loggedIn: true, reason: 'Found valid Firebase auth in localStorage (main site)' };
                    }
                  } catch (parseError) {
                    console.log('Firebase auth data invalid:', parseError);
                  }
                }
              }
              console.log('Main site - No valid Firebase auth found');
            } catch (e) {
              console.log('Error checking Firebase auth:', e);
            }

            // SECOND: Check for main site specific login indicators
            // Only check DOM if Firebase auth was inconclusive
            try {
              // Look for login forms - if visible, user is definitely not logged in
              const loginForms = document.querySelectorAll('form[id*="login"], form[class*="login"], form[id*="signin"], form[class*="signin"]');
              for (const form of loginForms) {
                if (window.getComputedStyle(form).display !== 'none' && form.offsetWidth > 0 && form.offsetHeight > 0) {
                  return { loggedIn: false, reason: 'Found visible login form' };
                }
              }

              // Look for login buttons that are visible (be more specific about login buttons)
              const loginButtons = Array.from(document.querySelectorAll('button, a, input[type="submit"]')).filter(el =>
                (el.textContent && (el.textContent.toLowerCase().includes('log in') || el.textContent.toLowerCase().includes('sign in'))) ||
                (el.value && (el.value.toLowerCase().includes('log in') || el.value.toLowerCase().includes('sign in'))) ||
                (el.type === 'submit' && el.form && (
                  el.form.id.toLowerCase().includes('login') ||
                  el.form.className.toLowerCase().includes('login') ||
                  el.form.action.toLowerCase().includes('login')
                ))
              );

              for (const btn of loginButtons) {
                if (window.getComputedStyle(btn).display !== 'none' && btn.offsetWidth > 0 && btn.offsetHeight > 0) {
                  return { loggedIn: false, reason: 'Found visible login button' };
                }
              }

              // Also check for password/email fields that are currently visible (indicating login form)
              const loginFields = Array.from(document.querySelectorAll('input[type="email"], input[type="password"]')).filter(el =>
                window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0 &&
                (el.placeholder && (el.placeholder.toLowerCase().includes('email') || el.placeholder.toLowerCase().includes('password')))
              );

              if (loginFields.length > 0) {
                return { loggedIn: false, reason: 'Found visible login fields' };
              }
            } catch (e) {
              console.log('Error checking logout indicators:', e);
            }

            // THIRD: Check other localStorage items
            try {

              // Main site specific auth items
              const mainSiteAuthItems = [
                'firebaseui:rememberedAccounts',
                'firebase:authUser',
                'auth',
                'authState',
                'authUser',
                'textrad_user',
                'user_data',
                'session',
                'sessionData'
              ];

              for (const key of mainSiteAuthItems) {
                const item = localStorage.getItem(key);
                if (item && item.length > 10) {
                  return { loggedIn: true, reason: `Found auth item in localStorage: ${key}` };
                }
              }
            } catch (e) {
              console.log('Error checking localStorage:', e);
            }

            // Check for login/logout UI elements that would only be visible for logged-in users
            try {
              // For main site, check for account menu, profile menu, or dashboard elements
              const mainSiteLoggedInSelectors = [
                // Navigation and menu elements when logged in
                '.account-menu', '.user-menu', '.profile-menu', '.dashboard-menu',
                // User display elements
                '.user-avatar', '.user-profile', '.user-initials', '.user-greeting',
                // Common dashboard elements
                '.dashboard', '.dashboard-panel', '.dashboard-content',
                // Account-related sections
                '.my-account', '.my-reports', '.my-dashboard', '.account-section',
                // Action buttons only visible when logged in
                '#logout-button', '.logout-button', '.sign-out-button',
                // App link that only appears when logged in
                '.app-link', '[href*="/app"]'
              ];

              // Look for these elements and check if they're visible
              for (const selector of mainSiteLoggedInSelectors) {
                const elements = document.querySelectorAll(selector);
                if (elements && elements.length > 0) {
                  for (const element of elements) {
                    // Check if element is visible and has content
                    if (window.getComputedStyle(element).display !== 'none' &&
                      element.offsetWidth > 0 && element.offsetHeight > 0) {
                      return { loggedIn: true, reason: `Found main site logged-in element: ${selector}` };
                    }
                  }
                }
              }

              // Check for login links/buttons which would only be visible when NOT logged in
              const mainSiteLoggedOutSelectors = [
                '.login-button', '.sign-in-button', '.auth-button',
                '#login-button', '#sign-in-button',
                '[href*="login"]', '[href*="signin"]', '[href*="auth"]'
              ];

              for (const selector of mainSiteLoggedOutSelectors) {
                const elements = document.querySelectorAll(selector);
                if (elements && elements.length > 0) {
                  for (const element of elements) {
                    // Check if the login element is visible
                    if (window.getComputedStyle(element).display !== 'none' &&
                      element.offsetWidth > 0 && element.offsetHeight > 0) {
                      return { loggedIn: false, reason: `Found main site login element: ${selector}` };
                    }
                  }
                }
              }

              // DO NOT use App links as login indicators - they can exist for non-logged-in users
              // Instead, look for more definitive login indicators

              // Only check for logout/sign out links as definitive login indicators
              const logoutLinks = Array.from(document.querySelectorAll('a')).filter(a =>
                a.textContent.includes('Logout') ||
                a.textContent.includes('Sign Out')
              );

              if (logoutLinks.length > 0) {
                for (const link of logoutLinks) {
                  if (window.getComputedStyle(link).display !== 'none' &&
                    link.offsetWidth > 0 && link.offsetHeight > 0) {
                    return { loggedIn: true, reason: 'Found logout link' };
                  }
                }
              }
            } catch (e) {
              console.log('Error checking DOM elements:', e);
            }

            // No conclusive indicators found
            return null;
          }
        });

        if (mainSiteCheckResults && mainSiteCheckResults[0] && mainSiteCheckResults[0].result) {
          console.log(`Main site check for tab ${tabId}: ${mainSiteCheckResults[0].result.loggedIn} (${mainSiteCheckResults[0].result.reason})`);
          return mainSiteCheckResults[0].result.loggedIn;
        }
      } catch (error) {
        console.log(`Error performing main site check in tab ${tabId}:`, error);
      }
    }

    // Standard checks for both sites
    try {
      // This script will run directly in the page context and check for login indicators
      const checkResults = await chrome.scripting.executeScript({
        target: { tabId: tabId },
        func: () => {
          // FIREBASE AUTH CHECK (HIGHEST PRIORITY - CHECK FIRST)
          // This is the most reliable method and should be checked before DOM elements
          try {
            // Check for any Firebase auth key (API key may vary)
            const firebaseKeys = Object.keys(localStorage).filter(key =>
              key.startsWith('firebase:authUser:')
            );

            for (const key of firebaseKeys) {
              const firebaseAuth = localStorage.getItem(key);
              if (firebaseAuth && firebaseAuth.length > 20) {
                try {
                  const authData = JSON.parse(firebaseAuth);
                  if (authData && authData.uid) {
                    return { loggedIn: true, reason: 'Found valid Firebase auth in localStorage' };
                  }
                } catch (parseError) {
                  console.log('Firebase auth data invalid:', parseError);
                }
              }
            }
          } catch (e) {
            console.log('Error checking Firebase auth:', e);
          }

          // LOGOUT DETECTION (secondary - check if Firebase was inconclusive)
          try {
            // Look for login forms - if visible, user is definitely not logged in
            const loginForms = document.querySelectorAll('form[id*="login"], form[class*="login"], form[id*="signin"], form[class*="signin"], .firebaseui-container');
            for (const form of loginForms) {
              if (window.getComputedStyle(form).display !== 'none' && form.offsetWidth > 0 && form.offsetHeight > 0) {
                return { loggedIn: false, reason: 'Found visible login form' };
              }
            }

            // Look for login buttons/fields that indicate logout state
            const logoutIndicators = Array.from(document.querySelectorAll('button, a, input')).filter(el =>
              (el.textContent && el.textContent.toLowerCase().includes('log in')) ||
              (el.textContent && el.textContent.toLowerCase().includes('sign in')) ||
              (el.value && el.value.toLowerCase().includes('log in')) ||
              (el.value && el.value.toLowerCase().includes('sign in')) ||
              (el.type === 'email' || el.type === 'password') ||
              (el.placeholder && (el.placeholder.toLowerCase().includes('email') || el.placeholder.toLowerCase().includes('password')))
            );

            for (const el of logoutIndicators) {
              if (window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0) {
                return { loggedIn: false, reason: 'Found visible login element' };
              }
            }
          } catch (e) {
            console.log('Error checking logout indicators:', e);
          }

          // STORAGE CHECKS (third priority - check other auth items)
          // Check localStorage for other auth tokens or user data
          try {
            // Look for common auth tokens in localStorage
            const authItems = [
              'firebase:authUser',
              'authUser',
              'textrad_user',
              'user_data',
              'auth_token',
              'session_token',
              'user'
            ];

            for (const key of authItems) {
              const item = localStorage.getItem(key);
              if (item && item.length > 10) {
                // Try to validate the auth data if it's JSON
                try {
                  if (item.startsWith('{') || item.startsWith('[')) {
                    const parsed = JSON.parse(item);
                    // Check if it contains actual user data
                    if (parsed && (parsed.uid || parsed.email || parsed.userId || parsed.user)) {
                      return { loggedIn: true, reason: `Found valid auth token in localStorage: ${key}` };
                    }
                  } else {
                    // Non-JSON auth token, assume valid if long enough
                    return { loggedIn: true, reason: `Found auth token in localStorage: ${key}` };
                  }
                } catch (e) {
                  // If can't parse as JSON, assume it's a valid token
                  return { loggedIn: true, reason: `Found auth token in localStorage: ${key}` };
                }
              }
            }

            // Also check if any localStorage key contains auth-related terms
            for (let i = 0; i < localStorage.length; i++) {
              const key = localStorage.key(i);
              if (key && (
                key.includes('auth') ||
                key.includes('user') ||
                key.includes('token') ||
                key.includes('session')
              )) {
                const value = localStorage.getItem(key);
                if (value && value.length > 10) {
                  return { loggedIn: true, reason: `Found auth-related localStorage item: ${key}` };
                }
              }
            }
          } catch (e) {
            console.log('Error checking localStorage:', e);
          }

          // WINDOW OBJECT CHECKS
          // Check for user data in global window object
          try {
            if (window.firebase && window.firebase.auth && window.firebase.auth().currentUser) {
              // Double-check that the user is actually authenticated and not null
              const currentUser = window.firebase.auth().currentUser;
              if (currentUser && currentUser.uid && !currentUser.isAnonymous) {
                return { loggedIn: true, reason: 'Found authenticated firebase auth user' };
              }
            }

            if (window.textradApp && window.textradApp.user) {
              return { loggedIn: true, reason: 'Found textradApp.user' };
            }

            if (window.textradApp && window.textradApp.isAuthenticated) {
              return { loggedIn: true, reason: 'Found textradApp.isAuthenticated = true' };
            }

            if (window.textradApp && window.textradApp.currentUser) {
              return { loggedIn: true, reason: 'Found textradApp.currentUser' };
            }

            if (window.USER_DATA || window.user || window.currentUser) {
              return { loggedIn: true, reason: 'Found user data in window object' };
            }
          } catch (e) {
            console.log('Error checking window objects:', e);
          }

          // DOM ELEMENT CHECKS
          // Check for user UI elements that would only be visible when logged in
          const userProfileElements = [
            '#user-display-name',
            '#user-initials',
            '#user-avatar',
            '#profile-email',
            '#logout-button',
            '.user-avatar',
            '.user-profile',
            '.user-menu'
          ];

          for (const selector of userProfileElements) {
            const elements = document.querySelectorAll(selector);
            if (elements && elements.length > 0) {
              for (const element of elements) {
                if ((element.innerText && element.innerText.trim().length > 0) ||
                  window.getComputedStyle(element).display !== 'none') {
                  return { loggedIn: true, reason: `Found user element ${selector}` };
                }
              }
            }
          }

          // Check for app-specific elements that would only be present for logged in users
          const appSpecificElements = [
            '#reports-section',
            '#recorder-section',
            '#report-section',
            '.reports-list-container',
            '#back-to-reports',
            '.my-account',
            '.my-reports',
            '.dashboard',
            '.account-section'
          ];

          for (const selector of appSpecificElements) {
            const elements = document.querySelectorAll(selector);
            if (elements && elements.length > 0) {
              for (const element of elements) {
                if (window.getComputedStyle(element).display !== 'none') {
                  return { loggedIn: true, reason: `Found app element ${selector}` };
                }
              }
            }
          }

          // Check for login page elements that would only be visible when not logged in
          const loginPageElements = [
            '#firebaseui-auth-container',
            '.firebaseui-container',
            '.login-container',
            '.login-form',
            '.sign-in-form',
            '.sign-in-button:not(.hidden)',
            '#login-button:not(.hidden)',
            '#sign-in-button:not(.hidden)'
          ];

          for (const selector of loginPageElements) {
            const elements = document.querySelectorAll(selector);
            if (elements && elements.length > 0) {
              for (const element of elements) {
                if (window.getComputedStyle(element).display !== 'none') {
                  return { loggedIn: false, reason: `Found login element ${selector}` };
                }
              }
            }
          }

          // Only check for logout/sign out links as definitive login indicators
          // BUT first check if there are also login elements visible
          const allLoginElements = Array.from(document.querySelectorAll('*')).filter(el =>
            (el.textContent && el.textContent.toLowerCase().includes('log in')) ||
            (el.textContent && el.textContent.toLowerCase().includes('sign in')) ||
            (el.id && el.id.toLowerCase().includes('login')) ||
            (el.className && el.className.toLowerCase().includes('login'))
          );

          const visibleLoginElements = allLoginElements.filter(el =>
            window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0
          );

          if (visibleLoginElements.length > 0) {
            return { loggedIn: false, reason: 'Found visible login elements' };
          }

          // Only then check for logout links
          const logoutLinks = Array.from(document.querySelectorAll('a, button')).filter(el =>
            (el.textContent && el.textContent.includes('Logout')) ||
            (el.textContent && el.textContent.includes('Sign Out'))
          );

          const visibleLogoutLinks = logoutLinks.filter(el =>
            window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0
          );

          if (visibleLogoutLinks.length > 0) {
            return { loggedIn: true, reason: 'Found logout/sign out link' };
          }

          // Default: Assume not logged in for safety
          return { loggedIn: false, reason: 'No login indicators found' };
        }
      });

      // Check if we got a result
      if (checkResults && checkResults[0] && typeof checkResults[0].result?.loggedIn === 'boolean') {
        const result = checkResults[0].result;
        console.log(`Standard login check for tab ${tabId}: ${result.loggedIn} (${result.reason})`);
        return result.loggedIn;
      }
    } catch (error) {
      console.log(`Error executing login check script in tab ${tabId}:`, error);
      // Continue to try content script approach
    }

    // Now try the content script approach as a backup
    try {
      // Ensure content script is loaded first
      await ensureContentScriptLoaded(tabId);

      const response = await chrome.tabs.sendMessage(tabId, { action: 'CHECK_LOGIN' });
      if (response && typeof response.loggedIn === 'boolean') {
        console.log(`Content script login check for tab ${tabId}: ${response.loggedIn}`);
        return response.loggedIn;
      }
    } catch (error) {
      console.log(`Content script login check failed for tab ${tabId}:`, error.message);
      // Continue to final checks
    }

    // If we get here, something went wrong with all approaches
    // Let's check specifically for main site login as a last resort if that's where we are
    if (isMainSite) {
      try {
        // Final strict check - look for definitive logout button or user profile elements
        const strictLoginCheck = await chrome.scripting.executeScript({
          target: { tabId: tabId },
          func: () => {
            // Only consider logged in if we can find a logout button or user profile
            const logoutButtons = Array.from(document.querySelectorAll('*')).filter(el =>
              (el.textContent && el.textContent.toLowerCase().includes('logout')) ||
              (el.textContent && el.textContent.toLowerCase().includes('sign out')) ||
              (el.id && el.id.toLowerCase().includes('logout')) ||
              (el.className && el.className.toLowerCase().includes('logout'))
            );

            const userProfileElements = Array.from(document.querySelectorAll('*')).filter(el =>
              (el.id && el.id.toLowerCase().includes('user-display-name')) ||
              (el.id && el.id.toLowerCase().includes('user-initials')) ||
              (el.id && el.id.toLowerCase().includes('user-avatar')) ||
              (el.className && el.className.toLowerCase().includes('user-avatar')) ||
              (el.className && el.className.toLowerCase().includes('user-profile'))
            );

            // Must have visible logout button OR user profile elements
            const hasLogout = logoutButtons.some(el =>
              window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0
            );

            const hasUserProfile = userProfileElements.some(el =>
              window.getComputedStyle(el).display !== 'none' && el.offsetWidth > 0 && el.offsetHeight > 0
            );

            return hasLogout || hasUserProfile;
          }
        });

        if (strictLoginCheck && strictLoginCheck[0] && strictLoginCheck[0].result === true) {
          console.log(`Strict login check passed for tab ${tabId} - found logout button or user profile`);
          return true;
        }
      } catch (error) {
        console.log(`Error performing final app link check in tab ${tabId}:`, error);
      }
    }

    console.log(`All login detection methods failed for tab ${tabId}, user is NOT logged in`);
    return false;
  } catch (error) {
    console.error(`Error checking login state for tab ${tabId}:`, error);
    return false;
  }
}

/**
 * Ensure content script is loaded in tab and ready for communication
 */
async function ensureContentScriptLoaded(tabId) {
  try {
    // First try to ping the content script
    const pingResponse = await chrome.tabs.sendMessage(tabId, { action: 'PING' });
    console.log('Content script already loaded in tab', tabId);
    return true;
  } catch (error) {
    // Content script not loaded, inject it
    console.log('Content script not found in tab', tabId, 'injecting...');

    try {
      await chrome.scripting.executeScript({
        target: { tabId: tabId },
        files: ['content/content-script.js']
      });

      // Wait for script to initialize
      await new Promise(resolve => setTimeout(resolve, 1000));

      // Verify injection worked
      try {
        await chrome.tabs.sendMessage(tabId, { action: 'PING' });
        console.log('Content script successfully injected and verified in tab', tabId);
        return true;
      } catch (verifyError) {
        throw new Error('Content script injection failed - could not verify');
      }
    } catch (injectionError) {
      throw new Error(`Failed to inject content script: ${injectionError.message}`);
    }
  }
}

/**
 * Handle loading formats from TextRad web app
 */
async function handleLoadFormats(tabId, sendResponse) {
  try {
    // If no tabId provided, find active TextRad tab
    if (!tabId) {
      const tabs = await findTextRadTabs();
      if (tabs.length === 0) {
        throw new Error('No logged-in TextRad tabs found. Please log in to TextRad in your browser.');
      }
      tabId = tabs[0].id;
    }

    // Verify tab is still valid
    const tab = await chrome.tabs.get(tabId);
    if (!tab || !isTextRadUrl(tab.url)) {
      throw new Error('The selected TextRad tab is no longer valid');
    }

    // Ensure content script is loaded
    await ensureContentScriptLoaded(tabId);

    // Send message to content script to get formats
    const response = await chrome.tabs.sendMessage(tabId, {
      action: 'GET_FORMATS'
    });

    if (response && response.success) {
      // Cache the formats
      await chrome.storage.local.set({
        cachedFormats: response.formats,
        formatsCacheTime: Date.now()
      });

      console.log('Successfully loaded', response.formats.length, 'formats from TextRad');
      sendResponse({ success: true, formats: response.formats });
    } else {
      throw new Error(response?.error || 'Failed to get formats from TextRad');
    }
  } catch (error) {
    console.error('Error loading formats:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Handle generating report using TextRad ReportGenerator
 */
async function handleGenerateReport(tabId, transcription, formatId, sendResponse) {
  try {
    console.log('Generating report...', { tabId, formatId, transcriptionLength: transcription?.length });

    // If no tabId provided, find active TextRad tab
    if (!tabId) {
      const tabs = await findTextRadTabs();
      if (tabs.length === 0) {
        throw new Error('No logged-in TextRad tabs found. Please log in to TextRad in your browser.');
      }
      tabId = tabs[0].id;
    }

    // Verify tab is still valid
    const tab = await chrome.tabs.get(tabId);
    if (!tab || !isTextRadUrl(tab.url)) {
      throw new Error('The selected TextRad tab is no longer valid');
    }

    // Validate inputs
    if (!transcription || transcription.trim().length === 0) {
      throw new Error('No transcription text provided');
    }

    if (!formatId) {
      throw new Error('No format selected');
    }

    // Ensure content script is loaded
    await ensureContentScriptLoaded(tabId);

    // Send message to content script to generate report
    const response = await chrome.tabs.sendMessage(tabId, {
      action: 'GENERATE_REPORT',
      transcription: transcription.trim(),
      formatId: formatId
    });

    if (response && response.success) {
      console.log('Report generated successfully, length:', response.report?.length);

      // Update current state to indicate report was generated
      currentState.lastTranscription = response.report;
      currentState.lastTranscriptionTimestamp = Date.now();
      saveState();

      sendResponse({ success: true, report: response.report });
    } else {
      throw new Error(response?.error || 'Failed to generate report');
    }
  } catch (error) {
    console.error('Error generating report:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Handle tab activation request - TextRad tabs now self-manage, just verify they exist
 */
async function handleActivateTextRadTabs(sendResponse) {
  console.log('🔄 Checking TextRad tab availability (tabs self-manage activation)...');

  try {
    // Find all TextRad tabs
    const tabs = await chrome.tabs.query({
      url: ['*://*.textrad.com/*', '*://*.textrad.in/*', '*://localhost/*', '*://augmento.vijayadiagnostic.com/*', '*://mhe-risweb05.manipalhospitals.com/*']
    });

    if (tabs.length === 0) {
      console.warn('No TextRad tabs found');
      sendResponse({ success: false, error: 'No TextRad tabs found' });
      return;
    }

    console.log(`Found ${tabs.length} self-managing TextRad tabs`);

    // Simply verify tabs are responsive (they manage their own keep-alive)
    let responsiveTabs = 0;
    for (const tab of tabs) {
      try {
        // Simple ping to verify communication works
        await chrome.tabs.sendMessage(tab.id, { action: 'PING' });
        responsiveTabs++;
        console.log(`✅ Tab ${tab.id} is responsive`);
      } catch (pingError) {
        console.warn(`Tab ${tab.id} not responsive:`, pingError.message);
      }
    }

    sendResponse({ success: true, totalTabs: tabs.length, responsiveTabs });
    console.log(`✅ Found ${responsiveTabs}/${tabs.length} responsive TextRad tabs`);

  } catch (error) {
    console.error('Error checking TextRad tabs:', error);
    sendResponse({ success: false, error: error.message });
  }
}

/**
 * Handle service health updates from content scripts
 */
function handleServiceHealthUpdate(tabId, message, sendResponse) {
  // Update the service health status for this tab
  tabServiceHealth[tabId] = {
    isReady: message.isReady,
    services: message.services,
    consecutiveFailures: message.consecutiveFailures,
    lastCheck: Date.now()
  };

  // Update tab tracking
  if (textradTabs[tabId]) {
    textradTabs[tabId].servicesReady = message.isReady;
    textradTabs[tabId].lastHealthCheck = Date.now();
  }

  console.log(`Service health update for tab ${tabId}:`, {
    isReady: message.isReady,
    consecutiveFailures: message.consecutiveFailures
  });

  // Service health is now managed by TextRad web app's keep-alive system
  if (message.consecutiveFailures >= 5) {
    console.info(`Tab ${tabId} reporting service issues - TextRad web app will self-manage recovery`);
  }

  saveState();
  sendResponse({ success: true });
}

/**
 * Get the best available TextRad tab (most recently active with ready services)
 */
function getBestTextRadTab() {
  const availableTabs = Object.entries(textradTabs)
    .filter(([tabId, tabInfo]) => {
      const health = tabServiceHealth[tabId];
      return health && health.isReady;
    })
    .sort(([, a], [, b]) => (b.lastActivated || 0) - (a.lastActivated || 0));

  return availableTabs.length > 0 ? parseInt(availableTabs[0][0]) : null;
}

/**
 * Enhanced tab management - proactively keep tabs active
 */
setInterval(async () => {
  // Every 2 minutes, check if we have active TextRad tabs with ready services
  const readyTabs = Object.entries(tabServiceHealth)
    .filter(([, health]) => health.isReady && (Date.now() - health.lastCheck) < 120000) // Ready within last 2 minutes
    .length;

  if (readyTabs === 0) {
    console.log('No TextRad tabs with ready services, proactively activating...');
    try {
      await handleActivateTextRadTabs(() => { });
    } catch (error) {
      console.warn('Proactive tab activation failed:', error);
    }
  }
}, 120000); // Every 2 minutes

/**
 * Save state to storage
 */
function saveState() {
  chrome.storage.local.set({
    textradTabs: textradTabs,
    currentState: currentState,
    tabServiceHealth: tabServiceHealth
  });
}

/**
 * TextRad web app now handles its own keep-alive internally
 * Extension focuses only on communication and message routing
 */
console.log('🚀 TextRad Extension - communication-focused mode (web app handles keep-alive)');

// Initialize service worker
initialize();