/**
 * TextRad Dictation Extension - Content Script
 *
 * Responsibilities:
 * - Inject into TextRad web app
 * - Communicate with extension background script
 * - Interact with TextRad web app
 * - Inject dictation data into TextRad UI
 */

// Prevent duplicate script execution
if (window.textRadExtensionContentScriptLoaded) {
  console.log('TextRad Extension Content Script already loaded, skipping...');
  // Don't return here, just log and continue - the script needs to work after extension reloads
}
window.textRadExtensionContentScriptLoaded = true;

console.log('TextRad Extension Content Script - Starting load...', {
  url: window.location.href,
  readyState: document.readyState,
  textRadApp: typeof window.textRadApp !== 'undefined'
});

// Track elements and state - use var to avoid redeclaration errors
var transcriptionElement = null;
var formatSelectorElement = null;
var currentFormat = null;
var isConnected = false;
var serviceHealthStatus = {
  isReady: false,
  lastCheck: null,
  checkInProgress: false,
  consecutiveFailures: 0,
  services: {
    textRadApp: false,
    firebase: false,
    auth: false,
    functions: false,
    formatManager: false,
    reportGenerator: false
  }
};
var serviceHealthInterval = null;

/**
 * Initialize content script
 */
function initialize() {
  console.log('TextRad Dictation Extension content script initialized');

  // ALWAYS set up message listeners, even on non-TextRad pages
  // This allows manual deployment via DEPLOY_MINI_EXTENSION message
  setupMessageListeners();

  // Initialize mini extension (will check for TextRad pages or session persistence)
  initializePersistentMiniExtension();

  // Check if this is a TextRad page
  if (!isTextRadPage()) {
    console.log('Non-TextRad page - listeners active, awaiting manual deployment');
    return;
  }

  // Log page info
  console.log('TextRad page detected:', window.location.href);

  // Setup mutation observer AFTER confirming this is a TextRad page
  // This prevents unnecessary DOM monitoring on non-app pages
  setupMutationObserver();

  // Wait for the page to be fully loaded
  if (document.readyState === 'complete') {
    initializeOnTextRadPage();
  } else {
    window.addEventListener('load', initializeOnTextRadPage);
  }
}

/**
 * Start continuous monitoring of the generate button
 * This ensures the button stays visible after format selection
 */
var generateButtonMonitoringInterval = null;
function startGenerateButtonMonitoring() {
  // Clear any existing interval
  if (generateButtonMonitoringInterval) {
    clearInterval(generateButtonMonitoringInterval);
  }

  // Set a new interval that regularly checks and enforces visibility
  generateButtonMonitoringInterval = setInterval(() => {
    // Check if we have the conditions for showing the generate button
    const transcriptionText = document.getElementById('transcription-text');
    const formatSelector = document.getElementById('format-selector');

    // If we have both transcription content and a selected format
    if (transcriptionText &&
      transcriptionText.value &&
      transcriptionText.value.trim().length > 0 &&
      formatSelector &&
      formatSelector.value &&
      formatSelector.value !== "") {

      // Check the visibility of the generate button
      const generateContainer = document.querySelector('#generate-container');
      const generateButton = document.querySelector('#generate-button');

      // If either the container or button exists and is hidden/not visible, make it visible
      if ((generateContainer &&
        (generateContainer.classList.contains('hidden') ||
          generateContainer.style.display === 'none')) ||
        (generateButton &&
          (generateButton.classList.contains('hidden') ||
            generateButton.style.display === 'none'))) {

        console.log('Generate button monitor: Button should be visible but is hidden, fixing...');
        ensureGenerateButtonVisible();
      }
    }
  }, 500); // Check every 500ms to avoid too much CPU usage

  console.log('Generate button continuous monitoring started');
}

/**
 * Initialize on confirmed TextRad page
 */
function initializeOnTextRadPage() {
  console.log('Initializing on TextRad page - skipping CSP-violating injections');

  // Inject page context script early for save functionality
  injectPageContextScript();

  // Connect to background script
  connectToBackground();

  // Note: setupListeners() is already called in initialize() for all pages

  // Find key elements
  findUIElements();

  // Try again in 2 seconds if elements weren't found
  if (!transcriptionElement) {
    setTimeout(findUIElements, 2000);
  }
}

/**
 * Inject a script to help with detecting login state
 * This runs in page context and can access window variables directly
 * DISABLED: Violates CSP - causes inline script errors
 */
function injectLoginDetectionScript_DISABLED() {
  const script = document.createElement('script');
  script.textContent = `
    // TextRad Login Detection Script
    (function() {
      console.log('TextRad Login Detection Script initialized');
      
      // Function to check login state
      function checkLoginState() {
        let isLoggedIn = false;
        let reason = 'unknown';
        
        // Detect if we're on main site or app
        const isMainSite = window.location.pathname === '/' || 
                         window.location.pathname.includes('/index') || 
                         window.location.pathname === '';
                         
        const isAppPage = window.location.pathname.includes('/app');
        
        console.log('Checking login state on ' + (isMainSite ? 'main site' : (isAppPage ? 'app page' : 'other page')));
        
        // Check for all possible auth indicators
        
        // 1. Check Firebase auth - most reliable when available
        if (window.firebase && window.firebase.auth && window.firebase.auth().currentUser) {
          isLoggedIn = true;
          reason = 'firebase auth detected';
        } 
        // 2. Check TextRad app state - for app page
        else if (window.textradApp && window.textradApp.user) {
          isLoggedIn = true;
          reason = 'textradApp.user detected';
        }
        // 3. Check TextRad app authentication state
        else if (window.textradApp && window.textradApp.isAuthenticated === true) {
          isLoggedIn = true;
          reason = 'textradApp.isAuthenticated = true';
        }
        // 4. Check TextRad app current user
        else if (window.textradApp && window.textradApp.currentUser) {
          isLoggedIn = true;
          reason = 'textradApp.currentUser detected';
        }
        // 5. Check other app variables
        else if (window.USER_DATA || window.user || window.currentUser) {
          isLoggedIn = true;
          reason = 'global user data object detected';
        }
        // 6. Check for Firebase auth in localStorage
        else if (localStorage.getItem('firebase:authUser:AIzaSyBiQwFjWCjJA1vXANQ2igNMlkTwAk8CJzA:[DEFAULT]')) {
          isLoggedIn = true;
          reason = 'firebase auth in localStorage';
        }
        // 7. Check other localStorage auth items
        else if (localStorage.getItem('authUser') || 
                localStorage.getItem('textrad_user') || 
                localStorage.getItem('user_data')) {
          isLoggedIn = true;
          reason = 'auth data in localStorage';
        }
        // 8. Check cookies for auth tokens
        else {
          try {
            const cookies = document.cookie.split('; ');
            for (const cookie of cookies) {
              const [name, value] = cookie.split('=');
              if ((name.includes('auth') || name.includes('session') || name.includes('token') || name.includes('user')) && 
                  value && value.length > 10) {
                isLoggedIn = true;
                reason = 'auth cookie detected: ' + name;
                break;
              }
            }
          } catch (e) {
            console.error('Error checking cookies:', e);
          }
        }
        
        // If still not determined, check DOM elements
        if (!isLoggedIn) {
          // Check for login/logout UI elements on main site
          if (isMainSite) {
            // Check for elements visible when logged in on main site
            const mainSiteLoggedInSelectors = [
              // Navigation and menu elements when logged in
              '.account-menu', '.user-menu', '.profile-menu', '.dashboard-menu',
              // User display elements
              '.user-avatar', '.user-profile', '.user-initials', '.user-greeting',
              // Common dashboard elements
              '.dashboard', '.dashboard-panel', '.dashboard-content',
              // Account-related sections
              '.my-account', '.my-reports', '.my-dashboard', '.account-section',
              // Action buttons only visible when logged in
              '#logout-button', '.logout-button', '.sign-out-button'
            ];
            
            for (const selector of mainSiteLoggedInSelectors) {
              const elements = document.querySelectorAll(selector);
              if (elements && elements.length > 0) {
                for (const element of elements) {
                  // Check if element is visible and has content
                  if (window.getComputedStyle(element).display !== 'none' &&
                      element.offsetWidth > 0 && element.offsetHeight > 0) {
                    isLoggedIn = true;
                    reason = 'main site logged-in element: ' + selector;
                    break;
                  }
                }
                if (isLoggedIn) break;
              }
            }
            
            // If not determined yet, check for visible links to /app page
            // On main site, a link to the app usually means the user is logged in
            if (!isLoggedIn) {
              const appLinks = Array.from(document.querySelectorAll('a[href*="/app"]')).filter(a => 
                window.getComputedStyle(a).display !== 'none' && a.offsetWidth > 0 && a.offsetHeight > 0
              );
              
              if (appLinks.length > 0) {
                isLoggedIn = true;
                reason = 'visible app link on main site';
              }
            }
            
            // If not determined yet, check for elements visible when NOT logged in
            if (!isLoggedIn) {
              const mainSiteLoggedOutSelectors = [
                '.login-button', '.sign-in-button', '.auth-button',
                '#login-button', '#sign-in-button',
                '[href*="login"]', '[href*="signin"]', '[href*="auth"]'
              ];
              
              let foundLoggedOutElement = false;
              for (const selector of mainSiteLoggedOutSelectors) {
                const elements = document.querySelectorAll(selector);
                if (elements && elements.length > 0) {
                  for (const element of elements) {
                    // Check if the login element is visible
                    if (window.getComputedStyle(element).display !== 'none' &&
                        element.offsetWidth > 0 && element.offsetHeight > 0) {
                      foundLoggedOutElement = true;
                      reason = 'visible login element: ' + selector;
                      break;
                    }
                  }
                  if (foundLoggedOutElement) break;
                }
              }
              
              // If we found logged-out element, we are NOT logged in
              if (foundLoggedOutElement) {
                isLoggedIn = false;
              }
            }
          }
          // For app page, check app-specific elements
          else if (isAppPage) {
            const appPageElements = [
              '#reports-section',
              '#recorder-section',
              '#transcription-container',
              '#format-selector',
              '.reports-list-container',
              '#back-to-reports',
              '#user-display-name',
              '#user-initials',
              '#user-avatar',
              '#profile-email',
              '#logout-button'
            ];
            
            for (const selector of appPageElements) {
              const elements = document.querySelectorAll(selector);
              if (elements && elements.length > 0) {
                for (const element of elements) {
                  if (window.getComputedStyle(element).display !== 'none') {
                    isLoggedIn = true;
                    reason = 'app element visible: ' + selector;
                    break;
                  }
                }
                if (isLoggedIn) break;
              }
            }
          }
        }
        
        // Dispatch result to content script
        document.dispatchEvent(new CustomEvent('textrad-login-state', {
          detail: { 
            isLoggedIn: isLoggedIn,
            reason: reason,
            isMainSite: isMainSite,
            isAppPage: isAppPage,
            timestamp: Date.now()
          }
        }));
        
        return isLoggedIn;
      }
      
      // Check immediately
      checkLoginState();
      
      // Set up interval to check periodically
      setInterval(checkLoginState, 10000);
      
      // Expose for extension to call
      window.checkTextRadLoginState = checkLoginState;
    })();
  `;

  // Append to document
  (document.head || document.documentElement).appendChild(script);

  // Clean up
  script.remove();

  // Listen for login state events from the injected script
  document.addEventListener('textrad-login-state', (event) => {
    const { isLoggedIn, reason, isMainSite, isAppPage } = event.detail;
    console.log(`Login state from injected script: ${isLoggedIn} (${reason}) on ${isMainSite ? 'main site' : (isAppPage ? 'app page' : 'other page')}`);

    // Store this information for the content script to use
    window.detectedLoginState = {
      isLoggedIn: isLoggedIn,
      reason: reason,
      isMainSite: isMainSite,
      isAppPage: isAppPage,
      timestamp: Date.now()
    };
  });
}

/**
 * Check if current page is a TextRad page and user is logged in
 */
function isTextRadPage() {
  // First check if we're on a TextRad domain
  const isTextRadDomain = window.location.hostname.includes('textrad.com') ||
    window.location.hostname.includes('textrad.in') ||
    window.location.hostname.includes('localhost') ||
    window.location.hostname.includes('augmento.vijayadiagnostic.com') ||
    window.location.hostname.includes('mhe-risweb05.manipalhospitals.com') ||
    window.location.hostname.includes('mhmkaengsrv');

  if (!isTextRadDomain) {
    return false;
  }

  // Check if user is logged in
  const isLoggedIn = checkIfUserIsLoggedIn();

  // For debugging - print the full URL to see if we're on main site or app page
  const isMainSite = window.location.pathname === '/' ||
    window.location.pathname.includes('/index.html') ||
    window.location.pathname === '';

  const isAppPage = window.location.pathname.includes('/app');

  console.log(`TextRad page check - URL: ${window.location.href}, Domain: ${isTextRadDomain}, ` +
    `Main site: ${isMainSite}, App page: ${isAppPage}, Logged in: ${isLoggedIn}`);

  if (!isLoggedIn) {
    console.log('User not logged in to TextRad, not considering as active TextRad page');
    return false;
  }

  // IMPORTANT: For TextRad domains, only activate on /app/* pages
  // Landing page (/) should not be considered an active TextRad page
  if (isTextRadDomain && (window.location.hostname.includes('textrad.com') || window.location.hostname.includes('textrad.in') || window.location.hostname.includes('localhost'))) {
    if (!isAppPage) {
      console.log('TextRad domain but not on /app/ page - not considering as active TextRad page');
      return false;
    }
  }

  return true;
}

/**
 * Get detailed TextRAD page status including whether user is on the /app/ page
 * This is used by popup to show guidance messages
 */
function getTextRadPageStatus() {
  const isTextRadDomain = window.location.hostname.includes('textrad.com') ||
    window.location.hostname.includes('textrad.in') ||
    window.location.hostname.includes('localhost');

  const isLoggedIn = checkIfUserIsLoggedIn();
  const isAppPage = window.location.pathname.includes('/app');
  const url = window.location.href;

  return {
    isTextRadDomain,
    isLoggedIn,
    isAppPage,
    url,
    needsAppPage: isTextRadDomain && isLoggedIn && !isAppPage // Logged in but not on /app/
  };
}

/**
 * Check if user is logged in to TextRad
 */
function checkIfUserIsLoggedIn() {
  // First, check our injected script results which is the most reliable
  if (window.detectedLoginState && window.detectedLoginState.isLoggedIn) {
    console.log(`Injected script detected logged in state: ${window.detectedLoginState.reason}`);
    return true;
  }

  // Try to call the window function if available
  try {
    if (window.checkTextRadLoginState && typeof window.checkTextRadLoginState === 'function') {
      const isLoggedIn = window.checkTextRadLoginState();
      if (isLoggedIn) {
        console.log('Login check via window function successful');
        return true;
      }
    }
  } catch (error) {
    console.log('Error calling window login check function:', error);
  }

  // Check for Firebase auth directly
  if (window.firebase && window.firebase.auth && window.firebase.auth().currentUser) {
    console.log('Firebase auth indicates user is logged in');
    return true;
  }

  // Check for authentication elements on the main page
  const authElements = [
    // User menu and profile elements (both main site and app)
    '.user-menu',
    '.user-profile',
    '.profile-menu',
    '.account-menu',
    '.user-avatar',
    '#user-avatar',
    '#user-profile-menu',
    '#user-menu',
    '#user-display-name',
    '#user-initials',
    '#logout-button',
    '#profile-email',

    // Dashboard elements that indicate logged-in state
    '.dashboard-content',
    '.account-dashboard',
    '.reports-dashboard',
    '.user-reports',

    // App-specific elements 
    '#reports-section:not(.hidden)',
    '#recorder-section:not(.hidden)',
    '#report-section:not(.hidden)',
    '.reports-list-container:not(.hidden)',
    '#back-to-reports',

    // Main site elements for logged-in users
    '.my-account',
    '.my-reports',
    '.subscription-details',
    '.welcome-message'
  ];

  for (const selector of authElements) {
    const elements = document.querySelectorAll(selector);
    if (elements && elements.length > 0) {
      for (const element of elements) {
        // Check if element has content or is visible
        if (
          (element.innerText && element.innerText.trim().length > 0) ||
          (element.textContent && element.textContent.trim().length > 0) ||
          element.classList.contains('logged-in') ||
          window.getComputedStyle(element).display !== 'none'
        ) {
          console.log(`Found logged-in indicator: ${selector}`);
          return true;
        }
      }
    }
  }

  // Check for login form elements - if these are visible, user is NOT logged in
  const loginFormElements = [
    '#firebaseui-auth-container',
    '.firebaseui-container',
    '.login-container',
    '.login-form',
    '.sign-in-form',
    '.sign-in-button:not(.hidden)',
    '#login-button:not(.hidden)',
    '#sign-in-button:not(.hidden)',
    '.auth-form'
  ];

  for (const selector of loginFormElements) {
    const elements = document.querySelectorAll(selector);
    if (elements && elements.length > 0) {
      for (const element of elements) {
        if (window.getComputedStyle(element).display !== 'none') {
          console.log(`Found login form element, user appears to be logged out: ${selector}`);
          return false;
        }
      }
    }
  }

  // On main page, check for special dashboard elements
  if (window.location.pathname === '/' || window.location.pathname === '/index.html') {
    // Check for dashboard elements
    const dashboardElements = [
      '.dashboard',
      '.dashboard-panel',
      '.reports-panel',
      '.user-welcome',
      '.account-section'
    ];

    for (const selector of dashboardElements) {
      if (document.querySelector(selector)) {
        console.log(`Found dashboard element on main page: ${selector}, user appears logged in`);
        return true;
      }
    }

    // Check for "My Account" or similar links
    const accountLinks = Array.from(document.querySelectorAll('a')).filter(a =>
      a.textContent.includes('Account') ||
      a.textContent.includes('Profile') ||
      a.textContent.includes('Dashboard') ||
      a.textContent.includes('Reports')
    );

    if (accountLinks.length > 0) {
      console.log('Found account-related links, user appears logged in');
      return true;
    }
  }

  // On app page, check if the format selector is present
  if (window.location.pathname.includes('/app')) {
    const appElements = [
      '#format-selector',
      '#transcription-container',
      '#recorder-section',
      '#reports-section'
    ];

    for (const selector of appElements) {
      if (document.querySelector(selector)) {
        console.log(`Found app element: ${selector}, user appears logged in`);
        return true;
      }
    }
  }

  // Check cookies for authentication token
  if (document.cookie.includes('auth') ||
    document.cookie.includes('token') ||
    document.cookie.includes('session')) {
    console.log('Found authentication cookie, user may be logged in');
    return true;
  }

  // Default: If we can't definitively determine login state, assume not logged in for safety
  console.log('Could not determine login state, assuming user is not logged in');
  return false;
}

/**
 * Connect to background script
 */
/**
 * Comprehensive service health check to ensure all TextRad services are ready
 */
async function checkServiceHealth() {
  if (serviceHealthStatus.checkInProgress) {
    console.log('Service health check already in progress, skipping...');
    return serviceHealthStatus.isReady;
  }

  serviceHealthStatus.checkInProgress = true;
  serviceHealthStatus.lastCheck = Date.now();

  try {
    const services = serviceHealthStatus.services;

    // Since keep-alive system is working and direct report generation works,
    // simplify the health check to always return true when on TextRad domain
    const isOnTextRadDomain = window.location.href.includes('textrad') ||
      window.location.href.includes('localhost') ||
      window.location.href.includes('augmento.vijayadiagnostic.com') ||
      window.location.href.includes('mhe-risweb05.manipalhospitals.com') ||
      window.location.href.includes('mhmkaengsrv');

    // Check for keep-alive system activity
    const keepAliveActive = !!(
      window.textRadKeepAlive ||
      localStorage.getItem('textrad-active') === 'true' ||
      localStorage.getItem('textrad-heartbeat')
    );

    // Basic check for any content that suggests the page is loaded
    const hasBasicContent = document.body && document.body.textContent.length > 100;

    // Simplified ready check - if we're on the right domain and have basic content, consider ready
    // For custom domains (augmento, manipal), be more lenient since they're custom domains
    const isCustomDomain = window.location.href.includes('augmento.vijayadiagnostic.com') ||
      window.location.href.includes('mhe-risweb05.manipalhospitals.com') ||
      window.location.href.includes('mhmkaengsrv');
    const isReady = isOnTextRadDomain && (keepAliveActive || hasBasicContent || isCustomDomain);

    // Update all services to ready if basic check passes
    if (isReady) {
      services.textRadApp = true;
      services.firebase = true;
      services.auth = true;
      services.functions = true;
      services.formatManager = true;
      services.reportGenerator = true;
    }

    serviceHealthStatus.isReady = isReady;

    if (isReady) {
      serviceHealthStatus.consecutiveFailures = 0;
      console.log('✅ Service health check passed - TextRad is ready (simplified check)');
    } else {
      serviceHealthStatus.consecutiveFailures++;
      console.warn('⚠️ Service health check failed (simplified):', {
        isOnTextRadDomain,
        keepAliveActive,
        hasBasicContent,
        consecutiveFailures: serviceHealthStatus.consecutiveFailures
      });
    }

    // Notify background script of service status
    try {
      chrome.runtime.sendMessage({
        action: 'SERVICE_HEALTH_UPDATE',
        isReady: isReady,
        services: services,
        consecutiveFailures: serviceHealthStatus.consecutiveFailures
      });
    } catch (error) {
      console.warn('Could not notify background of service health:', error.message);
    }

    return isReady;
  } catch (error) {
    console.error('Error during service health check:', error);
    serviceHealthStatus.isReady = false;
    serviceHealthStatus.consecutiveFailures++;
    return false;
  } finally {
    serviceHealthStatus.checkInProgress = false;
  }
}

/**
 * Start continuous service health monitoring
 */
function startServiceHealthMonitoring() {
  if (serviceHealthInterval) {
    clearInterval(serviceHealthInterval);
  }

  // Check immediately
  checkServiceHealth();

  // Check every 10 seconds, or more frequently if services are failing
  const getHealthCheckInterval = () => {
    if (serviceHealthStatus.consecutiveFailures >= 3) {
      return 5000; // Check every 5 seconds when failing
    }
    return 10000; // Normal 10 second interval
  };

  const scheduleHealthCheck = () => {
    serviceHealthInterval = setTimeout(async () => {
      await checkServiceHealth();

      // If services are not ready for too long, try to reactivate the tab
      if (serviceHealthStatus.consecutiveFailures >= 3) {
        console.warn('Services not ready for 3+ consecutive checks, requesting tab activation...');
        try {
          chrome.runtime.sendMessage({ action: 'ACTIVATE_TEXTRAD_TABS' });
        } catch (error) {
          console.warn('Could not request tab activation:', error.message);
        }
      }

      // Schedule the next check with dynamic interval
      scheduleHealthCheck();
    }, getHealthCheckInterval());
  };

  // Start the dynamic health checking
  scheduleHealthCheck();

  console.log('Service health monitoring started');
}

/**
 * Stop service health monitoring
 */
function stopServiceHealthMonitoring() {
  if (serviceHealthInterval) {
    clearTimeout(serviceHealthInterval);
    serviceHealthInterval = null;
    console.log('Service health monitoring stopped');
  }
}

/**
 * Wait for services to be ready with timeout
 */
async function waitForServicesReady(timeoutMs = 10000) {
  const startTime = Date.now();

  while (Date.now() - startTime < timeoutMs) {
    const isReady = await checkServiceHealth();
    if (isReady) {
      return true;
    }

    // Wait 1 second before next check (reduced from 2 seconds)
    await new Promise(resolve => setTimeout(resolve, 1000));
  }

  console.error('Services did not become ready within timeout');
  return false;
}

function connectToBackground() {
  try {
    chrome.runtime.sendMessage({ action: 'CONNECTED_TO_TEXTRAD' }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('Error connecting to background:', chrome.runtime.lastError);
        return;
      }

      if (response && response.success) {
        isConnected = true;
        console.log('Connected to TextRad Extension background script');

        // Start service health monitoring once connected
        startServiceHealthMonitoring();
      }
    });
  } catch (error) {
    console.log('Could not connect to background (extension context may be invalidated):', error.message);
  }
}

// Create persistent mini extension (convert popup mini view to floating overlay)
async function createPersistentMiniExtension() {
  // Check if mini extension already exists
  if (document.getElementById('textrad-mini-persistent')) {
    return;
  }

  // Pre-fetch saved position and size BEFORE creating element to avoid flash
  const currentDomain = window.location.hostname;
  let savedPosition = null;
  let savedSize = null;
  let useAppearAnimation = false;

  try {
    const result = await chrome.storage.local.get(['miniExtensionSizes', 'miniExtensionPositions']);
    const sizes = result.miniExtensionSizes || {};
    const positions = result.miniExtensionPositions || {};

    savedSize = sizes[currentDomain] || null;
    savedPosition = positions[currentDomain] || null;
    useAppearAnimation = !!savedPosition; // Use appear animation if we have a saved position
  } catch (error) {
    console.error('Error pre-fetching mini extension data:', error);
  }

  // Create the mini extension container
  const miniExtension = document.createElement('div');
  miniExtension.id = 'textrad-mini-persistent';
  miniExtension.innerHTML = `
    <div class="mini-extension-content">
      <div class="mini-extension-top-row">
        <button id="mini-record-btn" class="mini-record-button" title="Start Recording">
          <i class="fas fa-microphone"></i>
        </button>
        <button id="mini-expand-btn" class="mini-expand-button" title="Open Full Extension">
          <i class="fas fa-expand-alt"></i>
        </button>
        <button id="mini-close-btn" class="mini-close-button" title="Close Mini Extension">
          <i class="fas fa-times"></i>
        </button>
      </div>
      <div class="mini-extension-bottom-row" id="mini-bottom-row">
        <div class="mini-status-wrapper">
          <div class="status-indicator" id="mini-status-indicator"></div>
          <span class="status-text" id="mini-status-text"></span>
        </div>
      </div>
      <div class="mini-resize-handle" id="mini-resize-handle" title="Drag to resize">
        <svg width="12" height="12" viewBox="0 0 12 12" fill="none">
          <path d="M11 1L1 11M11 5L5 11M11 9L9 11" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"/>
        </svg>
      </div>
    </div>
  `;

  // Ensure Font Awesome is loaded for the mini extension
  if (!document.querySelector('link[href*="font-awesome"]')) {
    const fontAwesomeLink = document.createElement('link');
    fontAwesomeLink.rel = 'stylesheet';
    fontAwesomeLink.href = 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css';
    fontAwesomeLink.crossOrigin = 'anonymous';
    document.head.appendChild(fontAwesomeLink);
  }

  // Calculate dynamic sizing based on screen resolution
  const screenWidth = window.innerWidth;
  let sizeScale = 0.71; // Default scale (increased by 18%)

  if (screenWidth < 1366) {
    // Small screens (laptops, tablets)
    sizeScale = 0.59;
  } else if (screenWidth >= 1366 && screenWidth < 1920) {
    // Medium screens (standard monitors)
    sizeScale = 0.71;
  } else if (screenWidth >= 1920 && screenWidth < 2560) {
    // Large screens (full HD+)
    sizeScale = 0.77;
  } else {
    // Extra large screens (4K+)
    sizeScale = 0.83;
  }

  // Add CSS styles for the persistent mini extension
  const style = document.createElement('style');
  style.textContent = `
    /* TextRad Brand Color Variables */
    :root {
      --textrad-primary: #60a5fa;
      --textrad-secondary: #34d399;
      --textrad-accent: #8b5cf6;
      --textrad-dark: #111827;
      --textrad-card: #1f2937;
      --textrad-border: #4b5563;
      --textrad-text: #f9fafb;
      --textrad-text-muted: #d1d5db;
      --textrad-text-dim: #9ca3af;
      --textrad-glass: rgba(31, 41, 55, 0.85);
      --mini-scale: ${sizeScale};
    }

    #textrad-mini-persistent {
      position: fixed;
      top: 0;
      left: 0;
      z-index: 2147483647;
      font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      user-select: none;
      transform: translate(calc(100vw - 140px * var(--mini-scale)), calc(20px * var(--mini-scale)));
      transition: transform 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    }

    #textrad-mini-persistent.slide-in {
      animation: miniPersistentSlideIn 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
    }

    #textrad-mini-persistent.appear {
      animation: miniPersistentAppear 0.35s cubic-bezier(0.4, 0, 0.2, 1);
    }

    /* Enhanced Glassmorphism with Noise Texture */
    .mini-extension-content {
      position: relative;
      background: linear-gradient(135deg,
        rgba(31, 41, 55, 0.9) 0%,
        rgba(17, 24, 39, 0.85) 100%);
      backdrop-filter: blur(20px) saturate(180%);
      -webkit-backdrop-filter: blur(20px) saturate(180%);
      border-radius: calc(16px * var(--mini-scale));
      border: 1px solid rgba(96, 165, 250, 0.2);
      box-shadow:
        0 0 0 1px rgba(255, 255, 255, 0.05) inset,
        0 12px 40px -8px rgba(0, 0, 0, 0.4),
        0 8px 24px -4px rgba(0, 0, 0, 0.3),
        0 4px 12px -2px rgba(96, 165, 250, 0.1);
      transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
      overflow: hidden;
      box-sizing: border-box;
      width: calc(130px * var(--mini-scale));
      min-width: calc(130px * var(--mini-scale));
      max-width: 100vw;
    }

    /* Noise Texture Overlay */
    .mini-extension-content::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-image: url("data:image/svg+xml,%3Csvg viewBox='0 0 400 400' xmlns='http://www.w3.org/2000/svg'%3E%3Cfilter id='noiseFilter'%3E%3CfeTurbulence type='fractalNoise' baseFrequency='0.9' numOctaves='4' /%3E%3C/filter%3E%3Crect width='100%25' height='100%25' filter='url(%23noiseFilter)' opacity='0.05'/%3E%3C/svg%3E");
      opacity: 0.4;
      pointer-events: none;
      z-index: 1;
    }

    .mini-extension-content > * {
      position: relative;
      z-index: 2;
    }

    .mini-extension-content:hover {
      border-color: rgba(96, 165, 250, 0.4);
      box-shadow:
        0 0 0 1px rgba(255, 255, 255, 0.08) inset,
        0 16px 48px -10px rgba(0, 0, 0, 0.5),
        0 12px 32px -6px rgba(96, 165, 250, 0.2),
        0 8px 16px -4px rgba(52, 211, 153, 0.1);
      transform: translateY(-1px);
    }

    .mini-extension-top-row {
      display: flex;
      align-items: center;
      justify-content: center;
      gap: calc(10px * var(--mini-scale));
      padding: calc(10px * var(--mini-scale)) calc(7px * var(--mini-scale));
      cursor: move;
      box-sizing: border-box;
      width: 100%;
    }

    .mini-extension-bottom-row {
      display: flex;
      align-items: center;
      justify-content: center;
      padding: 0 calc(7px * var(--mini-scale)) calc(3px * var(--mini-scale)) calc(7px * var(--mini-scale));
      max-height: 0;
      opacity: 0;
      overflow: hidden;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      box-sizing: border-box;
      width: 100%;
    }

    .mini-extension-bottom-row.visible {
      max-height: calc(30px * var(--mini-scale));
      opacity: 1;
      padding-top: calc(4px * var(--mini-scale));
    }

    .mini-status-wrapper {
      display: flex;
      align-items: center;
      gap: calc(6px * var(--mini-scale));
      line-height: 1;
      max-width: 100%;
      overflow: hidden;
    }

    /* Enhanced Status Indicator */
    .status-indicator {
      width: calc(7px * var(--mini-scale));
      height: calc(7px * var(--mini-scale));
      border-radius: 50%;
      transition: all 0.3s ease;
      box-shadow: 0 0 4px rgba(0, 0, 0, 0.3);
      flex-shrink: 0;
    }

    .status-indicator.connected {
      background: var(--textrad-secondary);
      box-shadow: 0 0 6px rgba(52, 211, 153, 0.6);
    }

    .status-indicator.connecting {
      background: #fbbf24;
      animation: miniPersistentStatusBlink 1s infinite;
      box-shadow: 0 0 6px rgba(251, 191, 36, 0.6);
    }

    .status-indicator.disconnected {
      background: #ef4444;
      box-shadow: 0 0 6px rgba(239, 68, 68, 0.6);
    }

    .status-indicator.recording {
      background: #ef4444;
      animation: miniPersistentStatusBlink 1s infinite;
      box-shadow: 0 0 8px rgba(239, 68, 68, 0.8);
    }

    .status-indicator.transcribing {
      background: var(--textrad-primary);
      animation: miniPersistentStatusBlink 1s infinite;
      box-shadow: 0 0 6px rgba(96, 165, 250, 0.6);
    }

    /* Status Text */
    .status-text {
      font-size: calc(11px * var(--mini-scale));
      color: var(--textrad-text-muted);
      font-weight: 500;
      white-space: nowrap;
      letter-spacing: 0.01em;
      line-height: 1;
      display: flex;
      align-items: center;
      overflow: hidden;
      text-overflow: ellipsis;
      max-width: 100%;
    }

    .status-text.success {
      color: var(--textrad-secondary);
    }

    /* Enhanced Mini Record Button - Primary Action */
    .mini-record-button {
      position: relative;
      width: calc(40px * var(--mini-scale)) !important;
      height: calc(40px * var(--mini-scale)) !important;
      border-radius: 50% !important;
      background: linear-gradient(135deg, #60a5fa 0%, #34d399 100%) !important;
      color: white !important;
      border: none !important;
      cursor: pointer !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      font-size: calc(14px * var(--mini-scale)) !important;
      transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
      box-shadow:
        0 0 0 1px rgba(255, 255, 255, 0.15) inset,
        0 4px 12px rgba(96, 165, 250, 0.4),
        0 2px 6px rgba(52, 211, 153, 0.3),
        0 1px 3px rgba(0, 0, 0, 0.2) !important;
    }

    .mini-record-button::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      border-radius: 50%;
      background: linear-gradient(135deg, rgba(255, 255, 255, 0.2), transparent);
      opacity: 0;
      transition: opacity 0.3s ease;
    }

    .mini-record-button:hover::before {
      opacity: 1;
    }

    .mini-record-button:hover {
      transform: scale(1.15) !important;
      box-shadow:
        0 0 0 1px rgba(255, 255, 255, 0.2) inset,
        0 6px 16px rgba(96, 165, 250, 0.5),
        0 4px 10px rgba(52, 211, 153, 0.4),
        0 2px 6px rgba(0, 0, 0, 0.25) !important;
    }

    .mini-record-button:active {
      transform: scale(1.05) !important;
      transition: all 0.1s ease !important;
    }

    .mini-record-button.recording {
      background: linear-gradient(135deg, #ef4444 0%, #f97316 100%) !important;
      animation: miniPersistentRecordingPulse 2s infinite !important;
    }

    /* Enhanced Mini Expand Button */
    .mini-expand-button {
      width: calc(28px * var(--mini-scale)) !important;
      height: calc(28px * var(--mini-scale)) !important;
      border-radius: calc(8px * var(--mini-scale)) !important;
      background: rgba(75, 85, 99, 0.25) !important;
      border: 1px solid rgba(255, 255, 255, 0.1) !important;
      color: var(--textrad-text-dim) !important;
      cursor: pointer !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      font-size: calc(10px * var(--mini-scale)) !important;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
      backdrop-filter: blur(4px) !important;
      box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1) !important;
    }

    .mini-expand-button:hover {
      background: rgba(96, 165, 250, 0.25) !important;
      border-color: rgba(96, 165, 250, 0.3) !important;
      color: var(--textrad-primary) !important;
      transform: scale(1.1) !important;
      box-shadow: 0 3px 10px rgba(96, 165, 250, 0.25) !important;
    }

    .mini-expand-button:active {
      transform: scale(1.05) !important;
      transition: all 0.1s ease !important;
    }

    /* Mini Close Button */
    .mini-close-button {
      width: calc(28px * var(--mini-scale)) !important;
      height: calc(28px * var(--mini-scale)) !important;
      border-radius: calc(8px * var(--mini-scale)) !important;
      background: rgba(239, 68, 68, 0.2) !important;
      border: 1px solid rgba(239, 68, 68, 0.3) !important;
      color: #ef4444 !important;
      cursor: pointer !important;
      display: flex !important;
      align-items: center !important;
      justify-content: center !important;
      font-size: calc(10px * var(--mini-scale)) !important;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1) !important;
      backdrop-filter: blur(4px) !important;
      box-shadow: 0 2px 6px rgba(239, 68, 68, 0.15) !important;
    }

    .mini-close-button:hover {
      background: rgba(239, 68, 68, 0.35) !important;
      border-color: rgba(239, 68, 68, 0.5) !important;
      color: #dc2626 !important;
      transform: scale(1.1) rotate(90deg) !important;
      box-shadow: 0 3px 10px rgba(239, 68, 68, 0.3) !important;
    }

    .mini-close-button:active {
      transform: scale(1.05) !important;
      transition: all 0.1s ease !important;
    }

    /* Resize Handle */
    .mini-resize-handle {
      position: absolute;
      bottom: 0;
      right: 0;
      width: calc(20px * var(--mini-scale));
      height: calc(20px * var(--mini-scale));
      cursor: nwse-resize;
      display: flex;
      align-items: flex-end;
      justify-content: flex-end;
      padding: calc(4px * var(--mini-scale));
      z-index: 10;
      pointer-events: all;
    }

    .mini-resize-handle svg {
      display: none;
    }

    /* Enhanced Animations */
    @keyframes miniPersistentSlideIn {
      from {
        opacity: 0;
        transform: translate(calc(100vw - 120px * var(--mini-scale)), calc(10px * var(--mini-scale))) scale(0.95);
      }
      to {
        opacity: 1;
        transform: translate(calc(100vw - 140px * var(--mini-scale)), calc(20px * var(--mini-scale))) scale(1);
      }
    }

    @keyframes miniPersistentAppear {
      from {
        opacity: 0;
        transform: translate(var(--saved-x, 0), var(--saved-y, 0)) scale(0.95);
      }
      to {
        opacity: 1;
        transform: translate(var(--saved-x, 0), var(--saved-y, 0)) scale(1);
      }
    }

    @keyframes miniPersistentStatusBlink {
      0% { opacity: 0.6; }
      50% { opacity: 1; }
      100% { opacity: 0.6; }
    }

    @keyframes miniPersistentRecordingPulse {
      0% {
        box-shadow:
          0 2px 8px rgba(239, 68, 68, 0.4),
          0 1px 4px rgba(0, 0, 0, 0.2),
          0 0 0 0 rgba(239, 68, 68, 0.6);
      }
      50% {
        box-shadow:
          0 4px 12px rgba(239, 68, 68, 0.5),
          0 2px 6px rgba(0, 0, 0, 0.3),
          0 0 0 6px rgba(239, 68, 68, 0);
      }
      100% {
        box-shadow:
          0 2px 8px rgba(239, 68, 68, 0.4),
          0 1px 4px rgba(0, 0, 0, 0.2),
          0 0 0 0 rgba(239, 68, 68, 0);
      }
    }

    /* Ensure Font Awesome icons display properly */
    .mini-record-button i,
    .mini-expand-button i,
    .mini-close-button i {
      font-family: "Font Awesome 6 Free" !important;
      font-weight: 900 !important;
      font-style: normal !important;
    }
  `;

  document.head.appendChild(style);

  // Apply saved size BEFORE appending to DOM to avoid flash
  if (savedSize) {
    const content = miniExtension.querySelector('.mini-extension-content');
    if (content) {
      content.style.width = `${savedSize.width}px`;
      content.style.setProperty('--mini-scale', savedSize.scale);
      console.log(`📏 Pre-applied mini extension size for ${currentDomain}:`, savedSize);
    }
  }

  // Apply saved position and animation BEFORE appending to DOM
  if (savedPosition) {
    // Validate position is still within current viewport bounds
    const content = miniExtension.querySelector('.mini-extension-content');
    if (content) {
      const rect = content.getBoundingClientRect();
      const maxX = window.innerWidth - rect.width;
      const maxY = window.innerHeight - rect.height;

      // Constrain saved position to current viewport
      const validX = Math.min(Math.max(0, savedPosition.x), maxX);
      const validY = Math.min(Math.max(0, savedPosition.y), maxY);

      // Set CSS custom properties for animation
      miniExtension.style.setProperty('--saved-x', `${validX}px`);
      miniExtension.style.setProperty('--saved-y', `${validY}px`);

      // Set initial position directly (no flash)
      miniExtension.style.transform = `translate(${validX}px, ${validY}px)`;

      // Add appear animation class
      miniExtension.classList.add('appear');

      console.log(`📍 Pre-applied mini extension position for ${currentDomain}:`, { x: validX, y: validY });
    }
  } else {
    // No saved position - use slide-in animation from default position
    miniExtension.classList.add('slide-in');
  }

  // Now append to DOM (will appear with correct position/animation immediately)
  document.body.appendChild(miniExtension);

  // Make draggable
  makeDraggable(miniExtension);

  // Make resizable
  makeResizable(miniExtension);

  // Add event handlers
  const recordBtn = miniExtension.querySelector('#mini-record-btn');
  const expandBtn = miniExtension.querySelector('#mini-expand-btn');
  const closeBtn = miniExtension.querySelector('#mini-close-btn');

  recordBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    handleMiniRecording();
  });

  expandBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    // Open the extension popup in full mode
    chrome.runtime.sendMessage({ action: 'OPEN_POPUP_FULL_MODE' });
  });

  closeBtn.addEventListener('click', async (e) => {
    e.stopPropagation();

    // Remove this domain from local storage (stops auto-deployment on next page load)
    const currentDomain = window.location.hostname;
    try {
      // Remove from deployed domains, sizes, and positions
      const result = await chrome.storage.local.get(['miniExtensionDeployedDomains', 'miniExtensionSizes', 'miniExtensionPositions']);
      const deployedDomains = result.miniExtensionDeployedDomains || {};
      const sizes = result.miniExtensionSizes || {};
      const positions = result.miniExtensionPositions || {};

      delete deployedDomains[currentDomain];
      delete sizes[currentDomain]; // Also clear saved size
      delete positions[currentDomain]; // Also clear saved position

      await chrome.storage.local.set({
        miniExtensionDeployedDomains: deployedDomains,
        miniExtensionSizes: sizes,
        miniExtensionPositions: positions
      });
      console.log(`Removed ${currentDomain} from storage - mini extension won't auto-deploy on refresh`);
    } catch (error) {
      console.error('Error updating deployed domains:', error);
    }

    // Remove the mini extension from DOM
    miniExtension.remove();
    console.log('Mini extension closed by user');
  });

  // Prevent dragging when clicking buttons
  recordBtn.addEventListener('mousedown', (e) => e.stopPropagation());
  expandBtn.addEventListener('mousedown', (e) => e.stopPropagation());
  closeBtn.addEventListener('mousedown', (e) => e.stopPropagation());

  // Resize handle mousedown is already handled in makeResizable function

  console.log('✅ Persistent mini extension created');
}

// Make element draggable (fixed X and Y axis movement)
function makeDraggable(element) {
  let isDragging = false;
  let startX = 0;
  let startY = 0;
  let elementX = 0;
  let elementY = 0;
  let cachedWidth = 0; // Cache dimensions at drag start
  let cachedHeight = 0;
  let rafId = null; // For requestAnimationFrame
  let pendingX = 0;
  let pendingY = 0;

  element.addEventListener('mousedown', dragStart);
  document.addEventListener('mousemove', dragMove);
  document.addEventListener('mouseup', dragEnd);

  function dragStart(e) {
    // Only start dragging if clicking on drag area (not buttons or resize handle)
    if (e.target.closest('.mini-record-button') ||
      e.target.closest('.mini-expand-button') ||
      e.target.closest('.mini-close-button') ||
      e.target.closest('.mini-extension-bottom-row') ||
      e.target.closest('.mini-resize-handle')) {
      return; // Don't drag when clicking buttons, bottom row, or resize handle
    }

    isDragging = true;
    startX = e.clientX;
    startY = e.clientY;

    // Get current position from transform or use default
    const style = window.getComputedStyle(element);
    const transform = style.transform;

    if (transform && transform !== 'none') {
      const matrix = new DOMMatrix(transform);
      elementX = matrix.m41; // translateX
      elementY = matrix.m42; // translateY
    } else {
      elementX = 0;
      elementY = 0;
    }

    // Cache element dimensions at drag start to avoid recalculation during movement
    const rect = element.getBoundingClientRect();
    cachedWidth = rect.width;
    cachedHeight = rect.height;

    // Disable pointer events on all children during drag to prevent interference
    const children = element.querySelectorAll('*');
    children.forEach(child => {
      child.style.pointerEvents = 'none';
    });

    // Remove any transitions during drag to prevent magnetic wave effect
    element.style.transition = 'none';
    element.style.cursor = 'grabbing';

    // Set high z-index during drag
    element.style.zIndex = '2147483647';

    e.preventDefault();
  }

  function dragMove(e) {
    if (!isDragging) return;

    e.preventDefault();

    // Calculate new position
    const deltaX = e.clientX - startX;
    const deltaY = e.clientY - startY;

    pendingX = elementX + deltaX;
    pendingY = elementY + deltaY;

    // Keep within viewport bounds using cached dimensions
    const maxX = window.innerWidth - cachedWidth;
    const maxY = window.innerHeight - cachedHeight;

    pendingX = Math.min(Math.max(0, pendingX), maxX);
    pendingY = Math.min(Math.max(0, pendingY), maxY);

    // Use requestAnimationFrame for smooth rendering
    if (!rafId) {
      rafId = requestAnimationFrame(() => {
        element.style.transform = `translate(${pendingX}px, ${pendingY}px)`;
        rafId = null;
      });
    }
  }

  async function dragEnd() {
    if (!isDragging) return;

    isDragging = false;
    element.style.cursor = 'move';

    // Cancel any pending RAF
    if (rafId) {
      cancelAnimationFrame(rafId);
      rafId = null;
    }

    // Re-enable pointer events on all children
    const children = element.querySelectorAll('*');
    children.forEach(child => {
      child.style.pointerEvents = '';
    });

    // Update stored position
    const style = window.getComputedStyle(element);
    const transform = style.transform;

    if (transform && transform !== 'none') {
      const matrix = new DOMMatrix(transform);
      elementX = matrix.m41;
      elementY = matrix.m42;
    }

    // Recalculate dimensions after drag in case they changed
    const rect = element.getBoundingClientRect();
    cachedWidth = rect.width;
    cachedHeight = rect.height;

    // Snap to edge functionality - only when released near screen edges
    const snapThreshold = 30; // pixels from edge to trigger snap
    const snapMargin = 20; // final distance from edge after snap

    let snapX = elementX;
    let snapY = elementY;
    let snapped = false;

    // Check horizontal edges
    if (rect.left < snapThreshold) {
      // Snap to left edge
      snapX = snapMargin;
      snapped = true;
    } else if (window.innerWidth - rect.right < snapThreshold) {
      // Snap to right edge
      snapX = window.innerWidth - rect.width - snapMargin;
      snapped = true;
    }

    // Check vertical edges
    if (rect.top < snapThreshold) {
      // Snap to top edge
      snapY = snapMargin;
      snapped = true;
    } else if (window.innerHeight - rect.bottom < snapThreshold) {
      // Snap to bottom edge
      snapY = window.innerHeight - rect.height - snapMargin;
      snapped = true;
    }

    // Apply snap with smooth animation ONLY if near edge
    if (snapped) {
      // Enable transition only for snap animation
      element.style.transition = 'transform 0.3s cubic-bezier(0.4, 0, 0.2, 1)';
      element.style.transform = `translate(${snapX}px, ${snapY}px)`;

      // Update stored position
      elementX = snapX;
      elementY = snapY;

      // Remove transition after snap animation completes to prevent magnetic effect during next drag
      setTimeout(() => {
        element.style.transition = 'transform 0.2s cubic-bezier(0.4, 0, 0.2, 1)';
      }, 300);
    } else {
      // No snap - restore default transition for hover effects
      element.style.transition = 'transform 0.2s cubic-bezier(0.4, 0, 0.2, 1)';
    }

    // Save position to storage for persistence across page refreshes
    try {
      const currentDomain = window.location.hostname;
      const result = await chrome.storage.local.get(['miniExtensionPositions']);
      const positions = result.miniExtensionPositions || {};

      positions[currentDomain] = {
        x: elementX,
        y: elementY,
        timestamp: Date.now()
      };

      await chrome.storage.local.set({ miniExtensionPositions: positions });
      console.log(`💾 Saved mini extension position for ${currentDomain}:`, positions[currentDomain]);
    } catch (error) {
      console.error('Error saving mini extension position:', error);
    }
  }
}

// Make element resizable from bottom-right corner
function makeResizable(element) {
  const resizeHandle = element.querySelector('#mini-resize-handle');
  if (!resizeHandle) return;

  // Get the content wrapper to resize
  const content = element.querySelector('.mini-extension-content');
  if (!content) return;

  // Get the base scale from CSS variable
  const scaleVar = getComputedStyle(document.documentElement).getPropertyValue('--mini-scale').trim();
  const baseScale = parseFloat(scaleVar) || 0.71;

  // Store the base/minimum width (current size when created)
  const baseWidth = 130 * baseScale;
  const minWidth = baseWidth;
  const minHeight = 50 * baseScale;

  let isResizing = false;
  let startX = 0;
  let startY = 0;
  let startWidth = 0;
  let startHeight = 0;

  resizeHandle.addEventListener('mousedown', resizeStart);
  document.addEventListener('mousemove', resizeMove);
  document.addEventListener('mouseup', resizeEnd);

  function resizeStart(e) {
    isResizing = true;
    startX = e.clientX;
    startY = e.clientY;

    // Get current dimensions
    const rect = content.getBoundingClientRect();
    startWidth = rect.width;
    startHeight = rect.height;

    // Disable transitions during resize
    content.style.transition = 'none';

    // Prevent text selection during resize
    e.preventDefault();
    e.stopPropagation();
  }

  function resizeMove(e) {
    if (!isResizing) return;

    e.preventDefault();

    // Calculate new dimensions
    const deltaX = e.clientX - startX;
    const deltaY = e.clientY - startY;

    let newWidth = startWidth + deltaX;
    let newHeight = startHeight + deltaY;

    // Enforce minimum dimensions
    newWidth = Math.max(minWidth, newWidth);
    newHeight = Math.max(minHeight, newHeight);

    // Get element's current position to check viewport constraints
    const elementRect = element.getBoundingClientRect();
    const elementLeft = elementRect.left;
    const elementTop = elementRect.top;

    // Constrain to viewport - prevent overflow beyond right and bottom edges
    const maxWidth = window.innerWidth - elementLeft - 20; // 20px margin from edge
    const maxHeight = window.innerHeight - elementTop - 20;

    newWidth = Math.min(newWidth, maxWidth);
    newHeight = Math.min(newHeight, maxHeight);

    // Calculate new scale based on width ratio
    // This makes all internal elements scale proportionally
    const widthRatio = newWidth / baseWidth;
    const newScale = baseScale * widthRatio;

    // Update the scale for this specific mini extension
    content.style.setProperty('--mini-scale', newScale);

    // Apply new dimensions (only control width, let height auto-adjust to content)
    content.style.width = `${newWidth}px`;
    content.style.minWidth = `${minWidth}px`;
  }

  async function resizeEnd() {
    if (!isResizing) return;

    isResizing = false;

    // Restore transitions
    content.style.transition = 'all 0.4s cubic-bezier(0.4, 0, 0.2, 1)';

    // Save the current dimensions to storage for persistence
    try {
      const currentDomain = window.location.hostname;
      const rect = content.getBoundingClientRect();
      const currentScale = content.style.getPropertyValue('--mini-scale') || baseScale;

      const result = await chrome.storage.local.get(['miniExtensionSizes']);
      const sizes = result.miniExtensionSizes || {};

      sizes[currentDomain] = {
        width: rect.width,
        scale: parseFloat(currentScale)
      };

      await chrome.storage.local.set({ miniExtensionSizes: sizes });
      console.log(`💾 Saved mini extension size for ${currentDomain}:`, sizes[currentDomain]);
    } catch (error) {
      console.error('Error saving mini extension size:', error);
    }
  }

  // Prevent resize handle from triggering drag
  resizeHandle.addEventListener('mousedown', (e) => {
    e.stopPropagation();
  });
}

// Mini extension state tracking and handling
var miniExtensionState = {
  recording: false,
  transcribing: false,
  connected: false
};

// Mini extension audio recorder
var miniAudioRecorder = null;
var miniMediaRecorder = null;
var miniAudioChunks = [];
var miniRecordingTimer = null;
var miniRecordingStartTime = 0;

function updateMiniRecordingTimer() {
  // Safeguard: Check if miniRecordingStartTime is valid
  // If it's 0 (default value), reset it to prevent showing huge numbers
  if (miniRecordingStartTime === 0) {
    console.warn('miniRecordingStartTime was 0, setting to current time');
    miniRecordingStartTime = Date.now();
  }

  const elapsed = Math.floor((Date.now() - miniRecordingStartTime) / 1000);

  // Safeguard: If elapsed time is unreasonable (over 1 hour or negative), reset timestamp
  // This prevents displaying weird huge numbers due to timestamp corruption
  if (elapsed > 3600 || elapsed < 0) {
    console.error('Invalid elapsed time detected:', elapsed, 'seconds. Resetting timestamp.');
    miniRecordingStartTime = Date.now();
    // Show 0:00 instead of invalid value
    const statusText = document.getElementById('mini-status-text');
    const bottomRow = document.getElementById('mini-bottom-row');
    if (statusText && bottomRow) {
      statusText.textContent = 'Recording 0:00';
      bottomRow.classList.add('visible');
    }
    return;
  }

  const minutes = Math.floor(elapsed / 60);
  const seconds = elapsed % 60;
  const timeStr = `${minutes}:${seconds.toString().padStart(2, '0')}`;

  const statusText = document.getElementById('mini-status-text');
  const bottomRow = document.getElementById('mini-bottom-row');

  if (statusText && bottomRow) {
    statusText.textContent = `Recording ${timeStr}`;
    bottomRow.classList.add('visible');
  }
}

function handleMiniRecording() {
  console.log('Mini extension record button clicked...');

  if (miniExtensionState.recording) {
    // Stop recording
    stopMiniRecording();
  } else {
    // Start recording
    startMiniRecording();
  }
}

async function startMiniRecording() {
  try {
    console.log('Starting mini extension recording...');

    // Check if we're starting from a disconnected state (cold start scenario)
    const statusIndicator = document.querySelector('#mini-status-indicator');
    const wasDisconnected = !miniExtensionState.connected ||
      (statusIndicator && statusIndicator.classList.contains('disconnected'));

    if (wasDisconnected) {
      console.log('🔥 Mini extension detected cold start scenario - preparing for retry logic');
      miniExtensionState.needsRetryOnFailure = true;
      miniExtensionState.connected = false;
      updateMiniExtensionUI();
      showMiniNotification('Reconnecting...');
    }

    // Request microphone permission
    const stream = await navigator.mediaDevices.getUserMedia({
      audio: {
        sampleRate: 16000,
        channelCount: 1,
        echoCancellation: true,
        noiseSuppression: true
      }
    });

    // Create media recorder
    miniMediaRecorder = new MediaRecorder(stream, {
      mimeType: 'audio/webm;codecs=opus'
    });

    miniAudioChunks = [];

    miniMediaRecorder.ondataavailable = (event) => {
      if (event.data.size > 0) {
        miniAudioChunks.push(event.data);
      }
    };

    miniMediaRecorder.onstop = () => {
      console.log('Mini recording stopped, processing audio...');
      processMiniRecording();
    };

    miniMediaRecorder.start();

    // Update state
    miniExtensionState.recording = true;

    // Start recording timer
    miniRecordingStartTime = Date.now();
    miniRecordingTimer = setInterval(updateMiniRecordingTimer, 1000);
    updateMiniRecordingTimer(); // Update immediately

    updateMiniExtensionUI();

    console.log('Mini extension recording started');

  } catch (error) {
    console.error('Error starting mini recording:', error);
    showMiniNotification('Microphone access denied');
    miniExtensionState.recording = false;
    updateMiniExtensionUI();
  }
}

function stopMiniRecording() {
  console.log('Stopping mini extension recording...');

  if (miniMediaRecorder && miniMediaRecorder.state === 'recording') {
    miniMediaRecorder.stop();

    // Stop all tracks
    if (miniMediaRecorder.stream) {
      miniMediaRecorder.stream.getTracks().forEach(track => track.stop());
    }
  }

  // Stop recording timer
  if (miniRecordingTimer) {
    clearInterval(miniRecordingTimer);
    miniRecordingTimer = null;
  }

  // Update state
  miniExtensionState.recording = false;
  miniExtensionState.transcribing = true;
  updateMiniExtensionUI();

  showMiniStatusText('Processing...');
}

async function processMiniRecording() {
  try {
    console.log('Processing mini recording audio...');

    // Create audio blob
    const audioBlob = new Blob(miniAudioChunks, { type: 'audio/webm;codecs=opus' });
    console.log('Mini audio blob created:', audioBlob.size, 'bytes');

    if (audioBlob.size < 1000) {
      throw new Error('Recording too short or no audio detected');
    }

    // Send audio blob to background script for transcription
    const reader = new FileReader();
    reader.onload = async () => {
      const audioData = reader.result.split(',')[1]; // Remove data:audio/webm;base64, prefix

      console.log('Sending mini audio for transcription...');

      // Check if chrome.runtime is available
      if (!chrome || !chrome.runtime || !chrome.runtime.sendMessage) {
        console.error('Chrome runtime not available - extension context lost');
        showMiniNotification('Extension connection lost - please reload page');
        resetMiniState();
        return;
      }

      try {
        chrome.runtime.sendMessage({
          action: 'TRANSCRIBE_MINI_AUDIO',
          audioData: audioData,
          mimeType: 'audio/webm;codecs=opus'
        }, (response) => {
          if (chrome.runtime.lastError) {
            console.error('Error sending mini audio for transcription:', chrome.runtime.lastError);
            showMiniNotification('Transcription error');
            resetMiniState();
            return;
          }

          if (response && response.success) {
            console.log('Mini transcription successful:', response.transcription);
            miniExtensionState.connected = true;
            miniExtensionState.needsRetryOnFailure = false;
            handleMiniTranscriptionResult(response.transcription);
          } else {
            console.error('Mini transcription failed:', response?.error);

            // Check if this was a cold start and we should retry automatically
            if (miniExtensionState.needsRetryOnFailure && !miniExtensionState.hasRetriedOnce) {
              console.log('🔄 First transcription failed after disconnection - retrying automatically...');
              miniExtensionState.hasRetriedOnce = true;
              showMiniNotification('Retrying...');

              // Wait 2 seconds then retry the transcription
              setTimeout(() => {
                chrome.runtime.sendMessage({
                  action: 'TRANSCRIBE_MINI_AUDIO',
                  audioData: audioData,
                  mimeType: 'audio/webm;codecs=opus'
                }, (retryResponse) => {
                  if (retryResponse && retryResponse.success) {
                    console.log('🎉 Mini transcription retry successful:', retryResponse.transcription);
                    miniExtensionState.connected = true;
                    miniExtensionState.needsRetryOnFailure = false;
                    miniExtensionState.hasRetriedOnce = false;
                    handleMiniTranscriptionResult(retryResponse.transcription);
                  } else {
                    console.error('Mini transcription retry also failed:', retryResponse?.error);
                    showMiniNotification('Transcription failed');
                    resetMiniState();
                  }
                });
              }, 2000);
            } else {
              showMiniNotification('Transcription failed');
              resetMiniState();
            }
          }
        });
      } catch (error) {
        console.error('Error calling chrome.runtime.sendMessage:', error);

        // Handle extension context invalidated (happens when extension is reloaded)
        if (error.message.includes('Extension context invalidated')) {
          showMiniNotification('Extension updated - please reload page');
          // Show a more prominent notification
          setTimeout(() => {
            if (confirm('TextRad extension was updated. Reload page to continue using the mini extension?')) {
              window.location.reload();
            }
          }, 1000);
        } else {
          showMiniNotification('Extension error - please reload page');
        }
        resetMiniState();
      }
    };

    reader.readAsDataURL(audioBlob);

  } catch (error) {
    console.error('Error processing mini recording:', error);
    showMiniNotification('Recording error: ' + error.message);
    resetMiniState();
  }
}

function handleMiniTranscriptionResult(transcription) {
  console.log('Mini extension transcription result:', transcription);

  // Check if chrome.runtime is available
  if (!chrome || !chrome.runtime || !chrome.runtime.sendMessage) {
    console.error('Chrome runtime not available in handleMiniTranscriptionResult');
    showMiniNotification('Extension connection lost - please reload page');
    resetMiniState();
    return;
  }

  try {
    // Directly set the transcription and trigger the complete workflow
    chrome.runtime.sendMessage({
      action: 'SET_TRANSCRIPTION',
      transcription: transcription
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('Error setting mini transcription:', chrome.runtime.lastError);
        showMiniNotification('Error setting transcription');
        resetMiniState();
        return;
      }

      if (response && response.success) {
        console.log('Mini transcription set successfully');
        // Now send transcription directly to TextRad
        chrome.runtime.sendMessage({
          action: 'SEND_TRANSCRIPTION_TO_TEXTRAD'
        }, (sendResponse) => {
          if (sendResponse && sendResponse.success && sendResponse.injected) {
            showMiniStatusText('Transcription ready!', 2500, 'success');
          } else {
            showMiniStatusText('Transcription ready!', 2500, 'success');
          }
          // Delay resetMiniState to allow success message to be visible
          setTimeout(() => {
            resetMiniState();
          }, 2500);
        });
      } else {
        showMiniNotification('Failed to set transcription');
        resetMiniState();
      }
    });
  } catch (error) {
    console.error('Error in handleMiniTranscriptionResult:', error);
    showMiniNotification('Extension error - please reload page');
    resetMiniState();
  }
}

function resetMiniState() {
  miniExtensionState.recording = false;
  miniExtensionState.transcribing = false;
  miniExtensionState.connected = true;
  miniExtensionState.needsRetryOnFailure = false;
  miniExtensionState.hasRetriedOnce = false;

  // Clear recording timer if it exists
  if (miniRecordingTimer) {
    clearInterval(miniRecordingTimer);
    miniRecordingTimer = null;
  }

  updateMiniExtensionUI();
}

/**
 * Handle transcription injection specifically for format selection workflow
 */
async function handleTranscriptionForFormatSelection(transcription, sendResponse) {
  console.log('Handling transcription for format selection:', transcription.substring(0, 50) + '...');

  try {
    // First, detect UI elements
    findUIElements();

    // Check if there's existing transcription
    if (transcriptionElement && transcriptionElement.value && transcriptionElement.value.trim() !== '') {
      console.log('Found existing transcription, appending new transcription');
      // Append with a space separator
      const existingText = transcriptionElement.value.trim();
      const newText = existingText + ' ' + transcription;
      transcriptionElement.value = newText;

      // Trigger input event to notify the app
      transcriptionElement.dispatchEvent(new Event('input', { bubbles: true }));
      transcriptionElement.dispatchEvent(new Event('change', { bubbles: true }));
    } else {
      console.log('No existing transcription, setting new transcription');
      // Use the Type Report flow to set up transcription area
      const typeReportSuccess = await tryTypeReportFlow(transcription);
      if (!typeReportSuccess) {
        console.log('Type Report flow failed, trying direct injection');
        // Fallback to direct injection
        if (transcriptionElement) {
          transcriptionElement.value = transcription;
          transcriptionElement.dispatchEvent(new Event('input', { bubbles: true }));
          transcriptionElement.dispatchEvent(new Event('change', { bubbles: true }));
        }
      }
    }

    // Ensure format selector is visible (but don't auto-select)
    setTimeout(() => {
      const formatSelectorContainer = document.querySelector('#format-selector-container');
      if (formatSelectorContainer && formatSelectorContainer.classList.contains('hidden')) {
        console.log('Making format selector visible for user selection');
        formatSelectorContainer.classList.remove('hidden');
      }
    }, 600); // Increased delay to allow Type Report flow to complete

    sendResponse({ success: true, injected: true });
  } catch (error) {
    console.error('Error handling transcription for format selection:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Show status text in bottom row
function showMiniStatusText(message, duration = 0, statusType = 'default') {
  const statusText = document.getElementById('mini-status-text');
  const bottomRow = document.getElementById('mini-bottom-row');

  if (statusText && bottomRow) {
    statusText.textContent = message;
    bottomRow.classList.add('visible');

    // Remove all status classes
    statusText.classList.remove('success');

    // Apply status styling
    if (statusType === 'success') {
      statusText.classList.add('success');
    }

    // Auto-hide after duration if specified
    if (duration > 0) {
      setTimeout(() => {
        bottomRow.classList.remove('visible');
        statusText.classList.remove('success');
      }, duration);
    }
  }
}

// Legacy function for backward compatibility (now uses bottom row)
function showMiniNotification(message) {
  showMiniStatusText(message, 2000);
}

function updateMiniExtensionUI(state) {
  const recordBtn = document.querySelector('#mini-record-btn');
  const statusIndicator = document.querySelector('#mini-status-indicator');
  const micIcon = recordBtn?.querySelector('i');
  const bottomRow = document.querySelector('#mini-bottom-row');

  if (!recordBtn || !statusIndicator || !micIcon) return;

  // Update state
  miniExtensionState = { ...miniExtensionState, ...state };

  // Reset classes
  recordBtn.className = 'mini-record-button';
  statusIndicator.className = 'status-indicator';

  // Update based on current state
  if (miniExtensionState.recording) {
    micIcon.className = 'fas fa-stop';
    recordBtn.title = 'Stop Recording';
    recordBtn.classList.add('recording');
    statusIndicator.classList.add('recording');
    // Timer is handled by updateMiniRecordingTimer
  } else if (miniExtensionState.transcribing) {
    micIcon.className = 'fas fa-microphone';
    if (miniExtensionState.needsRetryOnFailure) {
      recordBtn.title = 'Reconnecting...';
      statusIndicator.classList.add('connecting');
      showMiniStatusText('Reconnecting...');
    } else {
      recordBtn.title = 'Transcribing...';
      statusIndicator.classList.add('transcribing');
      showMiniStatusText('Processing...');
    }
  } else {
    micIcon.className = 'fas fa-microphone';
    if (miniExtensionState.connected) {
      recordBtn.title = 'Start Recording';
      statusIndicator.classList.add('connected');
      // Hide bottom row when idle
      if (bottomRow) {
        bottomRow.classList.remove('visible');
      }
    } else {
      recordBtn.title = 'Disconnected - Click to reconnect';
      statusIndicator.classList.add('disconnected');
      showMiniStatusText('Disconnected');
    }
  }
}

// Handle state updates from background script
function handleMiniExtensionStateUpdate(state) {
  updateMiniExtensionUI(state);
}

// Initialize persistent mini extension when page loads
async function initializePersistentMiniExtension() {
  // Check if this domain should have mini extension auto-deployed
  const currentDomain = window.location.hostname;

  // Check domain preferences
  const prefResult = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
  const preferences = prefResult.miniExtensionDomainPreferences || {};
  const domainPreference = preferences[currentDomain];

  if (!domainPreference) {
    // No preference set, skip auto-deploy (except for TextRad domains below)
    console.log(`No deployment preference for ${currentDomain}`);
  } else if (domainPreference === 'always') {
    // Always auto-deploy
    console.log(`Auto-deploying mini extension on ${currentDomain} (always preference)`);
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', createPersistentMiniExtension);
    } else {
      createPersistentMiniExtension();
    }
    return;
  } else if (domainPreference === 'session') {
    // Session-based: check if current session matches stored session
    const sessionResult = await chrome.storage.local.get(['miniExtensionDeployedDomains', 'currentBrowserSessionId']);
    const deployedDomains = sessionResult.miniExtensionDeployedDomains || {};
    const currentSessionId = sessionResult.currentBrowserSessionId;
    const domainData = deployedDomains[currentDomain];

    // Auto-deploy only if session IDs match (same browser session)
    if (domainData && domainData.sessionId === currentSessionId) {
      console.log(`Auto-deploying mini extension on ${currentDomain} (session preference, same session)`);
      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', createPersistentMiniExtension);
      } else {
        createPersistentMiniExtension();
      }
      return;
    } else {
      console.log(`Session preference for ${currentDomain} but different browser session, skipping auto-deploy`);
    }
  }

  // Only auto-deploy on TextRad domains
  // For other domains, wait for manual deployment via DEPLOY_MINI_EXTENSION message
  if (isTextRadPage()) {
    // Extra safety check: verify we're on /app/ path for TextRad domains
    const isTextRadMainDomain = currentDomain.includes('textrad.com') ||
      currentDomain.includes('textrad.in') ||
      currentDomain.includes('localhost');
    const isAppPath = window.location.pathname.includes('/app');

    if (isTextRadMainDomain && !isAppPath) {
      console.log('TextRad domain but not on /app/ path - skipping mini extension auto-deploy');
      return;
    }

    // Wait for page to be ready
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', createPersistentMiniExtension);
    } else {
      createPersistentMiniExtension();
    }
  } else {
    console.log('Non-TextRad domain - mini extension will only deploy on manual request');
  }
}

// Mini extension initialization is now called from initialize() function
// This ensures listeners are set up before any mini extension deployment

/**
 * Set up message listeners only (without mutation observer)
 * This allows the extension to receive messages on all pages
 */
function setupMessageListeners() {
  // Listen for messages from background script
  try {
    chrome.runtime.onMessage.addListener(handleBackgroundMessage);
  } catch (error) {
    console.log('Could not set up chrome.runtime.onMessage listener (extension context may be invalidated):', error.message);
  }

  // Listen for custom events from the web app
  window.addEventListener('textrad-webapp-message', handleWebAppMessage);
}

/**
 * Set up message and event listeners
 * @deprecated Use setupMessageListeners() instead. setupMutationObserver() is now called separately in initialize()
 */
function setupListeners() {
  setupMessageListeners();
  // Note: setupMutationObserver() is now called in initialize() after isTextRadPage() check
}

/**
 * Handle messages from background script
 */
function handleBackgroundMessage(message, sender, sendResponse) {
  console.log('Received message from background:', message);

  // Handle checking if mini extension is deployed
  if (message.action === 'CHECK_MINI_EXTENSION_DEPLOYED') {
    const isDeployed = document.getElementById('textrad-mini-persistent') !== null;
    sendResponse({ deployed: isDeployed });
    return;
  }

  // Handle manual deployment of mini extension
  if (message.action === 'DEPLOY_MINI_EXTENSION') {
    // Use async IIFE and ensure we return true to keep channel open
    (async () => {
      try {
        // Check if mini extension already exists
        if (document.getElementById('textrad-mini-persistent')) {
          console.log('Mini extension already deployed');
          sendResponse({ success: true, alreadyDeployed: true });
          return;
        }

        const currentDomain = window.location.hostname;
        const preference = message.preference || 'session'; // Default to session

        // For session-based deployments, store session information
        if (preference === 'session') {
          const result = await chrome.storage.local.get(['miniExtensionDeployedDomains', 'currentBrowserSessionId']);
          const deployedDomains = result.miniExtensionDeployedDomains || {};
          const currentSessionId = result.currentBrowserSessionId;

          deployedDomains[currentDomain] = {
            sessionId: currentSessionId,
            timestamp: Date.now()
          };

          await chrome.storage.local.set({ miniExtensionDeployedDomains: deployedDomains });
          console.log(`Stored ${currentDomain} for session ${currentSessionId}`);
        }
        // For 'always', preference is already stored in miniExtensionDomainPreferences by popup

        // Deploy mini extension on this page
        if (document.readyState === 'loading') {
          document.addEventListener('DOMContentLoaded', () => {
            createPersistentMiniExtension();
            console.log('Mini extension deployed after DOMContentLoaded');
          });
          // Send response immediately for loading state
          sendResponse({ success: true, deployed: true });
        } else {
          createPersistentMiniExtension();
          console.log('Mini extension deployed immediately');
          sendResponse({ success: true, deployed: true });
        }
      } catch (error) {
        console.error('Error deploying mini extension:', error);
        sendResponse({ success: false, error: error.message });
      }
    })().catch(err => {
      console.error('Uncaught error in DEPLOY_MINI_EXTENSION handler:', err);
      sendResponse({ success: false, error: err.message });
    });
    return true; // Keep message channel open for async response
  }

  // Handle mini extension state updates
  if (message.action === 'UPDATE_MINI_EXTENSION_STATE') {
    handleMiniExtensionStateUpdate(message.state);
    sendResponse({ success: true });
    return;
  }

  // Handle different message types
  switch (message.action) {
    case 'PING':
      // Simple ping to check if content script is loaded
      console.log('Received ping from background script');
      // Also check login state and include it in the response
      const isLoggedIn = checkIfUserIsLoggedIn();
      // Try multiple ways to access TextRad app objects
      const textRadAppDirect = window.textRadApp;
      const textRadAppViaSelf = self.textRadApp;
      const textRadAppViaTop = top?.textRadApp;

      console.log('Content script PING - TextRad objects check:', {
        windowTextRadApp: !!window.textRadApp,
        selfTextRadApp: !!self.textRadApp,
        topTextRadApp: !!top?.textRadApp,
        directAccess: typeof textRadAppDirect,
        formatSelector: !!document.querySelector('#format-selector'),
        url: window.location.href
      });

      // Use the first available TextRad app reference
      const textRadApp = textRadAppDirect || textRadAppViaSelf || textRadAppViaTop;

      sendResponse({
        success: true,
        loggedIn: isLoggedIn,
        textradApp: !!textRadApp,
        formatManager: !!(textRadApp?.formatManager),
        reportGenerator: !!(textRadApp?.reportGenerator),
        hasFormatSelector: !!document.querySelector('#format-selector'),
        url: window.location.href
      });
      break;

    case 'CHECK_LOGIN':
      // Specifically check if the user is logged in
      const loginStatus = checkIfUserIsLoggedIn();
      console.log('Login check result:', loginStatus);
      sendResponse({
        loggedIn: loginStatus
      });
      break;

    case 'GET_PAGE_STATUS':
      // Get detailed page status including app page check
      const pageStatus = getTextRadPageStatus();
      console.log('Page status check:', pageStatus);
      sendResponse(pageStatus);
      break;

    case 'INJECT_TRANSCRIPTION':
      // Check login state before allowing injection
      if (!checkIfUserIsLoggedIn()) {
        console.error('Cannot inject transcription: User not logged in to TextRad');
        sendResponse({
          success: false,
          error: 'User not logged in to TextRad. Please log in and go to TextRad Web App to use the plugin.'
        });
        break;
      }

      const success = injectTranscription(message.transcription);
      sendResponse({ success: success });
      break;

    case 'INJECT_TRANSCRIPTION_FOR_FORMAT_SELECTION':
      // Check login state before allowing injection
      if (!checkIfUserIsLoggedIn()) {
        console.error('Cannot inject transcription: User not logged in to TextRad');
        sendResponse({
          success: false,
          error: 'User not logged in to TextRad. Please log in and go to TextRad Web App to use the plugin.'
        });
        break;
      }

      handleTranscriptionForFormatSelection(message.transcription, sendResponse);
      break;

    case 'GET_TEXTRAD_STATE':
      const state = getTextRadState();
      // Include login state in the response
      state.loggedIn = checkIfUserIsLoggedIn();
      sendResponse({ success: true, state: state });
      break;

    case 'GET_FORMATS':
      // Use existing FormatManager to get all formats
      (async () => {
        try {
          const formats = await getTextRadFormats();
          sendResponse({ success: true, formats: formats });
        } catch (error) {
          console.error('Error getting formats:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'GENERATE_REPORT':
      // Use existing ReportGenerator to generate report
      (async () => {
        try {
          const report = await generateTextRadReport(message.transcription, message.formatId);
          sendResponse({ success: true, report: report });
        } catch (error) {
          console.error('Error generating report:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'INJECT_REPORT':
      // Inject generated report into web app's report panel
      (async () => {
        try {
          const result = await injectReportIntoWebApp(
            message.report,
            message.formatName,
            message.autoSave || false
          );
          sendResponse(result);
        } catch (error) {
          console.error('Error injecting report:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'WAKE_UP_SERVICES':
      // Wake up services - trigger immediate health check and ensure readiness
      console.log('🔄 Received wake-up signal, checking service readiness...');
      (async () => {
        try {
          // Force an immediate service health check
          const isReady = await checkServiceHealth();

          // If services aren't ready, try to reinitialize
          if (!isReady) {
            console.log('Services not ready after wake-up, attempting reinitialization...');

            // Try to trigger any initialization that might be needed
            const textRadApp = window.textRadApp || self.textRadApp || top?.textRadApp;
            if (textRadApp && typeof textRadApp.initialize === 'function') {
              try {
                await textRadApp.initialize();
                console.log('Triggered textRadApp.initialize()');
              } catch (initError) {
                console.warn('textRadApp.initialize() failed:', initError);
              }
            }

            // Check again after initialization attempt
            await checkServiceHealth();
          }

          sendResponse({ success: true, servicesReady: serviceHealthStatus.isReady });
        } catch (error) {
          console.error('Error during wake-up service check:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'KEEP_ALIVE':
      // Simple keep-alive response - just acknowledge that tab is responsive
      sendResponse({ success: true, timestamp: Date.now() });
      return;

    case 'CHECK_AUTH_STATUS':
      // Check if user is authenticated
      const authStatus = checkIfUserIsLoggedIn();
      sendResponse({ authenticated: authStatus });
      return;

    case 'GET_USER_ID':
      // Get userId from page context (for popup/background scripts)
      (async () => {
        try {
          // Use the same page context retrieval we implemented for format selection
          const userIdResult = await new Promise((resolve) => {
            const messageHandler = (event) => {
              if (event.data && event.data.type === 'TEXTRAD_EXTENSION_USER_ID') {
                window.removeEventListener('message', messageHandler);
                resolve(event.data.userId);
              }
            };
            window.addEventListener('message', messageHandler);
            window.postMessage({ type: 'TEXTRAD_EXTENSION_GET_USER_ID' }, '*');

            // Timeout after 1 second
            setTimeout(() => {
              window.removeEventListener('message', messageHandler);
              resolve(null);
            }, 1000);
          });

          sendResponse({ success: true, userId: userIdResult || 'extension-user' });
        } catch (error) {
          console.error('Error getting userId:', error);
          sendResponse({ success: false, userId: 'extension-user' });
        }
      })();
      return true; // Keep message channel open for async response

    case 'SAVE_REPORT_VIA_CONTENT_SCRIPT':
      // Save report via web app's ReportStorage
      (async () => {
        try {
          // Check if user is authenticated
          if (!checkIfUserIsLoggedIn()) {
            sendResponse({ success: false, error: 'User not authenticated' });
            return;
          }

          // Access the web app's ReportStorage
          const textRadApp = window.textRadApp || window.app || self.textRadApp || top?.textRadApp;

          if (!textRadApp || !textRadApp.reportStorage) {
            sendResponse({ success: false, error: 'ReportStorage not available' });
            return;
          }

          // Prepare report data
          const reportData = {
            content: message.reportData.content,
            formatName: message.reportData.formatName,
            source: message.reportData.source || 'extension'
          };

          // Save the report
          const reportId = await textRadApp.reportStorage.addReport(reportData);

          if (reportId) {
            console.log('Report saved via content script:', reportId);
            sendResponse({ success: true, reportId: reportId });
          } else {
            sendResponse({ success: false, error: 'Failed to save report' });
          }
        } catch (error) {
          console.error('Error saving report via content script:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'UPDATE_USAGE_VIA_CONTENT_SCRIPT':
      // Update report usage counter
      (async () => {
        try {
          // Check if user is authenticated
          if (!checkIfUserIsLoggedIn()) {
            sendResponse({ success: false, error: 'User not authenticated' });
            return;
          }

          // Access the web app's ReportGenerator for usage updates
          const textRadApp = window.textRadApp || window.app || self.textRadApp || top?.textRadApp;

          if (!textRadApp || !textRadApp.reportGenerator) {
            sendResponse({ success: false, error: 'ReportGenerator not available' });
            return;
          }

          // Get userId from auth
          const userId = textRadApp.auth?.currentUser?.uid;
          if (!userId) {
            sendResponse({ success: false, error: 'User ID not available' });
            return;
          }

          // Update usage counter with both userId and count
          await textRadApp.reportGenerator.updateReportUsage(userId, message.count);

          console.log(`Updated report usage (+${message.count}) via content script`);
          sendResponse({ success: true });
        } catch (error) {
          console.error('Error updating usage via content script:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response

    case 'CHECK_SUBSCRIPTION_VIA_CONTENT_SCRIPT':
      // Check subscription status
      (async () => {
        try {
          // Check if user is authenticated
          if (!checkIfUserIsLoggedIn()) {
            sendResponse({ success: false, valid: false, reason: 'not_authenticated' });
            return;
          }

          // Access the web app's ReportGenerator for subscription checks
          const textRadApp = window.textRadApp || window.app || self.textRadApp || top?.textRadApp;

          if (!textRadApp || !textRadApp.reportGenerator) {
            sendResponse({ success: false, error: 'ReportGenerator not available' });
            return;
          }

          // Check subscription status (this should be implemented in the web app's ReportGenerator)
          // For now, we'll return a basic valid response
          sendResponse({
            success: true,
            valid: true,
            subscriptionStatus: 'active',
            reportsUsed: 0,
            maxReports: 1000,
            reason: 'valid'
          });
        } catch (error) {
          console.error('Error checking subscription via content script:', error);
          sendResponse({ success: false, error: error.message });
        }
      })();
      return true; // Keep message channel open for async response
  }

  // Return true to indicate async response
  return true;
}

/**
 * Handle custom events from the web app
 */
function handleWebAppMessage(event) {
  const { type, payload } = event.detail;

  console.log('Received message from web app:', type, payload);

  // Forward message to background script
  try {
    chrome.runtime.sendMessage({
      action: 'WEBAPP_MESSAGE',
      type: type,
      payload: payload
    });
  } catch (error) {
    console.log('Could not forward message to background (extension context may be invalidated):', error.message);
  }
}

/**
 * Set up mutation observer to detect UI changes
 */
function setupMutationObserver() {
  const observer = new MutationObserver((mutations) => {
    // Check for relevant UI changes
    let needsUIRefresh = false;

    for (const mutation of mutations) {
      if (mutation.type === 'childList' || mutation.type === 'attributes') {
        needsUIRefresh = true;
        break;
      }
    }

    if (needsUIRefresh) {
      // Refresh UI elements
      findUIElements();
    }
  });

  // Start observing
  observer.observe(document.body, {
    childList: true,
    subtree: true,
    attributes: true,
    attributeFilter: ['class', 'style']
  });
}

/**
 * Find important UI elements in the TextRad app
 */
function findUIElements() {
  console.log('Finding UI elements in TextRad app');

  // Start with known selectors for the TextRad app
  // Try to find the transcription container first using various selectors
  let transcriptionContainer = document.querySelector('.transcription-container');
  if (!transcriptionContainer) {
    // Try alternative selectors in order of likelihood
    const alternativeContainers = [
      '.recorder-container',
      '.dictation-container',
      '.editor-container',
      '.text-input-container',
      '.report-container'
    ];

    for (const selector of alternativeContainers) {
      transcriptionContainer = document.querySelector(selector);
      if (transcriptionContainer) {
        console.log(`Found container using selector: ${selector}`);
        break;
      }
    }
  }

  // If we found a container, look for input elements within it
  if (transcriptionContainer) {
    console.log('Found transcription container:', transcriptionContainer);

    // Try specific ID first (most reliable)
    transcriptionElement = document.querySelector('#transcription-text');

    if (!transcriptionElement) {
      // Look for common TextRad selectors
      const textradSelectors = [
        '#dictation-text',
        '#report-text',
        '#editor-text',
        '.transcription-input',
        '.report-text-area'
      ];

      for (const selector of textradSelectors) {
        transcriptionElement = document.querySelector(selector);
        if (transcriptionElement) {
          console.log(`Found transcription element using selector: ${selector}`);
          break;
        }
      }

      // If still not found, look for any textarea or contenteditable div within the container
      if (!transcriptionElement) {
        transcriptionElement = transcriptionContainer.querySelector('textarea, [contenteditable="true"], .ProseMirror, .editor, .quill-editor');
        console.log('Looked for textarea/contenteditable/editor:', transcriptionElement);
      }
    }

    // If still not found, look more broadly
    if (!transcriptionElement) {
      // Look for any textarea or rich text editor on the page
      const possibleEditors = [
        'textarea',
        '[contenteditable="true"]',
        '.ProseMirror',
        '.ql-editor',
        '.CodeMirror',
        '.editor-instance',
        '.monaco-editor'
      ];

      for (const selector of possibleEditors) {
        transcriptionElement = document.querySelector(selector);
        if (transcriptionElement) {
          console.log(`Found editor using selector: ${selector}`);
          break;
        }
      }
    }

    console.log('Found transcription element:', transcriptionElement);
  } else {
    console.log('No transcription container found, looking for standalone editor elements');

    // If we couldn't find a container, try directly finding editor elements
    const editorSelectors = [
      '#transcription-text',
      '#dictation-text',
      '#report-text',
      'textarea',
      '[contenteditable="true"]',
      '.ProseMirror',
      '.ql-editor',
      '.editor-instance',
      '.CodeMirror',
      '.monaco-editor'
    ];

    for (const selector of editorSelectors) {
      transcriptionElement = document.querySelector(selector);
      if (transcriptionElement) {
        console.log(`Found standalone editor using selector: ${selector}`);
        break;
      }
    }
  }

  // Find format selector - try multiple selectors
  formatSelectorElement = document.querySelector('#format-selector');
  if (!formatSelectorElement) {
    // Try alternative selectors
    const formatSelectors = [
      'select.format-selector',
      'select.report-format',
      'select.template-selector',
      'select:not([disabled])'
    ];

    for (const selector of formatSelectors) {
      formatSelectorElement = document.querySelector(selector);
      if (formatSelectorElement) {
        console.log(`Found format selector using selector: ${selector}`);
        break;
      }
    }
  }

  if (formatSelectorElement) {
    console.log('Found format selector:', formatSelectorElement);

    // Get selected format
    currentFormat = {
      id: formatSelectorElement.value,
      name: formatSelectorElement.options[formatSelectorElement.selectedIndex]?.text || ''
    };

    // Listen for format changes
    formatSelectorElement.addEventListener('change', handleFormatChange);

    console.log('Current format:', currentFormat);
  } else {
    console.log('No format selector found');
  }

  // Log overall findings
  console.log('UI element detection results:', {
    foundTranscriptionElement: !!transcriptionElement,
    foundFormatSelector: !!formatSelectorElement
  });
}

/**
 * Handle format selection changes
 */
function handleFormatChange(event) {
  currentFormat = {
    id: event.target.value,
    name: event.target.options[event.target.selectedIndex]?.text || ''
  };

  // Notify background script of format change
  try {
    chrome.runtime.sendMessage({
      action: 'FORMAT_CHANGED',
      format: currentFormat
    });
  } catch (error) {
    console.log('Could not notify background of format change (extension context may be invalidated):', error.message);
  }

  // Ensure the generate button is visible after format selection
  // This is crucial as the web app often hides it after format selection
  console.log('Format changed to:', currentFormat.name);
  setTimeout(ensureGenerateButtonVisible, 100);
  setTimeout(ensureGenerateButtonVisible, 500);
  setTimeout(ensureGenerateButtonVisible, 1000);
}

/**
 * Get current TextRad state
 */
function getTextRadState() {
  return {
    hasTranscriptionElement: !!transcriptionElement,
    hasFormatSelector: !!formatSelectorElement,
    currentFormat: currentFormat,
    url: window.location.href,
    title: document.title
  };
}

/**
 * Initialize injection bridge for communication with TextRad app
 * DISABLED: Violates CSP - causes inline script errors
 */
function initializeInjectionBridge_DISABLED() {
  // Create and inject a bridge script element
  const script = document.createElement('script');
  script.textContent = `
    // TextRad Extension Bridge
    (function() {
      console.log('TextRad Extension Bridge initialized');
      
      // Set up event listeners for web app to extension communication
      document.addEventListener('textrad-extension-message', (event) => {
        console.log('Bridge received message for extension:', event.detail);
      });
      
      // Expose method to send messages from web app to extension
      window.sendToTextRadExtension = function(type, payload) {
        document.dispatchEvent(new CustomEvent('textrad-webapp-message', {
          detail: { type, payload }
        }));
      };
      
      // Store original app methods
      let originalHandleFormatSelection = null;
      let originalShowGenerateButton = null;
      let originalHideGenerateButton = null;
      let originalUpdateUI = null;
      
      // Monitor the TextRad app object and override methods as soon as they're available
      function setupAppOverrides() {
        if (window.textradApp) {
          console.log('Found textradApp object, overriding methods');
          
          // Store original app functions if they exist
          originalShowGenerateButton = window.textradApp.showGenerateButton;
          originalHideGenerateButton = window.textradApp.hideGenerateButton;
          originalUpdateUI = window.textradApp.updateUI;
          originalHandleFormatSelection = window.textradApp.handleFormatSelection;
          
          // Override the hideGenerateButton function to completely prevent hiding
          window.textradApp.hideGenerateButton = function() {
            console.log('Extension prevented generate button from being hidden');
            // Don't call the original function
            
            // Instead, FORCE the button to be visible
            const generateContainer = document.querySelector('#generate-container');
            if (generateContainer) {
              generateContainer.classList.remove('hidden');
              generateContainer.setAttribute('style', 'display: block !important');
            }
            
            const generateButton = document.querySelector('#generate-button');
            if (generateButton) {
              generateButton.classList.remove('hidden');
              generateButton.setAttribute('style', 'display: inline-block !important');
            }
            
            return false;
          };
          
          // Override the showGenerateButton function to ensure it actually shows
          window.textradApp.showGenerateButton = function() {
            console.log('Extension enforcing generate button visibility');
            
            // Call original function
            if (typeof originalShowGenerateButton === 'function') {
              originalShowGenerateButton.apply(this, arguments);
            }
            
            // Also force our own implementation with !important styles
            const generateContainer = document.querySelector('#generate-container');
            if (generateContainer) {
              generateContainer.classList.remove('hidden');
              generateContainer.setAttribute('style', 'display: block !important');
            }
            
            const generateButton = document.querySelector('#generate-button');
            if (generateButton) {
              generateButton.classList.remove('hidden');
              generateButton.setAttribute('style', 'display: inline-block !important');
            }
            
            return true;
          };
          
          // Override the handleFormatSelection function to keep generate button visible
          if (originalHandleFormatSelection) {
            window.textradApp.handleFormatSelection = function() {
              console.log('Extension detected format selection');
              
              // Call the original function first
              if (typeof originalHandleFormatSelection === 'function') {
                originalHandleFormatSelection.apply(this, arguments);
              }
              
              // Then force the generate button to be visible after a delay
              // to let the app's own logic complete
              setTimeout(() => {
                console.log('Forcing generate button visibility after format selection');
                const generateContainer = document.querySelector('#generate-container');
                if (generateContainer) {
                  generateContainer.classList.remove('hidden');
                  generateContainer.setAttribute('style', 'display: block !important');
                }
                
                const generateButton = document.querySelector('#generate-button');
                if (generateButton) {
                  generateButton.classList.remove('hidden');
                  generateButton.setAttribute('style', 'display: inline-block !important');
                }
              }, 100);
            };
          }
          
          // Hook into the updateUI function to ensure our elements stay visible
          if (originalUpdateUI) {
            window.textradApp.updateUI = function() {
              // Call original function
              if (typeof originalUpdateUI === 'function') {
                originalUpdateUI.apply(this, arguments);
              }
              
              // Check if we have transcription content
              const transcriptionText = document.getElementById('transcription-text');
              if (transcriptionText && transcriptionText.value && transcriptionText.value.trim().length > 0) {
                // If we have content and a format is selected, force show the generate button
                const formatSelector = document.getElementById('format-selector');
                if (formatSelector && formatSelector.value && formatSelector.value !== "") {
                  const generateContainer = document.querySelector('#generate-container');
                  if (generateContainer) {
                    console.log('Updating UI - enforcing generate button visibility');
                    generateContainer.classList.remove('hidden');
                    generateContainer.setAttribute('style', 'display: block !important');
                  }
                  
                  const generateButton = document.querySelector('#generate-button');
                  if (generateButton) {
                    generateButton.classList.remove('hidden');
                    generateButton.setAttribute('style', 'display: inline-block !important');
                  }
                }
              }
            };
          }
        }
      }
      
      // Run setup immediately and also check again after a delay
      setupAppOverrides();
      setTimeout(setupAppOverrides, 500);
      setTimeout(setupAppOverrides, 1000);
      
      // Set up a MutationObserver to watch for generate button hiding
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          if (mutation.type === 'attributes' && 
              (mutation.attributeName === 'class' || mutation.attributeName === 'style')) {
            const target = mutation.target;
            
            // Check if this is the generate container or button being hidden
            if ((target.id === 'generate-container' || target.id === 'generate-button') && 
                (target.classList.contains('hidden') || target.style.display === 'none')) {
                
              // Check if we have transcription and format
              const transcriptionText = document.getElementById('transcription-text');
              const formatSelector = document.getElementById('format-selector');
              
              if (transcriptionText && 
                  transcriptionText.value && 
                  transcriptionText.value.trim().length > 0 &&
                  formatSelector && 
                  formatSelector.value && 
                  formatSelector.value !== "") {
                
                console.log('Observer detected generate button being hidden, making it visible again');
                setTimeout(() => {
                  target.classList.remove('hidden');
                  target.setAttribute('style', 'display: ' + 
                                     (target.id === 'generate-button' ? 'inline-block' : 'block') + 
                                     ' !important');
                }, 0);
              }
            }
          }
        });
      });
      
      // Start observing
      observer.observe(document.body, { 
        attributes: true, 
        attributeFilter: ['class', 'style'],
        subtree: true,
        childList: true
      });
      
      // Listen for format changes to update the generate button visibility
      const formatSelector = document.getElementById('format-selector');
      if (formatSelector) {
        formatSelector.addEventListener('change', (event) => {
          // Small delay to let the app process the change
          setTimeout(() => {
            const transcriptionText = document.getElementById('transcription-text');
            if (transcriptionText && 
                transcriptionText.value && 
                transcriptionText.value.trim().length > 0 &&
                event.target.value && 
                event.target.value !== "") {
              
              console.log('Format changed, forcing generate button visibility');
              
              // Show generate button with !important styles
              const generateContainer = document.querySelector('#generate-container');
              if (generateContainer) {
                generateContainer.classList.remove('hidden');
                generateContainer.setAttribute('style', 'display: block !important');
              }
              
              const generateButton = document.querySelector('#generate-button');
              if (generateButton) {
                generateButton.classList.remove('hidden');
                generateButton.setAttribute('style', 'display: inline-block !important');
              }
            }
          }, 100);
          
          // Check again after a longer delay in case app's logic runs later
          setTimeout(() => {
            const transcriptionText = document.getElementById('transcription-text');
            if (transcriptionText && 
                transcriptionText.value && 
                transcriptionText.value.trim().length > 0 &&
                event.target.value && 
                event.target.value !== "") {
              
              console.log('Format changed (delayed check), forcing generate button visibility');
              
              // Show generate button with !important styles
              const generateContainer = document.querySelector('#generate-container');
              if (generateContainer) {
                generateContainer.classList.remove('hidden');
                generateContainer.setAttribute('style', 'display: block !important');
              }
              
              const generateButton = document.querySelector('#generate-button');
              if (generateButton) {
                generateButton.classList.remove('hidden');
                generateButton.setAttribute('style', 'display: inline-block !important');
              }
            }
          }, 500);
        });
      }
      
      // Announce bridge presence to web app
      document.dispatchEvent(new CustomEvent('textrad-extension-bridge-ready'));
      
      // Announce bridge presence to extension
      document.dispatchEvent(new CustomEvent('textrad-webapp-message', {
        detail: {
          type: 'APP_READY',
          payload: {
            url: window.location.href,
            title: document.title,
            timestamp: Date.now()
          }
        }
      }));
    })();
  `;

  // Append to document
  (document.head || document.documentElement).appendChild(script);

  // Clean up
  script.remove();
}

/**
 * Inject transcription into TextRad UI
 */
function injectTranscription(transcription) {
  console.log('Attempting to inject transcription:', transcription.substring(0, 50) + '...');

  // Refresh UI elements if needed
  if (!transcriptionElement) {
    findUIElements();
  }

  // Check if we still can't find the transcription element
  if (!transcriptionElement) {
    console.error('Transcription element not found, trying one more scan with delay');

    // Try again after a short delay (UI might still be initializing)
    setTimeout(() => {
      findUIElements();

      if (!transcriptionElement) {
        console.error('Transcription element still not found after retry');
        return false;
      } else {
        // Found it on retry, now inject
        performInjection(transcription);
      }
    }, 500);

    return false;
  }

  return performInjection(transcription);
}

/**
 * Actually perform the transcription injection
 */
function performInjection(transcription) {
  // Try to use the "type a report" flow instead of dictation flow
  // This should prevent the generate button from disappearing
  if (tryTypeReportFlow(transcription)) {
    console.log('Successfully used Type Report flow for injection');
    return true;
  }

  // If Type Report flow fails, fall back to the original method
  console.log('Falling back to standard transcription injection flow');

  // First, make sure the transcription container is visible in the web app
  showTranscriptionContainer();

  // Inject transcription
  try {
    if (transcriptionElement.tagName === 'TEXTAREA' || transcriptionElement.tagName === 'INPUT') {
      // For textarea/input elements
      transcriptionElement.value = transcription;

      // Trigger input event
      transcriptionElement.dispatchEvent(new Event('input', { bubbles: true }));

      // Trigger change event
      transcriptionElement.dispatchEvent(new Event('change', { bubbles: true }));

      // Focus the element to ensure TextRad app registers the change
      transcriptionElement.focus();
    } else if (transcriptionElement.isContentEditable) {
      // For contenteditable elements
      transcriptionElement.innerHTML = transcription.replace(/\n/g, '<br>');

      // Trigger input event
      transcriptionElement.dispatchEvent(new InputEvent('input', { bubbles: true }));

      // Focus the element
      transcriptionElement.focus();
    } else if (transcriptionElement.classList.contains('ProseMirror') ||
      transcriptionElement.classList.contains('ql-editor') ||
      transcriptionElement.classList.contains('CodeMirror') ||
      transcriptionElement.classList.contains('monaco-editor')) {
      // Handle special editor frameworks

      // Set content - first try innerHTML
      transcriptionElement.innerHTML = transcription.replace(/\n/g, '<br>');

      // Trigger both input and change events to ensure the framework detects the change
      transcriptionElement.dispatchEvent(new InputEvent('input', { bubbles: true }));
      transcriptionElement.dispatchEvent(new Event('change', { bubbles: true }));

      // If ProseMirror, try its API
      if (transcriptionElement.classList.contains('ProseMirror') && window.ProseMirror) {
        try {
          const view = window.ProseMirror.findView(transcriptionElement);
          if (view && view.dispatch) {
            // This is a simplified approach - actual implementation would depend on ProseMirror setup
            view.dispatch(view.state.tr.insertText(transcription));
          }
        } catch (e) {
          console.log('ProseMirror API access failed, falling back to DOM manipulation:', e);
        }
      }

      // Focus the element
      transcriptionElement.focus();
    } else {
      // Fallback to direct property setting
      transcriptionElement.textContent = transcription;

      // Also try setting innerHTML as a fallback
      try {
        transcriptionElement.innerHTML = transcription.replace(/\n/g, '<br>');
      } catch (e) {
        console.log('innerHTML setting failed, already tried textContent');
      }

      // Try dispatching events anyway
      try {
        transcriptionElement.dispatchEvent(new InputEvent('input', { bubbles: true }));
        transcriptionElement.dispatchEvent(new Event('change', { bubbles: true }));
      } catch (e) {
        console.log('Event dispatching failed for this element type');
      }
    }

    // Trigger custom event for TextRad app
    document.dispatchEvent(new CustomEvent('textrad-transcription-received', {
      detail: { transcription }
    }));

    // Also dispatch an additional custom event for the web app
    // This helps ensure the app's own code responds to the transcription
    document.dispatchEvent(new CustomEvent('transcription-ready', {
      detail: { text: transcription, source: 'extension' }
    }));

    // Make sure the generate button appears
    const generateContainer = document.querySelector('#generate-container');
    if (generateContainer && generateContainer.classList.contains('hidden')) {
      console.log('Making Generate Report button visible');
      generateContainer.classList.remove('hidden');
    }

    // Try to use the app's own function to show the generate button if available
    if (window.textradApp && typeof window.textradApp.showGenerateButton === 'function') {
      console.log('Calling app\'s showGenerateButton function');
      window.textradApp.showGenerateButton();
    }

    // Also try to dispatch a custom event that might trigger the app to show the button
    document.dispatchEvent(new CustomEvent('transcription-content-updated', {
      detail: { hasContent: true }
    }));

    // Set up repeated checks to ensure the generate button stays visible
    // First immediate check
    ensureGenerateButtonVisible();

    // Then check again a few times with increasing delays
    setTimeout(ensureGenerateButtonVisible, 500);
    setTimeout(ensureGenerateButtonVisible, 1000);
    setTimeout(ensureGenerateButtonVisible, 2000);

    // Set up a continuous monitoring interval for the generate button
    // This ensures it's always visible even if the app tries to hide it
    startGenerateButtonMonitoring();

    // Also set up a listener for format selector changes to re-show the button
    const formatSelector = document.querySelector('#format-selector');
    if (formatSelector && !formatSelector.hasChangeListener) {
      formatSelector.addEventListener('change', () => {
        console.log('Format changed, ensuring generate button visibility');
        // Give the app a moment to process the change
        setTimeout(ensureGenerateButtonVisible, 100);
        setTimeout(ensureGenerateButtonVisible, 500);
      });
      formatSelector.hasChangeListener = true;
    }

    console.log('Transcription injected successfully');
    return true;
  } catch (error) {
    console.error('Error injecting transcription:', error);
    return false;
  }
}

/**
 * Try to use the "Type a report" flow instead of dictation
 * According to the user, this flow doesn't have the issue with the generate button disappearing
 */
function tryTypeReportFlow(transcription) {
  try {
    console.log('Attempting to use Type Report flow...');

    // 1. Find the "Type a report" or "Type directly" button/link
    const typeButtonSelectors = [
      '#type-directly-button',
      'button:contains("Type directly")',
      'button:contains("Type a report")',
      'a:contains("Type directly")',
      'a:contains("Type a report")',
      '.type-report-btn',
      '.type-directly-btn'
    ];

    let typeButton = null;

    // Try each selector
    for (const selector of typeButtonSelectors) {
      if (selector.includes(':contains')) {
        // Handle custom :contains selector
        const containsText = selector.match(/:contains\("(.+?)"\)/)[1];
        const elements = document.querySelectorAll(selector.split(':')[0]);
        for (const el of elements) {
          if (el.textContent.includes(containsText)) {
            typeButton = el;
            break;
          }
        }
      } else {
        // Regular selector
        typeButton = document.querySelector(selector);
      }

      if (typeButton) {
        console.log(`Found Type Report button using selector: ${selector}`);
        break;
      }
    }

    // If we couldn't find the button, try looking through all buttons
    if (!typeButton) {
      const allButtons = document.querySelectorAll('button, a');
      for (const button of allButtons) {
        if (button.textContent.toLowerCase().includes('type') ||
          button.innerText.toLowerCase().includes('type') ||
          button.textContent.toLowerCase().includes('manually') ||
          button.innerText.toLowerCase().includes('manually')) {
          typeButton = button;
          console.log('Found Type Report button by scanning all buttons:', button);
          break;
        }
      }
    }

    if (!typeButton) {
      console.log('Could not find Type Report button, aborting Type Report flow');
      return false;
    }

    // 2. Click the button to switch to the "Type a report" flow
    console.log('Clicking Type Report button...');
    typeButton.click();

    // 3. Wait a moment for the UI to update
    // Use a promise to make this function awaitable
    return new Promise((resolve) => {
      setTimeout(() => {
        // 4. Find the transcription textarea in this flow
        const typeTextareaSelectors = [
          '#transcription-text',
          '#type-text',
          '#report-text',
          'textarea',
          '[contenteditable="true"]'
        ];

        let typeTextarea = null;

        for (const selector of typeTextareaSelectors) {
          typeTextarea = document.querySelector(selector);
          if (typeTextarea) {
            console.log(`Found Type Report textarea using selector: ${selector}`);
            break;
          }
        }

        if (!typeTextarea) {
          console.log('Could not find Type Report textarea, aborting Type Report flow');
          resolve(false);
          return;
        }

        // 5. Inject the transcription into the textarea
        try {
          console.log('Injecting transcription into Type Report textarea...');

          if (typeTextarea.tagName === 'TEXTAREA' || typeTextarea.tagName === 'INPUT') {
            typeTextarea.value = transcription;
            typeTextarea.dispatchEvent(new Event('input', { bubbles: true }));
            typeTextarea.dispatchEvent(new Event('change', { bubbles: true }));
          } else if (typeTextarea.isContentEditable) {
            typeTextarea.innerHTML = transcription.replace(/\n/g, '<br>');
            typeTextarea.dispatchEvent(new InputEvent('input', { bubbles: true }));
          } else {
            typeTextarea.textContent = transcription;
            try {
              typeTextarea.innerHTML = transcription.replace(/\n/g, '<br>');
            } catch (e) { /* Ignore */ }

            try {
              typeTextarea.dispatchEvent(new InputEvent('input', { bubbles: true }));
              typeTextarea.dispatchEvent(new Event('change', { bubbles: true }));
            } catch (e) { /* Ignore */ }
          }

          // Focus the textarea
          typeTextarea.focus();

          // 6. Dispatch events to ensure the app processes the change
          document.dispatchEvent(new CustomEvent('textrad-transcription-received', {
            detail: { transcription }
          }));

          document.dispatchEvent(new CustomEvent('transcription-ready', {
            detail: { text: transcription, source: 'extension' }
          }));

          // Log success
          console.log('Successfully injected transcription using Type Report flow');
          resolve(true);
        } catch (error) {
          console.error('Error injecting into Type Report textarea:', error);
          resolve(false);
        }
      }, 500); // Wait for UI to update
    });
  } catch (error) {
    console.error('Error during Type Report flow:', error);
    return false;
  }
}

/**
 * Show the transcription container and necessary UI elements
 */
function showTranscriptionContainer() {
  console.log('Ensuring transcription container is visible');

  // Step 1: Make sure we're in the correct section of the app
  // First, check if we need to open the recorder section
  const recorderSection = document.getElementById('recorder-section');
  const reportsSection = document.getElementById('reports-section');

  // If the recorder section is hidden, we need to show it
  if (recorderSection && recorderSection.classList.contains('hidden')) {
    console.log('Opening recorder section');

    // Try to find and click the "New Report" button to show the recorder
    const newReportButton = Array.from(document.querySelectorAll('button')).find(
      button => button.textContent.includes('New Report') ||
        button.innerHTML.includes('fa-plus')
    );

    if (newReportButton) {
      console.log('Found New Report button, clicking it');
      newReportButton.click();
    } else {
      // Manual show/hide if button not found
      console.log('New Report button not found, manually showing recorder section');
      if (reportsSection) reportsSection.classList.add('hidden');
      recorderSection.classList.remove('hidden');
    }

    // Give the UI a moment to update
    setTimeout(() => ensureTranscriptionContainerVisible(), 100);
  } else {
    // We're already in the recorder section, just ensure the transcription container is visible
    ensureTranscriptionContainerVisible();
  }
}

/**
 * Ensure the transcription container and related UI elements are visible
 */
function ensureTranscriptionContainerVisible() {
  // Step 2: Make sure the transcription container and related UI elements are visible
  const containers = [
    { selector: '#transcription-container', name: 'Transcription container' },
    { selector: '.transcription-container', name: 'Transcription container (class)' },
    { selector: '#format-selector-container', name: 'Format selector' },
    { selector: '#generate-container', name: 'Generate button container' },
    { selector: '#generate-button', name: 'Generate button' },
    { selector: '.recorder-container', name: 'Recorder container' }
  ];

  for (const { selector, name } of containers) {
    const container = document.querySelector(selector);
    if (container) {
      if (container.classList.contains('hidden')) {
        console.log(`Un-hiding ${name}`);
        container.classList.remove('hidden');
      } else {
        console.log(`${name} is already visible`);
      }
    }
  }

  // Step 3: Make sure the format selector is visible, but don't auto-select any format
  const formatSelectorContainer = document.querySelector('#format-selector-container');
  if (formatSelectorContainer && formatSelectorContainer.classList.contains('hidden')) {
    console.log('Making format selector visible');
    formatSelectorContainer.classList.remove('hidden');
  }

  // Do NOT auto-select any format - leave it to the user to choose
}

/**
 * Ensure the generate button is visible
 * This function forces the generate button to appear regardless of app state
 */
function ensureGenerateButtonVisible() {
  const generateContainer = document.querySelector('#generate-container');
  if (generateContainer) {
    if (generateContainer.classList.contains('hidden')) {
      console.log('Making Generate Report button container visible');
      generateContainer.classList.remove('hidden');
    }

    // Also ensure the button itself is visible
    const generateButton = document.querySelector('#generate-button');
    if (generateButton && generateButton.classList.contains('hidden')) {
      console.log('Making Generate Report button visible');
      generateButton.classList.remove('hidden');
    }

    // If there's an inline style hiding it, remove that too
    if (generateContainer.style.display === 'none') {
      generateContainer.style.display = 'block';
    }

    // Force the container to be shown with !important flag via style attribute
    generateContainer.setAttribute('style', 'display: block !important');

    // Also force the button to be shown with !important
    if (generateButton) {
      generateButton.setAttribute('style', 'display: inline-block !important');
    }
  }

  // Try triggering the app's internal functions if available
  if (window.textradApp) {
    if (typeof window.textradApp.showGenerateButton === 'function') {
      window.textradApp.showGenerateButton();
    }

    if (typeof window.textradApp.enableGenerateButton === 'function') {
      window.textradApp.enableGenerateButton();
    }

    if (typeof window.textradApp.updateUI === 'function') {
      window.textradApp.updateUI();
    }

    // Add button back to DOM if it was removed
    if (!document.querySelector('#generate-button')) {
      console.log('Generate button not found in DOM, recreating it');
      const newButton = document.createElement('button');
      newButton.id = 'generate-button';
      newButton.className = 'bg-blue-600 hover:bg-blue-700 text-white font-medium py-2 px-4 rounded-md';
      newButton.innerHTML = 'Generate Report';

      // Add the event listener that would generate a report
      newButton.addEventListener('click', () => {
        if (window.textradApp && typeof window.textradApp.generateReport === 'function') {
          window.textradApp.generateReport();
        }
      });

      // Create container if it doesn't exist
      if (!generateContainer) {
        const newContainer = document.createElement('div');
        newContainer.id = 'generate-container';
        newContainer.className = 'mt-4';
        newContainer.setAttribute('style', 'display: block !important');
        newContainer.appendChild(newButton);

        // Add it to the transcription container
        const transcriptionContainer = document.querySelector('.transcription-container');
        if (transcriptionContainer) {
          transcriptionContainer.appendChild(newContainer);
        }
      } else {
        generateContainer.appendChild(newButton);
      }
    }
  }

  // Dispatch events that might trigger the app
  document.dispatchEvent(new CustomEvent('format-selected', { detail: { hasValidFormat: true } }));
  document.dispatchEvent(new CustomEvent('transcription-ready', { detail: { hasContent: true } }));
}

/**
 * Get formats from TextRad web app using existing FormatManager
 */
async function getTextRadFormats() {
  console.log('Getting formats from TextRad web app...');
  // Try multiple ways to access TextRad app
  const textRadApp = window.textRadApp || self.textRadApp || top?.textRadApp;

  console.log('Available window objects:', {
    textradApp: !!window.textradApp,
    textRadApp: !!window.textRadApp,
    selfTextRadApp: !!self.textRadApp,
    topTextRadApp: !!top?.textRadApp,
    finalTextRadApp: !!textRadApp,
    formatManager: !!(textRadApp?.formatManager),
    isInitialized: textRadApp?.isInitialized,
    isAuthenticated: textRadApp?.isAuthenticated,
    hasFormatSelector: !!document.querySelector('#format-selector')
  });

  // Method 1: Use existing TextRad app's FormatManager (preferred)
  if (textRadApp && textRadApp.formatManager) {
    console.log('Using textRadApp.formatManager.getAllFormats()');

    try {
      const formats = textRadApp.formatManager.getAllFormats();
      console.log('Got formats from FormatManager:', formats?.length);

      if (!formats || formats.length === 0) {
        console.warn('FormatManager returned no formats');
        // Don't throw error, try next method
      } else {
        // Transform to match expected format
        return formats.map(format => ({
          id: format.id,
          name: format.name,
          content: format.content,
          metadata: format.metadata || {}
        }));
      }
    } catch (error) {
      console.error('Error getting formats from FormatManager:', error);
      // Don't throw, try next method
    }
  } else {
    console.warn('TextRad app or FormatManager not available via any method');
  }

  // Method 2: Enhanced DOM format selector fallback with retry logic
  console.log('🔄 Attempting DOM scraping fallback for formats...');

  let formatSelector = null;
  let retryCount = 0;
  const maxRetries = 3;
  const retryDelay = 500;

  // Enhanced format selector discovery with retry logic
  while (!formatSelector && retryCount < maxRetries) {
    console.log(`DOM scraping attempt ${retryCount + 1}/${maxRetries}`);

    // Try multiple selector patterns
    const selectorPatterns = [
      '#format-selector',
      'select[id*="format"]',
      'select[name*="format"]',
      'select.format-selector',
      'select.report-format',
      'select.template-selector'
    ];

    for (const pattern of selectorPatterns) {
      formatSelector = document.querySelector(pattern);
      if (formatSelector && formatSelector.options.length > 1) {
        console.log(`✅ Found format selector using pattern: ${pattern}`);
        break;
      }
    }

    // If still not found, wait and retry
    if (!formatSelector && retryCount < maxRetries - 1) {
      console.log(`⏳ Format selector not found, waiting ${retryDelay}ms before retry...`);
      await new Promise(resolve => setTimeout(resolve, retryDelay));
      retryCount++;
    } else {
      break;
    }
  }

  if (formatSelector && formatSelector.options.length > 1) {
    console.log(`🎯 Successfully using DOM format selector with ${formatSelector.options.length - 1} formats`);

    const formats = [];
    for (let i = 1; i < formatSelector.options.length; i++) {
      const option = formatSelector.options[i];
      if (option.value && option.text) {  // Validate option has required data
        formats.push({
          id: option.value,
          name: option.text.trim(),
          content: null, // Content not available from DOM
          metadata: {
            source: 'DOM',
            selector: formatSelector.id || formatSelector.className || 'unknown'
          }
        });
      }
    }

    console.log('✅ Successfully extracted formats from DOM:', formats.length);
    return formats;
  } else {
    console.log('❌ DOM scraping failed: No suitable format selector found after retries');
  }

  // Method 3: Direct Firestore access if app components aren't ready
  if (window.textRadApp && window.textRadApp.db && window.textRadApp.auth) {
    console.log('Trying direct Firestore access');

    try {
      // Check if we have current user
      if (!window.textRadApp.auth.currentUser) {
        throw new Error('No authenticated user');
      }

      // Use Firestore v9 API - get collection reference
      // Check if we can access collection method (v8 syntax) or need v9 syntax
      let formats = [];

      if (window.textRadApp.db.collection) {
        // Legacy v8 syntax
        console.log('Using Firestore v8 API');
        const userDoc = window.textRadApp.db.collection('users').doc(window.textRadApp.auth.currentUser.uid);
        const formatsCollection = userDoc.collection('formats');
        const snapshot = await formatsCollection.get();

        snapshot.forEach(doc => {
          formats.push({
            id: doc.id,
            ...doc.data()
          });
        });
      } else {
        // Skip direct Firestore access for now - rely on FormatManager
        throw new Error('Direct Firestore access not available - need FormatManager');
      }

      console.log('Got formats from Firestore:', formats.length);
      return formats;
    } catch (error) {
      console.error('Error accessing Firestore directly:', error);
      throw error;
    }
  }

  throw new Error('Cannot access formats - TextRad app may not be fully loaded or user not logged in');
}

/**
 * Get format by ID from available formats with complete template content
 */
async function getFormatById(formatId) {
  try {
    console.log('Getting format by ID:', formatId);

    // First try to get from TextRad app if available
    if (window.textRadApp && window.textRadApp.formatManager) {
      const format = window.textRadApp.formatManager.getFormat(formatId);
      console.log('Format from FormatManager:', {
        name: format?.name,
        hasContent: !!(format?.content || format?.format),
        contentLength: (format?.content || format?.format)?.length || 0
      });

      // Check if format has content - if not, try to trigger format loading
      if (format && !(format.content || format.format)) {
        console.log('Format found but no content, attempting to load format preview...');
        await triggerFormatPreviewLoad(formatId);

        // Try again after loading
        const formatAfterLoad = window.textRadApp.formatManager.getFormat(formatId);
        if (formatAfterLoad && (formatAfterLoad.content || formatAfterLoad.format)) {
          console.log('Format content loaded successfully after preview trigger');
          return formatAfterLoad;
        }
      }

      if (format && (format.content || format.format)) {
        return format;
      }
    }

    // Fallback: get from our cached formats
    console.log('Falling back to cached formats...');
    const formats = await getTextRadFormats();
    const format = formats.find(f => f.id === formatId);

    if (format) {
      console.log('Format from cache:', {
        name: format.name,
        hasContent: !!(format.content || format.format),
        contentLength: (format.content || format.format)?.length || 0
      });

      // If cached format also lacks content, try format preview loading
      if (!(format.content || format.format)) {
        console.log('Cached format also lacks content, attempting format preview load...');
        const loadedFormat = await triggerFormatPreviewLoad(formatId);
        if (loadedFormat && (loadedFormat.content || loadedFormat.format)) {
          return loadedFormat;
        }
      }
    }

    return format;
  } catch (error) {
    console.error('Error getting format by ID:', error);
    return null;
  }
}

/**
 * Trigger format preview loading and extract content directly from DOM
 * This mimics what the DOM simulation method does successfully
 */
async function triggerFormatPreviewLoad(formatId) {
  try {
    console.log('Triggering format preview load for:', formatId);

    // Find the format selector
    const formatSelector = document.querySelector('#format-selector');
    if (!formatSelector) {
      console.log('Format selector not found, cannot trigger preview load');
      return null;
    }

    // Store current value
    const currentValue = formatSelector.value;

    // Set the format and trigger change event to load preview
    console.log('Setting format selector to:', formatId);
    formatSelector.value = formatId;
    formatSelector.dispatchEvent(new Event('change', { bubbles: true }));
    formatSelector.dispatchEvent(new Event('input', { bubbles: true }));

    // Wait for the format preview to load - this is when the content becomes available
    console.log('Waiting for format preview to load...');
    await new Promise(resolve => setTimeout(resolve, 2000));

    // Now extract the format content directly from the DOM preview elements
    // This is the same approach that works in the DOM simulation method
    let formatContent = null;

    console.log('Extracting format content from DOM preview elements...');

    // Look for format preview content in the same places the DOM simulation looks
    const previewSelectors = [
      '.format-preview-content',
      '#format-preview-content',
      '.format-preview',
      '#format-preview',
      '[data-original-content]',
      '.prose.format-transform',
      '.format-content'
    ];

    for (const selector of previewSelectors) {
      const element = document.querySelector(selector);
      if (element) {
        console.log(`Checking preview element ${selector}:`, {
          exists: true,
          hasTextContent: !!element.textContent,
          hasDataOriginal: !!element.getAttribute('data-original-content'),
          textLength: element.textContent?.length || 0,
          dataLength: element.getAttribute('data-original-content')?.length || 0
        });

        // Try to get content from data-original-content first (most reliable)
        let content = element.getAttribute('data-original-content');

        // If not found, try textContent
        if (!content || content.trim().length < 50) {
          content = element.textContent?.trim();
        }

        // If still not found, try innerHTML
        if (!content || content.trim().length < 50) {
          content = element.innerHTML?.trim();
        }

        if (content && content.length > 50) {
          formatContent = {
            id: formatId,
            name: formatSelector.options[formatSelector.selectedIndex]?.text || formatId,
            content: content,
            format: content
          };
          console.log('✅ Format content extracted from DOM element:', {
            selector,
            contentLength: content.length,
            preview: content.substring(0, 100) + '...'
          });
          break;
        }
      }
    }

    // If still no content, try a broader search like the DOM simulation does
    if (!formatContent) {
      console.log('Trying broader DOM search for format content...');

      // Look for any elements that might contain the format template
      const broadSelectors = [
        '.prose',
        'pre',
        '[class*="format"]',
        '[class*="preview"]',
        '[class*="content"]'
      ];

      for (const selector of broadSelectors) {
        const elements = document.querySelectorAll(selector);
        for (const element of elements) {
          if (element.textContent && element.textContent.length > 100) {
            // Check if this looks like a medical format template
            const content = element.textContent.trim();
            const hasFormatKeywords = /TECHNIQUE|FINDINGS|IMPRESSION|CLINICAL HISTORY/i.test(content);

            if (hasFormatKeywords) {
              formatContent = {
                id: formatId,
                name: formatSelector.options[formatSelector.selectedIndex]?.text || formatId,
                content: content,
                format: content
              };
              console.log('✅ Format content found via broad search:', {
                selector: element.tagName + '.' + element.className,
                contentLength: content.length,
                keywords: 'Found medical template keywords'
              });
              break;
            }
          }
        }
        if (formatContent) break;
      }
    }

    // Restore original value
    if (currentValue !== formatId) {
      console.log('Restoring original format selection:', currentValue);
      formatSelector.value = currentValue;
      formatSelector.dispatchEvent(new Event('change', { bubbles: true }));
    }

    if (formatContent) {
      console.log('✅ Successfully loaded format content:', {
        name: formatContent.name,
        contentLength: (formatContent.content || formatContent.format)?.length || 0,
        preview: (formatContent.content || formatContent.format)?.substring(0, 100) || 'EMPTY'
      });
    } else {
      console.log('❌ Failed to extract format content from DOM elements');
    }

    return formatContent;
  } catch (error) {
    console.error('Error triggering format preview load:', error);
    return null;
  }
}

// Global flag to prevent multiple simultaneous report generations
var isGeneratingReport = false;

/**
 * Generate report using existing TextRad ReportGenerator
 */
async function generateTextRadReport(transcription, formatId, retryCount = 0) {
  const maxRetries = 3;
  const retryDelay = 2000; // 2 seconds between retries

  console.log(`Generating report with TextRad ReportGenerator... (attempt ${retryCount + 1}/${maxRetries + 1})`, {
    formatId,
    transcriptionLength: transcription.length,
    retryCount
  });

  // Prevent multiple simultaneous generations
  if (isGeneratingReport && retryCount === 0) {
    console.log('Report generation already in progress, waiting...');
    await new Promise(resolve => {
      let waitTime = 0;
      const maxWaitTime = 60000; // 60 seconds max wait
      const checkInterval = setInterval(() => {
        waitTime += 1000;
        if (!isGeneratingReport || waitTime >= maxWaitTime) {
          if (waitTime >= maxWaitTime) {
            console.warn('⚠️ Report generation was stuck, forcing reset after 60 seconds');
            isGeneratingReport = false; // Force reset if stuck too long
          }
          clearInterval(checkInterval);
          resolve();
        }
      }, 1000);
    });
  }

  isGeneratingReport = true;

  // CRITICAL: Ensure services are ready before attempting report generation
  console.log(`🔍 Checking service readiness before report generation (attempt ${retryCount + 1})...`);

  try {
    // First, request tab activation to wake up suspended tabs
    chrome.runtime.sendMessage({ action: 'ACTIVATE_TEXTRAD_TABS' });

    // Wait longer for tab activation on retries
    const activationDelay = retryCount === 0 ? 1000 : 3000 + (retryCount * 1000);
    await new Promise(resolve => setTimeout(resolve, activationDelay));

    // Check if services are ready
    const servicesReady = await waitForServicesReady(15000); // 15 second timeout

    if (!servicesReady) {
      console.error(`❌ Services not ready for report generation (attempt ${retryCount + 1})`);

      // If this is not the last retry, try again
      if (retryCount < maxRetries) {
        console.log(`🔄 Retrying report generation in ${retryDelay}ms... (${maxRetries - retryCount} attempts remaining)`);
        isGeneratingReport = false;
        await new Promise(resolve => setTimeout(resolve, retryDelay));
        return generateTextRadReport(transcription, formatId, retryCount + 1);
      }

      throw new Error('Cannot generate report - TextRad services not available after multiple attempts. Please refresh the TextRad tab and try again.');
    }

    console.log(`✅ Services ready, proceeding with report generation (attempt ${retryCount + 1})`);
  } catch (serviceError) {
    isGeneratingReport = false;
    console.error(`Service readiness check failed (attempt ${retryCount + 1}):`, serviceError);

    // If this is not the last retry and the error might be temporary, try again
    if (retryCount < maxRetries && (
      serviceError.message.includes('not available') ||
      serviceError.message.includes('timeout') ||
      serviceError.message.includes('suspended'))) {
      console.log(`🔄 Retrying after service error in ${retryDelay}ms... (${maxRetries - retryCount} attempts remaining)`);
      await new Promise(resolve => setTimeout(resolve, retryDelay));
      return generateTextRadReport(transcription, formatId, retryCount + 1);
    }

    throw new Error(`Cannot generate report - ${serviceError.message}`);
  }

  try {
    // Try multiple ways to access TextRad app
    const textRadApp = window.textRadApp || self.textRadApp || top?.textRadApp;

    // Method 1: Direct Cloud Function API call (PRIMARY METHOD)
    console.log('🚀 Attempting direct report generation API call...');

    try {
      // Import the direct transcription service for report generation
      const { DirectTranscriptionService } = await import(chrome.runtime.getURL('lib/direct-transcription-service.js'));
      const directService = new DirectTranscriptionService();

      // Get the format details first for the function call
      const format = await getFormatById(formatId);
      if (!format) {
        throw new Error(`Format not found: ${formatId}`);
      }

      console.log('✅ Format retrieved:', format.name);

      // Debug format data
      const formatTemplate = format.content || format.format;
      console.log('🔍 Format debug:', {
        name: format.name,
        hasContent: !!format.content,
        hasFormat: !!format.format,
        templateLength: formatTemplate?.length || 0,
        templatePreview: formatTemplate?.substring(0, 100) || 'EMPTY'
      });

      if (!formatTemplate || formatTemplate.trim().length === 0) {
        throw new Error(`Format template is empty for format: ${format.name}`);
      }

      console.log('🔥 Calling direct report generation API...');

      // Get the actual user ID from page context (content scripts can't access window.textRadApp directly)
      let userId = 'extension-user';
      try {
        // Use page injector to get userId from page context
        const userIdResult = await new Promise((resolve) => {
          const messageHandler = (event) => {
            if (event.data && event.data.type === 'TEXTRAD_EXTENSION_USER_ID') {
              window.removeEventListener('message', messageHandler);
              resolve(event.data.userId);
            }
          };
          window.addEventListener('message', messageHandler);

          // Request userId from page context
          window.postMessage({ type: 'TEXTRAD_EXTENSION_GET_USER_ID' }, '*');

          // Timeout after 1 second
          setTimeout(() => {
            window.removeEventListener('message', messageHandler);
            resolve(null);
          }, 1000);
        });

        if (userIdResult) {
          userId = userIdResult;
        }
      } catch (error) {
        console.warn('Could not get userId from page context:', error);
      }
      console.log('📝 Using userId for report generation:', userId);

      // Call the direct report generation API
      const reportData = await directService.generateReport(
        transcription,
        format.name,
        formatTemplate,
        userId
      );

      console.log('✅ Direct report generation successful!');
      console.log('📊 Report data:', {
        reportLength: reportData.report?.length || 0,
        tokenUsage: reportData.tokenUsage || 0,
        provider: reportData.provider,
        reportId: reportData.reportId
      });

      isGeneratingReport = false; // Reset flag on success

      // SUCCESS - Return immediately, don't continue to fallback methods
      return reportData.report;

    } catch (directApiError) {
      console.error('❌ Direct API call failed:', directApiError);
      console.log('🔄 Falling back to DOM simulation method...');

      // Continue to DOM simulation fallback below
    }


    // Method 1.5: Hook into existing web app report generation
    console.log('Trying to hook into web app report generation...');
    console.log('Available TextRad objects at hook time:', {
      textRadApp: !!window.textRadApp,
      textradApp: !!window.textradApp,
      reportGenerator: !!(window.textRadApp && window.textRadApp.reportGenerator),
      generateReportFunc: !!(window.textRadApp && window.textRadApp.reportGenerator && window.textRadApp.reportGenerator.generateReport),
      globalGenerateReport: !!window.generateReport
    });

    try {
      // Set up a listener for the report generation completion
      let reportGenerationPromise = new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error('Report generation timeout'));
        }, 30000); // 30 second timeout

        // Hook into multiple possible report generation methods
        let hookApplied = false;

        // Try to hook into textRadApp.reportGenerator
        if (window.textRadApp && window.textRadApp.reportGenerator && window.textRadApp.reportGenerator.generateReport) {
          console.log('Hooking into textRadApp.reportGenerator.generateReport');
          const originalGenerate = window.textRadApp.reportGenerator.generateReport;
          window.textRadApp.reportGenerator.generateReport = async function (...args) {
            try {
              console.log('Report generation hooked - executing original function with args:', args);
              const result = await originalGenerate.apply(this, args);
              clearTimeout(timeout);
              console.log('Intercepted report generation result:', result);

              // Restore original function
              window.textRadApp.reportGenerator.generateReport = originalGenerate;

              if (result && (result.formattedReport || result.report)) {
                resolve(result.formattedReport || result.report);
              } else {
                resolve(result); // Return whatever we got
              }
              return result;
            } catch (error) {
              clearTimeout(timeout);
              // Restore original function
              window.textRadApp.reportGenerator.generateReport = originalGenerate;
              reject(error);
              throw error;
            }
          };
          hookApplied = true;
        }

        // Also try to hook into the global generateReport function if it exists
        if (window.generateReport && typeof window.generateReport === 'function') {
          console.log('Hooking into global generateReport function');
          const originalGlobalGenerate = window.generateReport;
          window.generateReport = async function (...args) {
            try {
              console.log('Global generateReport hooked - executing with args:', args);
              const result = await originalGlobalGenerate.apply(this, args);
              clearTimeout(timeout);
              console.log('Intercepted global generateReport result:', result);

              // Restore original function
              window.generateReport = originalGlobalGenerate;

              resolve(result);
              return result;
            } catch (error) {
              clearTimeout(timeout);
              window.generateReport = originalGlobalGenerate;
              reject(error);
              throw error;
            }
          };
          hookApplied = true;
        }

        if (!hookApplied) {
          console.log('No report generation functions found to hook into');
          reject(new Error('No report generation functions available for hooking'));
        }
      });

      // Trigger the web app's generate button
      const transcriptionTextarea = document.querySelector('#transcription-text');
      if (transcriptionTextarea) {
        transcriptionTextarea.value = transcription;
        transcriptionTextarea.dispatchEvent(new Event('input', { bubbles: true }));
      }

      const formatSelector = document.querySelector('#format-selector');
      if (formatSelector) {
        formatSelector.value = formatId;
        formatSelector.dispatchEvent(new Event('change', { bubbles: true }));
      }

      await new Promise(resolve => setTimeout(resolve, 500));

      const generateButton = document.querySelector('#generate-button');
      if (generateButton && !generateButton.disabled) {
        console.log('Clicking web app generate button to trigger hooked generation...');
        generateButton.click();

        // Wait for the hooked function to capture the result
        try {
          const result = await reportGenerationPromise;
          console.log('Successfully captured report from hooked generation:', result.substring(0, 200));
          return result;
        } catch (hookError) {
          console.log('Hook method failed:', hookError.message);
          // Continue to next method
        }
      }
    } catch (error) {
      console.error('Error with hook method:', error);
    }

    // Method 2: Simple textarea monitoring approach
    console.log('Setting up simple textarea monitoring for report detection...');
    try {
      const textareaResult = await new Promise(async (resolve, reject) => {
        const timeout = setTimeout(() => {
          reject(new Error('Report generation timed out - no content changes detected in textarea'));
        }, 45000); // 45 second timeout - increased for better reliability

        // Simple polling approach - check textarea content every second
        let checkCount = 0;
        const maxChecks = 45; // 45 seconds total

        const checkForReport = async () => {
          checkCount++;
          console.log(`Comprehensive report check ${checkCount}/${maxChecks}`);

          // Method 1: Check the main transcription textarea
          const transcriptionTextarea = document.querySelector('#transcription-text');
          if (transcriptionTextarea) {
            const currentContent = transcriptionTextarea.value.trim();
            console.log(`Transcription textarea: ${currentContent.length} chars, "${currentContent.substring(0, 50)}..."`);

            if (currentContent.length > transcription.length * 1.2 && currentContent !== transcription.trim()) {
              const medicalPatterns = [
                /TECHNIQUE:\s*[\w\s]+/i,
                /FINDINGS:\s*[\w\s]+/i,
                /IMPRESSION:\s*[\w\s]+/i,
                /CLINICAL HISTORY:\s*[\w\s]+/i
              ];

              const hasMedicalContent = medicalPatterns.some(pattern => pattern.test(currentContent));
              if (hasMedicalContent) {
                console.log('Found medical report in transcription textarea:', currentContent.substring(0, 200));
                clearTimeout(timeout);
                const cleanedContent = cleanReportContent(currentContent);
                resolve(cleanedContent);
                return;
              }
            }
          }

          // Method 2: Check all textareas on the page
          const allTextareas = document.querySelectorAll('textarea');
          console.log(`Found ${allTextareas.length} textareas on page`);

          for (let i = 0; i < allTextareas.length; i++) {
            const textarea = allTextareas[i];
            const content = textarea.value.trim();

            if (content.length > transcription.length * 1.2) {
              console.log(`Textarea ${i}: ${content.length} chars, "${content.substring(0, 50)}..."`);

              const medicalPatterns = [
                /TECHNIQUE:\s*[\w\s]+/i,
                /FINDINGS:\s*[\w\s]+/i,
                /IMPRESSION:\s*[\w\s]+/i,
                /CLINICAL HISTORY:\s*[\w\s]+/i
              ];

              const hasMedicalContent = medicalPatterns.some(pattern => pattern.test(content));
              if (hasMedicalContent && content !== transcription.trim()) {
                console.log(`Found medical report in textarea ${i}:`, content.substring(0, 200));
                clearTimeout(timeout);
                const cleanedContent = cleanReportContent(content);
                resolve(cleanedContent);
                return;
              }
            }
          }

          // Method 3: Check for specific report-results-content div with stale content protection
          console.log('Checking for report-results-content div...');

          // Target the exact div that contains the medical report
          const reportContentDiv = document.querySelector('#report-results-content');

          if (reportContentDiv) {
            console.log('✅ SUCCESS: Found report-results-content div!');
            console.log('Div element:', reportContentDiv);
            console.log('Div class list:', reportContentDiv.className);

            // CRITICAL: Capture initial state to detect changes and prevent stale content
            const initialContent = reportContentDiv.getAttribute('data-original-content');
            console.log('Initial data-original-content captured:', {
              exists: !!initialContent,
              length: initialContent?.length || 0,
              preview: initialContent ? initialContent.substring(0, 100) + '...' : 'none'
            });

            // Wait for the content to actually change (new report generation)
            let contentChangeAttempts = 0;
            const maxContentChangeAttempts = 10; // Wait up to 10 seconds for content to change

            while (contentChangeAttempts < maxContentChangeAttempts) {
              await new Promise(resolve => setTimeout(resolve, 1000));
              contentChangeAttempts++;

              const currentContent = reportContentDiv.getAttribute('data-original-content');
              console.log(`Content change attempt ${contentChangeAttempts}/${maxContentChangeAttempts}:`, {
                contentChanged: currentContent !== initialContent,
                currentLength: currentContent?.length || 0,
                initialLength: initialContent?.length || 0
              });

              // Check if content has actually changed (new report generated)
              if (currentContent && currentContent !== initialContent) {
                console.log('✅ CONTENT CHANGED: New report detected, proceeding with freshness check');
                break;
              } else if (currentContent && currentContent === initialContent) {
                console.log(`⏳ Waiting for content change... attempt ${contentChangeAttempts}/${maxContentChangeAttempts}`);
                continue;
              } else if (!currentContent && contentChangeAttempts > 5) {
                console.log('❌ No content detected after 5 attempts, may need to wait longer');
                continue;
              }
            }

            // Try to get the clean content from data-original-content attribute first
            const originalContent = reportContentDiv.getAttribute('data-original-content');

            console.log('Final content verification:', {
              exists: !!originalContent,
              length: originalContent?.length || 0,
              transcriptionLength: transcription.length,
              isNew: originalContent !== initialContent,
              preview: originalContent ? originalContent.substring(0, 100) + '...' : 'none'
            });

            // Enhanced validation: ensure we have fresh content
            if (!originalContent) {
              console.log('❌ No content in data-original-content attribute after waiting');
              return null;
            }

            if (originalContent === initialContent && contentChangeAttempts >= maxContentChangeAttempts) {
              console.log('⚠️ WARNING: Content did not change after waiting, but proceeding with enhanced freshness check');
            }

            if (originalContent && originalContent.trim().length > transcription.length * 0.8) {
              console.log('Found clean report in data-original-content attribute');
              console.log('Original content length:', originalContent.length);
              console.log('Original content preview:', originalContent.substring(0, 300));

              // CRITICAL: Check if this content is actually NEW by comparing with transcription
              // The new report should contain elements from the current transcription
              const transcriptionWords = transcription.toLowerCase().split(/\s+/).filter(word => word.length > 3);
              const reportWords = originalContent.toLowerCase();

              let matchCount = 0;
              transcriptionWords.forEach(word => {
                if (reportWords.includes(word)) {
                  matchCount++;
                }
              });

              const matchPercentage = (matchCount / transcriptionWords.length) * 100;
              console.log('Enhanced content freshness check:', {
                transcriptionWords: transcriptionWords.length,
                matchCount,
                matchPercentage: matchPercentage.toFixed(1) + '%',
                contentChanged: originalContent !== initialContent,
                isNewGeneration: contentChangeAttempts > 0
              });

              // Enhanced freshness validation with stricter criteria
              const isContentChanged = originalContent !== initialContent;
              const hasGoodWordMatch = matchPercentage > 20; // At least 20% of words should match

              if (hasGoodWordMatch && (isContentChanged || contentChangeAttempts < maxContentChangeAttempts)) {
                console.log('✅ ENHANCED FRESHNESS VERIFIED - this appears to be a NEW report for current transcription');
                console.log('Validation details:', {
                  hasGoodWordMatch,
                  isContentChanged,
                  contentChangeAttempts,
                  decision: 'PROCEED'
                });

                // Clean up the markdown formatting and "End of Report"
                let cleanReport = originalContent
                  .replace(/\*\*/g, '') // Remove markdown bold markers
                  .replace(/End of Report\s*$/gi, '') // Remove "End of Report" from end
                  .trim();

                console.log('🎉 SUCCESS: Returning fresh medical report from data-original-content!');
                console.log('Final clean report preview:', cleanReport.substring(0, 200) + '...');

                // CRITICAL: Click Save Report button with enhanced detection for subsequent reports
                console.log('Attempting to save report to reset TextRad state...');

                // Wait a moment for UI to settle, then find and click save button
                setTimeout(() => {
                  // Enhanced save button detection with multiple strategies
                  let saveReportButton = document.querySelector('#save-inline-report');

                  if (!saveReportButton) {
                    // Try broader search for save buttons
                    const allSaveButtons = document.querySelectorAll('button');
                    for (const btn of allSaveButtons) {
                      if (btn.textContent.trim().toLowerCase() === 'save report' ||
                        btn.id.includes('save') && btn.textContent.toLowerCase().includes('report')) {
                        saveReportButton = btn;
                        console.log('Found save button via text search:', btn.id || btn.className);
                        break;
                      }
                    }
                  }

                  if (saveReportButton && !saveReportButton.disabled) {
                    console.log('Clicking Save Report button to reset state...', {
                      id: saveReportButton.id,
                      text: saveReportButton.textContent.trim(),
                      disabled: saveReportButton.disabled
                    });

                    saveReportButton.click();
                    console.log('✅ Save Report button clicked successfully');

                    // Verify the save worked by checking if button is still there
                    setTimeout(() => {
                      const buttonStillExists = document.querySelector('#save-inline-report');
                      console.log('Save verification:', {
                        buttonStillExists: !!buttonStillExists,
                        status: buttonStillExists ? 'Save may not have completed' : 'Save appears successful'
                      });
                    }, 1000);
                  } else {
                    console.warn('⚠️ Could not find or click Save Report button');
                    console.log('Button search results:', {
                      foundButton: !!saveReportButton,
                      disabled: saveReportButton?.disabled,
                      text: saveReportButton?.textContent?.trim()
                    });
                  }
                }, 300); // Small delay to ensure UI is ready

                console.log('STOPPING COMPREHENSIVE MONITORING - FRESH REPORT FOUND!');
                clearTimeout(timeout);
                resolve(cleanReport);
                return; // CRITICAL: Stop here, don't continue to other methods
              } else {
                console.log('❌ ENHANCED STALE CONTENT DETECTION - content rejected');
                console.log('Stale content analysis:', {
                  hasGoodWordMatch,
                  isContentChanged,
                  contentChangeAttempts,
                  initialContentLength: initialContent?.length || 0,
                  currentContentLength: originalContent?.length || 0,
                  reason: !hasGoodWordMatch ? 'Poor word match' : 'Content unchanged',
                  decision: 'SKIP_AND_CONTINUE'
                });
                console.log('Skipping this stale content and continuing to wait for fresh report...');
                // Don't resolve, continue monitoring for fresh content
              }
            }

            // Fallback: extract from the div's text content if data-original-content is not available
            const divTextContent = reportContentDiv.textContent || reportContentDiv.innerText || '';

            if (divTextContent && divTextContent.trim().length > transcription.length * 1.2) {
              console.log('Using div text content as fallback');
              console.log('Div text content length:', divTextContent.length);

              // Clean the text content
              let cleanReport = divTextContent
                .replace(/End of Report\s*$/gi, '') // Remove "End of Report" from end
                .trim();

              // Verify this looks like a medical report
              const medicalPatterns = [
                /TECHNIQUE:|CLINICAL HISTORY:|FINDINGS:|IMPRESSION:/i,
                /normal|abnormal|study/i,
                /brain|skull|cerebellum|ventricles|attenuation/i
              ];

              const hasMedicalContent = medicalPatterns.some(pattern => pattern.test(cleanReport));

              if (hasMedicalContent) {
                console.log('🎉 SUCCESS: Returning medical report from div text content!');
                console.log('Final report preview:', cleanReport.substring(0, 200) + '...');

                // Click Save Report button to reset state
                const saveReportButton = document.querySelector('#save-inline-report');
                if (saveReportButton) {
                  console.log('Clicking Save Report button to reset TextRad state...');
                  saveReportButton.click();
                }

                console.log('STOPPING COMPREHENSIVE MONITORING - REPORT FOUND!');
                clearTimeout(timeout);
                resolve(cleanReport);
                return; // CRITICAL: Stop here, don't continue to other methods
              } else {
                console.log('Div content does not appear to be a valid medical report');
              }
            }
          } else {
            console.log('report-results-content div not found');
          }

          // Method 4: DISABLED - was capturing "My Reports" content
          console.log('Method 4 (large content scan) disabled - was capturing wrong content');
          // This method was finding the "My Reports" page content instead of the actual generated report

          // Continue checking if we haven't found the report yet
          if (checkCount < maxChecks) {
            setTimeout(() => checkForReport(), 1000);
          } else {
            console.log('Comprehensive report check completed without finding medical content');
          }
        };

        // CRITICAL: Set the transcription in the web app with enhanced verification
        console.log('Setting transcription in TextRad web app...', { transcriptionLength: transcription.length });
        const transcriptionTextarea = document.querySelector('#transcription-text');

        if (!transcriptionTextarea) {
          console.error('CRITICAL ERROR: Transcription textarea not found! Cannot set transcription.');
          reject(new Error('Transcription textarea not found in TextRad web app'));
          return;
        }

        // Clear the field first
        transcriptionTextarea.value = '';
        transcriptionTextarea.focus();

        // Set the transcription with multiple event triggers
        transcriptionTextarea.value = transcription;

        // Trigger multiple events to ensure TextRad app recognizes the change
        transcriptionTextarea.dispatchEvent(new Event('input', { bubbles: true }));
        transcriptionTextarea.dispatchEvent(new Event('change', { bubbles: true }));
        transcriptionTextarea.dispatchEvent(new Event('keyup', { bubbles: true }));
        transcriptionTextarea.dispatchEvent(new Event('paste', { bubbles: true }));

        // Verify the transcription was set correctly
        await new Promise(resolve => setTimeout(resolve, 200));
        const verifyContent = transcriptionTextarea.value.trim();
        console.log('Transcription verification:', {
          expected: transcription.length,
          actual: verifyContent.length,
          matches: verifyContent === transcription.trim(),
          preview: verifyContent.substring(0, 50) + '...'
        });

        if (verifyContent !== transcription.trim()) {
          console.warn('WARNING: Transcription verification failed! Retrying...');

          // Retry with different method
          transcriptionTextarea.focus();
          transcriptionTextarea.select();
          document.execCommand('insertText', false, transcription);

          // Verify again
          await new Promise(resolve => setTimeout(resolve, 200));
          const retryVerify = transcriptionTextarea.value.trim();
          if (retryVerify !== transcription.trim()) {
            console.error('CRITICAL: Failed to set transcription even after retry!');
            console.log('Attempting fallback via direct Firebase call...');
            // Try to bypass the UI completely and call Firebase directly
            try {
              const directResult = await callFirebaseFunction('generateReport', {
                transcription: transcription,
                formatId: formatId || 'general-medicine'
              });
              if (directResult && directResult.data) {
                console.log('✅ Fallback Firebase call succeeded');
                resolve(directResult.data.formattedReport || directResult.data.report || directResult.data);
                return;
              }
            } catch (fbError) {
              console.error('Fallback Firebase call also failed:', fbError);
            }
            reject(new Error('Failed to set transcription in TextRad web app and fallback failed'));
            return;
          }
          console.log('Transcription set successfully on retry');
        } else {
          console.log('Transcription set successfully on first attempt');
        }

        // Set the format selection
        console.log('Setting format selection...', { formatId });
        const formatSelector = document.querySelector('#format-selector');
        if (formatSelector) {
          formatSelector.value = formatId;
          formatSelector.dispatchEvent(new Event('change', { bubbles: true }));

          // Verify format selection
          await new Promise(resolve => setTimeout(resolve, 200));
          const selectedFormat = formatSelector.value;
          console.log('Format selection verification:', {
            expected: formatId,
            actual: selectedFormat,
            matches: selectedFormat === formatId
          });
        } else {
          console.error('Format selector not found!');
          reject(new Error('Format selector not found in TextRad web app'));
          return;
        }

        // Wait longer for the UI to update
        console.log('Waiting for TextRad UI to update...');
        await new Promise(resolve => setTimeout(resolve, 1000));

        // Final verification before clicking generate button
        const finalTextareaCheck = document.querySelector('#transcription-text');
        const finalFormatCheck = document.querySelector('#format-selector');

        console.log('Final verification before generate:', {
          textareaContent: finalTextareaCheck?.value?.length || 0,
          selectedFormat: finalFormatCheck?.value || 'none',
          transcriptionMatches: finalTextareaCheck?.value?.trim() === transcription.trim()
        });

        if (!finalTextareaCheck?.value?.trim()) {
          console.error('CRITICAL: No transcription content in textarea before generate!');
          reject(new Error('No transcription content found before report generation'));
          return;
        }

        // Click the generate button
        const generateButton = document.querySelector('#generate-button');
        if (generateButton && !generateButton.disabled) {
          console.log('All verifications passed. Clicking generate button...');
          console.log('Generate button state:', {
            exists: !!generateButton,
            disabled: generateButton.disabled,
            visible: generateButton.offsetParent !== null,
            text: generateButton.textContent?.trim()
          });

          generateButton.click();

          // Start checking for the report
          console.log('Report generation triggered, starting comprehensive monitoring in 1 second...');

          // Do an immediate check to see current state
          setTimeout(() => {
            console.log('=== IMMEDIATE STATE CHECK ===');
            const immediateTextarea = document.querySelector('#transcription-text');
            if (immediateTextarea) {
              console.log(`Current textarea content: "${immediateTextarea.value.substring(0, 100)}..."`);
              console.log(`Current textarea length: ${immediateTextarea.value.length}`);
            }

            const allTextareas = document.querySelectorAll('textarea');
            console.log(`Total textareas found: ${allTextareas.length}`);

            allTextareas.forEach((ta, i) => {
              if (ta.value.length > 50) {
                console.log(`Textarea ${i}: ${ta.value.length} chars, "${ta.value.substring(0, 50)}..."`);
              }
            });
            console.log('=== END IMMEDIATE CHECK ===');
          }, 100);

          setTimeout(() => checkForReport(), 1000); // Start checking after 1 second
        } else {
          clearTimeout(timeout);
          reject(new Error('Generate button not found or disabled'));
        }
      });

      // If we successfully got a result from the comprehensive monitoring, return it
      console.log('Comprehensive monitoring method succeeded, returning result');
      isGeneratingReport = false; // Reset flag on success
      return textareaResult;

    } catch (error) {
      console.error('Error with comprehensive monitoring method:', error);
      // Continue to next method if this fails
    }

    // Method 3: Disabled - problematic polling method that captures wrong content
    console.log('Skipping problematic polling method that captures My Reports page');
    // DISABLED: This method was capturing "My Reports" page content instead of generated reports

    /* DISABLED METHOD 3 - WAS CAPTURING WRONG CONTENT
    try {
      // Set the transcription in the web app
      const transcriptionTextarea = document.querySelector('#transcription-text');
      if (transcriptionTextarea) {
        transcriptionTextarea.value = transcription;
        transcriptionTextarea.dispatchEvent(new Event('input', { bubbles: true }));
      }
  
      // Set the format selection
      const formatSelector = document.querySelector('#format-selector');
      if (formatSelector) {
        formatSelector.value = formatId;
        formatSelector.dispatchEvent(new Event('change', { bubbles: true }));
      }
  
      // Wait a moment for the UI to update
      await new Promise(resolve => setTimeout(resolve, 500));
  
      // Click the generate button
      const generateButton = document.querySelector('#generate-button');
      if (generateButton && !generateButton.disabled) {
        console.log('Clicking web app generate button for polling...');
        generateButton.click();
  
        // Wait for report generation (check multiple possible locations)
        let attempts = 0;
        while (attempts < 60) { // Wait up to 60 seconds
          await new Promise(resolve => setTimeout(resolve, 1000));
  
          console.log(`Report detection attempt ${attempts + 1}/60`);
  
          // Method 1: Check if transcription textarea content changed to formatted report
          const transcriptionTextarea = document.querySelector('#transcription-text');
          if (transcriptionTextarea) {
            const currentContent = transcriptionTextarea.value.trim();
            console.log(`Transcription content length: ${currentContent.length}, original: ${transcription.length}`);
            console.log(`Transcription content preview: "${currentContent.substring(0, 100)}..."`);
  
            if (currentContent !== transcription.trim() && currentContent.length > transcription.length) {
              console.log('Report detected in transcription textarea, content changed');
              const cleanedReport = cleanReportContent(currentContent);
              console.log('Final cleaned report length:', cleanedReport?.length);
              isGeneratingReport = false; // Reset flag on success
              return cleanedReport;
            }
          }
  
          // Method 2: Check for report section with various selectors
          const reportSelectors = [
            '#formatted-report',
            '#report-content',
            '#report-text',
            '.formatted-report',
            '.report-content',
            '.report-display',
            '.generated-content',
            '.report-output',
            '[data-report-content]',
            '.format-content',
            '.content-display'
          ];
  
          for (const selector of reportSelectors) {
            const reportElement = document.querySelector(selector);
            if (reportElement) {
              const content = reportElement.textContent || reportElement.innerText || '';
              console.log(`Checking selector ${selector}: found=${!!reportElement}, length=${content.length}`);
              console.log(`Content preview from ${selector}: "${content.substring(0, 200)}..."`);
  
              if (content.trim().length > transcription.length &&
                  !content.includes('Generated Report') &&
                  !content.includes('Your completed report')) {
                console.log(`Valid report detected in element: ${selector}`);
                return cleanReportContent(content.trim());
              }
            }
          }
  
          // Method 3: Check if generate button text changed (indicating completion)
          const generateButton = document.querySelector('#generate-button');
          if (generateButton) {
            const buttonText = generateButton.textContent.trim();
            if (buttonText.includes('Generated') || buttonText.includes('Complete') || !buttonText.includes('Generating')) {
              // Button indicates completion, try to find the report content
              const allTextAreas = document.querySelectorAll('textarea');
              for (const textarea of allTextAreas) {
                if (textarea.value.length > transcription.length * 1.5) {
                  console.log('Report detected via button state change');
                  return cleanReportContent(textarea.value.trim());
                }
              }
            }
          }
  
          // Method 4: Check for clean text content (exclude UI elements)
          const contentElements = document.querySelectorAll('textarea');
          for (const element of contentElements) {
            const content = element.value || '';
            if (content.length > transcription.length * 1.5 && content.trim() !== transcription.trim()) {
              console.log('Report detected via textarea content analysis');
              return cleanReportContent(content.trim());
            }
          }
  
          // Method 5: Look for TextRad-specific content after report generation
          try {
            // Check if the report was populated into the transcription field
            const transcriptionTextarea = document.querySelector('#transcription-text');
            if (transcriptionTextarea) {
              const textareaContent = transcriptionTextarea.value.trim();
              console.log(`Textarea content check: length=${textareaContent.length}, preview="${textareaContent.substring(0, 100)}..."`);
  
              // Look for medical report content indicators
              const medicalIndicators = [
                'TECHNIQUE:', 'FINDINGS:', 'IMPRESSION:', 'HISTORY:', 'INDICATION:',
                'CLINICAL HISTORY:', 'CONTRAST:', 'CONCLUSION:', 'RECOMMENDATION:',
                'CT ', 'MRI ', 'scan', 'normal', 'abnormal', 'study'
              ];
  
              const hasMedicalContent = medicalIndicators.some(indicator =>
                textareaContent.toLowerCase().includes(indicator.toLowerCase())
              );
  
              if (hasMedicalContent && textareaContent.length > transcription.length * 2) {
                console.log('Medical report content detected in textarea');
                return cleanReportContent(textareaContent);
              }
            }
          } catch (error) {
            console.log('Error checking for medical content:', error);
          }
  
          // Method 6: Look for specific report containers (avoid reports list)
          const specificReportSelectors = [
            '.generated-report-content',
            '.report-display-content',
            '.current-report',
            '.active-report',
            '[data-generated-report]',
            '.formatted-output'
          ];
  
          for (const selector of specificReportSelectors) {
            const container = document.querySelector(selector);
            if (container) {
              const content = container.textContent || container.innerText || '';
              console.log(`Checking specific report selector ${selector}: length=${content.length}`);
  
              if (content.length > transcription.length &&
                  !content.includes('My Reports') &&
                  !content.includes('New Report') &&
                  !content.includes('Generated Report Your completed report')) {
  
                // Look for medical report patterns
                const medicalPatterns = [
                  /TECHNIQUE:\s*\w+/i,
                  /FINDINGS:\s*\w+/i,
                  /IMPRESSION:\s*\w+/i,
                  /CLINICAL HISTORY:/i
                ];
  
                const hasMedicalPattern = medicalPatterns.some(pattern => pattern.test(content));
                if (hasMedicalPattern) {
                  console.log(`Valid medical report found in: ${selector}`);
                  return cleanReportContent(content.trim());
                }
              }
            }
          }
  
          console.log(`Waiting for report generation... attempt ${attempts + 1}/60`);
          attempts++;
        }
  
        // If we get here, try one last check for any changed content
        const transcriptionTextarea = document.querySelector('#transcription-text');
        if (transcriptionTextarea && transcriptionTextarea.value.trim() !== transcription.trim()) {
          const content = transcriptionTextarea.value.trim();
          console.log('Using final transcription content as report');
          return cleanReportContent(content);
        }
  
        throw new Error('Report generation timed out - could not detect completion');
      } else {
        throw new Error('Generate button not found or disabled');
      }
    } catch (error) {
      console.error('Error triggering web app generate:', error);
      // Continue to next method
    }
    END OF DISABLED METHOD 3 */

    // Method 4: Direct Firebase function call as fallback
    if (window.textRadApp && window.textRadApp.functions) {
      console.log('Falling back to direct Firebase function call');

      try {
        // Get the format
        const formats = await getTextRadFormats();
        const format = formats.find(f => f.id === formatId);
        if (!format) {
          throw new Error(`Format not found: ${formatId}`);
        }

        // Call Firebase function directly (same as ReportGenerator does)
        const generateReport = window.textRadApp.functions.httpsCallable('generateReport');
        const result = await generateReport({
          transcription: transcription,
          formatId: formatId,
          formatContent: format.content,
          formatName: format.name
        });

        console.log('Report generated via direct Firebase call');
        isGeneratingReport = false; // Reset flag on success
        return result.data.formattedReport || result.data.report || result.data;

      } catch (error) {
        console.error('Error with direct Firebase call:', error);
        throw error;
      }
    }

    throw new Error('Cannot generate report - TextRad services not available');

  } catch (error) {
    console.error(`Report generation failed (attempt ${retryCount + 1}):`, error);

    // If this is not the last retry and the error might be temporary, try again
    if (retryCount < maxRetries && (
      error.message.includes('not available') ||
      error.message.includes('timeout') ||
      error.message.includes('suspended') ||
      error.message.includes('timed out') ||
      error.message.includes('services not ready'))) {

      console.log(`🔄 Retrying report generation due to error in ${retryDelay}ms... (${maxRetries - retryCount} attempts remaining)`);
      isGeneratingReport = false;
      await new Promise(resolve => setTimeout(resolve, retryDelay));
      return generateTextRadReport(transcription, formatId, retryCount + 1);
    }

    // Reset flag and throw error for non-retryable errors
    isGeneratingReport = false;
    throw error;
  } finally {
    // Ensure flag is always reset if we're not retrying
    console.log(`Report generation attempt ${retryCount + 1} completed`);
  }
}

/**
 * Inject generated report into web app's report results panel
 * @param {string} reportContent - The generated report content
 * @param {string} formatName - The format name used
 * @param {boolean} autoSave - Whether to auto-trigger save
 * @returns {Promise<Object>} - Success status
 */
async function injectReportIntoWebApp(reportContent, formatName = '', autoSave = false) {
  console.log('🔄 Injecting report into web app...', {
    reportLength: reportContent?.length,
    formatName,
    autoSave
  });

  try {
    // Use page injector to set currentReport and display the report
    // This ensures event listeners are properly attached
    const saveSuccess = await triggerWebAppSave(reportContent, formatName);

    if (!saveSuccess) {
      throw new Error('Failed to inject report via page injector');
    }

    console.log('✅ Report injected successfully via page injector');

    // Optional: Auto-save and reset web app state
    if (autoSave) {
      console.log('⏱️ Auto-save enabled, triggering save in background...');

      // Trigger save in background without blocking
      // This allows the popup UI to collapse immediately
      setTimeout(async () => {
        // Short wait for DOM to update (reduced from 1500ms to 500ms)
        await new Promise(resolve => setTimeout(resolve, 500));

        // Click the save button that was set up by displayInlineReport
        const saveButton = document.getElementById('save-inline-report');
        if (saveButton) {
          console.log('🖱️ Clicking save button...');
          saveButton.click();

          console.log('✅ Save triggered, waiting for completion...');
          // Wait for save to complete, then reset (in background)
          await new Promise(resolve => setTimeout(resolve, 1500));

          // Note: Usage counter is now incremented in the Cloud Function
          // No need to increment here - directGenerateReportHttp handles it

          // Reset web app to new report state
          resetWebAppState();
        } else {
          console.error('❌ Save button not found after report injection');
        }
      }, 0);
    }

    // Return immediately so UI can collapse without waiting
    return { success: true, message: 'Report injected successfully' };
  } catch (error) {
    console.error('❌ Error injecting report:', error);
    return { success: false, error: error.message };
  }
}

/**
 * Inject page context script that can access window.textRadApp
 * This bypasses CSP restrictions by using web_accessible_resources
 */
function injectPageContextScript() {
  // Use window property to avoid redeclaration errors on script reload
  if (window.textRadPageInjectorLoaded) {
    console.log('Page injector already loaded, skipping injection');
    return;
  }

  try {
    const script = document.createElement('script');
    script.src = chrome.runtime.getURL('lib/page-injector.js');
    script.onload = () => {
      console.log('✅ Page injector script loaded');
      window.textRadPageInjectorLoaded = true;
    };
    script.onerror = (error) => {
      console.error('❌ Failed to load page injector:', error);
    };
    (document.head || document.documentElement).appendChild(script);
  } catch (error) {
    console.error('❌ Error injecting page context script:', error);
  }
}

/**
 * Trigger web app's save report function
 * Sets currentReport in the web app instance and clicks save button
 * @param {string} reportContent - Report content to save
 * @param {string} formatName - Format/template name
 * @returns {Promise<boolean>} - True if save was successful
 */
async function triggerWebAppSave(reportContent, formatName) {
  try {
    console.log('💾 Triggering web app save function...', {
      contentLength: reportContent?.length,
      formatName
    });

    // Ensure page injector is loaded
    if (!window.textRadPageInjectorLoaded) {
      injectPageContextScript();
      // Wait a bit for script to load
      await new Promise(resolve => setTimeout(resolve, 300));
    }

    // Use postMessage to communicate with page context
    // Set up listener for response
    const responsePromise = new Promise((resolve) => {
      const messageHandler = (event) => {
        if (event.origin !== window.location.origin) {
          return;
        }

        if (event.data && event.data.type === 'TEXTRAD_EXTENSION_REPORT_SET') {
          console.log('✅ Received response from page injector:', event.data);
          window.removeEventListener('message', messageHandler);
          resolve(event.data.success);
        }
      };

      window.addEventListener('message', messageHandler);

      // Timeout after 2 seconds
      setTimeout(() => {
        window.removeEventListener('message', messageHandler);
        resolve(false);
      }, 2000);
    });

    // Send message to page context
    window.postMessage({
      type: 'TEXTRAD_EXTENSION_SET_REPORT',
      content: reportContent,
      formatName: formatName || 'Extension Report',
      source: 'extension'
    }, window.location.origin);

    // Wait for confirmation
    const success = await responsePromise;

    if (!success) {
      console.error('❌ Failed to set currentReport in page context');
      return false;
    }

    // Report has been set and displayed via page injector
    // The web app's displayInlineReport has set up all event listeners
    console.log('✅ Report prepared for saving');
    return true;

  } catch (error) {
    console.error('❌ Error triggering web app save:', error);
    return false;
  }
}

/**
 * Reset web app state after saving report
 * Hides report panel and shows format preview for next report
 */
function resetWebAppState() {
  try {
    console.log('🔄 Resetting web app state...');

    const reportPanel = document.getElementById('report-results-panel');
    const reportContentDiv = document.getElementById('report-results-content');
    const formatPreviewPanel = document.getElementById('format-preview-panel');

    // Clear report content
    if (reportContentDiv) {
      reportContentDiv.innerHTML = '';
      reportContentDiv.removeAttribute('data-original-content');
    }

    // Hide report panel
    if (reportPanel) {
      reportPanel.classList.add('hidden');
    }

    // Show format preview panel (ready for next report)
    if (formatPreviewPanel) {
      formatPreviewPanel.classList.remove('hidden');
    }

    console.log('✅ Web app state reset successfully');
  } catch (error) {
    console.error('❌ Error resetting web app state:', error);
  }
}

/**
 * Format report content as HTML for web app display
 * Converts markdown-style formatting to HTML
 * @param {string} content - Raw report content
 * @returns {string} - HTML formatted content
 */
function formatReportContentAsHTML(content) {
  if (!content) return '<p class="text-gray-400">No content</p>';

  // Wrap in a div with inherited styling
  let html = '<div style="color: inherit; background-color: transparent;">';

  // Split by double line breaks to create paragraphs
  const lines = content.split('\n');

  for (let i = 0; i < lines.length; i++) {
    let line = lines[i].trim();

    if (!line) {
      // Empty line - keep for spacing
      html += '<p class="" style=""></p>';
      continue;
    }

    // Check for bold headings (e.g., **TECHNIQUE:** or **FINDINGS**)
    if (line.startsWith('**') && line.includes('**')) {
      // Extract content between ** markers
      const boldMatch = line.match(/\*\*([^*]+)\*\*/g);
      if (boldMatch) {
        let processedLine = line;

        // Process all bold sections
        boldMatch.forEach(match => {
          const text = match.replace(/\*\*/g, '');
          // Determine if it's a main heading or inline bold
          const isHeading = line.trim().startsWith('**') && (line.endsWith('**') || line.indexOf(':') > 0);

          if (isHeading && text.length < 50) {
            // Main heading
            processedLine = processedLine.replace(match, `<strong class="text-xl" style="">${text}</strong>`);
          } else {
            // Regular bold text
            processedLine = processedLine.replace(match, `<strong class="" style="">${text}</strong>`);
          }
        });

        html += `<p class="" style="">${processedLine}</p>`;
        continue;
      }
    }

    // Check for bullet points (• or - at start)
    if (line.startsWith('•') || line.startsWith('-')) {
      // Start or continue list
      if (i === 0 || !lines[i - 1].trim().startsWith('•') && !lines[i - 1].trim().startsWith('-')) {
        html += '<ul style="list-style-type: disc; margin-left: 20px;">';
      }

      const bulletContent = line.substring(1).trim();
      // Process bold within bullets
      let processedContent = bulletContent.replace(/\*\*([^*]+)\*\*/g, '<strong class="" style="">$1</strong>');
      html += `<li class="" style=""> ${processedContent}</li>`;

      // Close list if next line is not a bullet
      if (i === lines.length - 1 || (!lines[i + 1].trim().startsWith('•') && !lines[i + 1].trim().startsWith('-'))) {
        html += '</ul>';
      }
      continue;
    }

    // Regular paragraph - process inline bold
    let processedLine = line.replace(/\*\*([^*]+)\*\*/g, '<strong class="" style="">$1</strong>');
    html += `<p class="" style="">${processedLine}</p>`;
  }

  html += '</div>';
  return html;
}

/**
 * Enhanced state getter that includes format information
 */
function getTextRadState() {
  const baseState = {
    hasTranscriptionElement: !!transcriptionElement,
    hasFormatSelector: !!formatSelectorElement,
    currentFormat: currentFormat,
    url: window.location.href,
    title: document.title
  };

  // Add information about available services
  if (window.textRadApp) {
    baseState.services = {
      formatManager: !!window.textRadApp.formatManager,
      reportGenerator: !!window.textRadApp.reportGenerator,
      isInitialized: window.textRadApp.isInitialized,
      isAuthenticated: window.textRadApp.isAuthenticated
    };
  }

  return baseState;
}

/**
 * Clean report content by removing HTML markup and UI elements
 */
function cleanReportContent(content) {
  if (!content || typeof content !== 'string') {
    console.log('cleanReportContent: Invalid input, returning as-is:', typeof content, content?.length);
    return content;
  }

  console.log('cleanReportContent: Input content length:', content.length);
  console.log('cleanReportContent: Input preview:', content.substring(0, 200));

  // Remove common UI text patterns that shouldn't be in medical reports
  let cleaned = content;

  // Remove UI button text and navigation elements
  const uiPatterns = [
    /Generated Report Your completed report Edit Save Report?/gi,
    /- Generated Report/gi,
    /Your completed report/gi,
    /Edit Save Report/gi,
    /Edit\s+Save\s+Report/gi,
    /Back to Reports/gi,
    /Generate Report/gi,
    /Copy to Clipboard/gi,
    /Download Report/gi,
    /Share Report/gi,
    /Print Report/gi
  ];

  let removedPatterns = [];
  uiPatterns.forEach(pattern => {
    const beforeLength = cleaned.length;
    cleaned = cleaned.replace(pattern, '');
    if (cleaned.length !== beforeLength) {
      removedPatterns.push(pattern.toString());
    }
  });

  if (removedPatterns.length > 0) {
    console.log('cleanReportContent: Removed UI patterns:', removedPatterns);
  }

  // Remove HTML tags if any
  const beforeHtmlRemoval = cleaned.length;
  cleaned = cleaned.replace(/<[^>]*>/g, '');
  if (cleaned.length !== beforeHtmlRemoval) {
    console.log('cleanReportContent: Removed HTML tags, length changed from', beforeHtmlRemoval, 'to', cleaned.length);
  }

  // Remove extra whitespace and normalize line breaks
  cleaned = cleaned.replace(/\s+/g, ' ').trim();
  cleaned = cleaned.replace(/\n\s*\n/g, '\n');

  console.log('cleanReportContent: After cleaning, length:', cleaned.length);
  console.log('cleanReportContent: After cleaning, word count:', cleaned.split(' ').length);
  console.log('cleanReportContent: Cleaned preview:', cleaned.substring(0, 200));

  // If the content appears to be mostly UI elements, return the original content instead of empty
  if (cleaned.length < 20 || cleaned.split(' ').length < 5) {
    console.log('cleanReportContent: Content appears to be mostly UI elements, returning original content');
    // Return original content without cleaning rather than empty
    return content.replace(/<[^>]*>/g, '').replace(/\s+/g, ' ').trim();
  }

  return cleaned;
}

/**
 * Warmup report generation services to prevent cold starts
 */
async function warmupReportServices() {
  try {
    console.log('🔥 Starting service warmup for report generation...');

    // Warmup the generateReport cloud function with a minimal request
    if (window.firebase && window.firebase.functions) {
      try {
        const generateReportFunction = window.firebase.functions().httpsCallable('generateReport');

        // Send a minimal warmup request (uses special warmup handling in Cloud Function)
        await Promise.race([
          generateReportFunction({
            userId: 'warmup-user',
            source: 'extension-warmup'
          }),
          new Promise((_, reject) => setTimeout(() => reject(new Error('warmup timeout')), 5000))
        ]);

        console.log('🔥 Report generation service warmed up successfully');
      } catch (error) {
        // Warmup failures are non-critical
        console.log('🔥 Report warmup completed (non-critical error):', error.message);
      }
    }

    // Small delay to let services stabilize
    await new Promise(resolve => setTimeout(resolve, 500));

  } catch (error) {
    console.log('🔥 Service warmup completed with errors (non-critical):', error);
  }
}

/**
 * Create Firebase Functions instance using v9+ modular SDK
 */
async function createFirebaseFunctionsInstance() {
  try {
    // Import Firebase Functions module
    const { getFunctions, httpsCallable } = await import("https://www.gstatic.com/firebasejs/11.2.0/firebase-functions.js");

    // Create functions instance with asia-south1 region (matching TextRad pattern)
    const functions = getFunctions(window.firebase, 'asia-south1');

    return {
      httpsCallable: (name) => httpsCallable(functions, name)
    };
  } catch (error) {
    console.error('Failed to create Firebase Functions instance:', error);
    return null;
  }
}

/**
 * Call report generation with retry logic for cold starts
 */
async function callReportGenerationWithRetry(functionsInstance, functionData, maxRetries = 3) {
  const baseDelay = 2000; // Start with 2 seconds

  for (let attempt = 0; attempt <= maxRetries; attempt++) {
    try {
      console.log(`📊 Report generation attempt ${attempt + 1}/${maxRetries + 1}`);

      // Create timeout for the call
      const timeout = attempt === 0 ? 45000 : 30000; // Longer timeout for first attempt

      // Handle different Firebase function calling patterns
      let generateReportFunction;
      if (typeof functionsInstance.httpsCallable === 'function') {
        // v9+ modular SDK or custom wrapper
        const callable = functionsInstance.httpsCallable('generateReport');

        // Handle both sync and async httpsCallable patterns
        if (callable && typeof callable.then === 'function') {
          // Async pattern (dynamic import)
          generateReportFunction = await callable;
        } else {
          // Sync pattern
          generateReportFunction = callable;
        }
      } else if (functionsInstance && typeof functionsInstance === 'function') {
        // Direct function reference
        generateReportFunction = functionsInstance;
      } else {
        throw new Error('Invalid functions instance provided');
      }

      const result = await Promise.race([
        generateReportFunction(functionData),
        new Promise((_, reject) =>
          setTimeout(() => reject(new Error('Report generation timeout')), timeout)
        )
      ]);

      console.log(`📊 Report generation successful on attempt ${attempt + 1}`);
      return result;

    } catch (error) {
      console.error(`📊 Report generation attempt ${attempt + 1} failed:`, error);

      // Check if this is a retryable error
      const isRetryable =
        error.message.includes('timeout') ||
        error.message.includes('internal') ||
        error.message.includes('500') ||
        error.message.includes('502') ||
        error.message.includes('503') ||
        error.message.includes('504') ||
        error.code === 'internal' ||
        error.code === 'deadline-exceeded';

      // If this is the last attempt or error is not retryable, throw
      if (attempt === maxRetries || !isRetryable) {
        if (error.message.includes('timeout')) {
          throw new Error('Report generation timed out. This often happens after periods of inactivity. Please try again.');
        }
        throw error;
      }

      // Calculate exponential backoff delay
      const delay = baseDelay * Math.pow(2, attempt);
      console.log(`📊 Retrying report generation in ${delay}ms...`);
      await new Promise(resolve => setTimeout(resolve, delay));
    }
  }
}

// Initialize content script when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}

console.log('TextRad Extension Content Script loaded');