/**
 * TextRad Dictation Extension - Transcription Service
 * 
 * Adapted from the original TextRad TranscriptionService for Chrome extension use
 */

export class ExtensionTranscriptionService {
  constructor(storageManager) {
    this.db = null;
    this.auth = null;
    this.functions = null;
    this.storageManager = storageManager;
    this.isTranscribing = false;
    this.timeoutId = null;
    this.currentTranscription = '';
    
    // Event callbacks
    this.onTranscriptionStart = null;
    this.onTranscriptionComplete = null;
    this.onTranscriptionError = null;
    this.onTranscriptionProgress = null;
  }
  
  /**
   * Initialize Firebase services for extension
   */
  async initializeFirebaseServices() {
    try {
      // Import our mock Firebase implementation instead of using remote scripts
      // This works around Content Security Policy restrictions in Chrome extensions
      const { 
        initializeApp, 
        getFirestore, 
        getAuth, 
        getFunctions, 
        firebaseConfig 
      } = await import('../firebase/mock-firebase.js');
      
      console.log('Using local Firebase implementation for extension');
      
      // Initialize Firebase with config
      const firebaseApp = initializeApp(firebaseConfig);
      
      // Set up Firebase services
      this.db = getFirestore(firebaseApp);
      this.auth = getAuth(firebaseApp);
      this.functions = getFunctions(firebaseApp);
      
      console.log('Firebase services initialized for transcription');
      return true;
    } catch (error) {
      console.error('Error initializing Firebase:', error);
      return false;
    }
  }
  
  /**
   * Transcribe audio using the TextRad web app as a proxy
   * @param {Blob} audioBlob - The recorded audio blob
   */
  async transcribeAudio(audioBlob) {
    if (this.isTranscribing) return;
    this.isTranscribing = true;
    
    // Check if user is logged in to TextRad before transcribing
    try {
      const response = await chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' });
      if (!response || !response.success || !response.tabs || response.tabs.length === 0) {
        this.isTranscribing = false;
        const error = new Error('You must be logged in to TextRad to transcribe audio. Please log in and try again.');
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(error);
        }
        throw error;
      }
      
      // Additional check: verify that we have actual logged-in tabs
      const hasLoggedInTabs = response.tabs.some(tab => 
        response.loggedInTabs && response.loggedInTabs[tab.id] === true
      );
      
      if (!hasLoggedInTabs) {
        this.isTranscribing = false;
        const error = new Error('No logged-in TextRad tabs found. Please log in to TextRad and try again.');
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(error);
        }
        throw error;
      }
      
      console.log('Login verification passed for transcription service');
    } catch (error) {
      this.isTranscribing = false;
      const loginError = new Error('Cannot verify login status. Please ensure you are logged in to TextRad.');
      if (typeof this.onTranscriptionError === 'function') {
        this.onTranscriptionError(loginError);
      }
      throw loginError;
    }
    
    // Check if the audio blob is valid and has sufficient data
    if (!audioBlob || audioBlob.size < 1000) {
      this.isTranscribing = false;
      const error = new Error('No audio recorded or recording too short. Please try again and speak clearly into your microphone.');
      if (typeof this.onTranscriptionError === 'function') {
        this.onTranscriptionError(error);
      }
      throw error;
    }
    
    // Save last audio blob for retry
    if (this.storageManager) {
      await this.storageManager.setLastAudioBlob(audioBlob);
    }
    
    // Call transcription start callback
    if (typeof this.onTranscriptionStart === 'function') {
      this.onTranscriptionStart();
    }
    
    // Set up timeout (90 seconds for transcription)
    this.timeoutId = setTimeout(() => {
      if (this.isTranscribing) {
        this.isTranscribing = false;
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(new Error('Transcription timed out. Please check your internet connection and try again.'));
        }
      }
    }, 90000);
    
    try {
      // Try three different approaches for transcription:
      // 1. Web Proxy - Use the TextRad web app to transcribe
      // 2. Mock Transcription - Fallback for testing
      // 3. Local placeholder - If all else fails
      
      // Convert audio to base64 first (needed for all approaches)
      const base64Audio = await this.blobToBase64(audioBlob);
      console.log("Audio converted to base64, size:", Math.round(base64Audio.length / 1024), "KB");
      
      // Check if the base64 data is too small (indicates potential recording issues)
      if (base64Audio.length < 1000) {
        throw new Error('Audio recording is too short or empty. Please try again and speak clearly into your microphone.');
      }
      
      if (typeof this.onTranscriptionProgress === 'function') {
        this.onTranscriptionProgress(0.3);
      }
      
      // APPROACH 1: Try direct API approach first (most reliable)
      try {
        // Import the direct transcription service
        const { DirectTranscriptionService } = await import('./direct-transcription-service.js');
        
        // Create and initialize the service
        const directService = new DirectTranscriptionService();
        
        // Pass the current transcription to the direct service for context
        if (this.currentTranscription) {
          directService.setCurrentTranscription(this.currentTranscription);
        }
        
        // Set up event handlers
        if (this.onTranscriptionProgress) {
          directService.onTranscriptionProgress = this.onTranscriptionProgress;
        }
        
        // Call the transcription service
        console.log('Attempting direct API transcription...');
        const transcription = await directService.transcribeAudio(audioBlob);
        
        if (transcription) {
          console.log('Direct API transcription successful');
          return this.handleTranscriptionSuccess(transcription);
        }
      } catch (directError) {
        console.warn('Direct API transcription failed:', directError);
        // Check if error indicates audio issues rather than login problems
        if (directError.message && (
            directError.message.includes('audio') || 
            directError.message.includes('empty') || 
            directError.message.includes('short') ||
            directError.message.includes('no speech detected'))) {
          throw directError; // Pass audio-specific errors through
        }
        // Continue to fallback methods for other types of errors
      }
      
      // APPROACH 2: Try web proxy if available
      try {
        // Import the web proxy service
        const { transcribeAudioViaProxy, initializeProxyService } = await import('../content/web-proxy-service.js');
        
        // Find TextRad tabs
        const tabs = await this.findTextRadTabs();
        
        if (tabs && tabs.length > 0) {
          // Initialize proxy service in the first TextRad tab
          const tabId = tabs[0].id;
          console.log('Initializing proxy service in tab:', tabId);
          await initializeProxyService(tabId);
          
          // Try to transcribe via the proxy
          console.log('Attempting transcription via web proxy...');
          const transcription = await transcribeAudioViaProxy(tabId, base64Audio);
          
          if (transcription) {
            console.log('Transcription via web proxy successful');
            return this.handleTranscriptionSuccess(transcription);
          } else {
            // Empty transcription from a functioning service likely means no speech was detected
            throw new Error('No speech detected in the recording. Please try again and speak clearly into your microphone.');
          }
        } else {
          console.log('No TextRad tabs found for proxy transcription');
        }
      } catch (proxyError) {
        console.warn('Web proxy transcription failed:', proxyError);
        // Check if error indicates audio issues rather than login problems
        if (proxyError.message && (
            proxyError.message.includes('audio') || 
            proxyError.message.includes('empty') || 
            proxyError.message.includes('short') ||
            proxyError.message.includes('no speech'))) {
          throw proxyError; // Pass audio-specific errors through
        }
        // Continue to fallback methods for other types of errors
      }
      
      // APPROACH 3: Initialize Firebase implementation for mock/direct transcription
      if (!this.db || !this.auth || !this.functions) {
        await this.initializeFirebaseServices();
      }
      
      // Return error if we have no service
      if (!this.functions) {
        console.warn('No transcription service available');
        const error = new Error('No transcription service available. Please check your connection to TextRad Web App.');
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(error);
        }
        throw error;
      }
      
      // APPROACH 4: Check if we need to show login error or audio error
      // If we have tabs but transcription still failed, it's more likely an audio issue
      const tabs = await this.findTextRadTabs();
      if (tabs && tabs.length > 0) {
        console.warn('All transcription methods failed despite having TextRad tabs');
        const error = new Error('Transcription failed. The recording may be too short or no speech was detected. Please try again and speak clearly.');
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(error);
        }
        throw error;
      } else {
        // No tabs, so likely a login issue
        console.warn('All transcription methods failed with no TextRad tabs');
        const error = new Error('Transcription failed. Please check your connection and login to TextRad.');
        if (typeof this.onTranscriptionError === 'function') {
          this.onTranscriptionError(error);
        }
        throw error;
      }
    } catch (error) {
      console.error('Transcription error:', error);
      
      // Clear timeout
      if (this.timeoutId) {
        clearTimeout(this.timeoutId);
        this.timeoutId = null;
      }
      
      this.isTranscribing = false;
      
      // Call error callback
      if (typeof this.onTranscriptionError === 'function') {
        this.onTranscriptionError(error);
      }
      
      throw error;
    }
  }
  
  /**
   * Handle successful transcription
   * @param {string} transcription - The transcription text
   */
  handleTranscriptionSuccess(transcription) {
    // Clear timeout
    if (this.timeoutId) {
      clearTimeout(this.timeoutId);
      this.timeoutId = null;
    }
    
    this.isTranscribing = false;
    
    // Store transcription
    this.currentTranscription = transcription;
    if (this.storageManager) {
      this.storageManager.setLastTranscription(transcription);
    }
    
    // Call completion callback
    if (typeof this.onTranscriptionComplete === 'function') {
      this.onTranscriptionComplete(transcription);
    }
    
    console.log("Transcription completed successfully");
    return transcription;
  }
  
  /**
   * Fallback mock transcription
   * @param {string} base64Audio - Base64-encoded audio data
   * @returns {Promise<string>} - Mock transcription result
   */
  async mockTranscribe(base64Audio) {
    // Simulate network delay
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    // We no longer use mock transcriptions - throw an error instead
    throw new Error('Transcription service unavailable. Please log in to TextRad.');
  }
  
  /**
   * Find TextRad tabs for proxy transcription
   * @returns {Promise<Array>} - Array of TextRad tabs
   */
  async findTextRadTabs() {
    return new Promise((resolve) => {
      chrome.tabs.query({ 
        url: ['*://*.textrad.com/*', '*://*.textrad.in/*', '*://localhost/*', '*://augmento.vijayadiagnostic.com/*'] 
      }, (tabs) => {
        resolve(tabs || []);
      });
    });
  }
  
  /**
   * Convert Blob to Base64 string
   * @param {Blob} blob - The blob to convert
   * @returns {Promise<string>} Base64 string
   */
  blobToBase64(blob) {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onloadend = () => {
        // Get the base64 string (remove the data URL prefix)
        const base64String = reader.result.split(',')[1];
        resolve(base64String);
      };
      reader.onerror = reject;
      reader.readAsDataURL(blob);
    });
  }
  
  /**
   * Cancel ongoing transcription
   */
  cancelTranscription() {
    if (this.timeoutId) {
      clearTimeout(this.timeoutId);
      this.timeoutId = null;
    }
    
    this.isTranscribing = false;
  }
  
  /**
   * Reset the current transcription
   */
  resetTranscription() {
    console.log("Transcription reset - next recording will start fresh");
    this.currentTranscription = '';
  }
  
  /**
   * Set the current transcription
   * @param {string} text - The text to store
   */
  async setCurrentTranscription(text) {
    this.currentTranscription = text;
    
    // Also update the DirectTranscriptionService if available
    try {
      const { DirectTranscriptionService } = await import('./direct-transcription-service.js');
      const directService = new DirectTranscriptionService();
      directService.setCurrentTranscription(text);
    } catch (err) {
      console.warn('Could not update DirectTranscriptionService:', err);
    }
    
    // Store in local storage
    if (this.storageManager) {
      this.storageManager.setLastTranscription(text);
    }
  }
}