/**
 * TextRad Dictation Extension - Audio Recorder
 * 
 * Adapted from the original TextRad AudioRecorder for Chrome extension use
 */

export class ExtensionAudioRecorder {
  constructor(visualizerCanvas, storageManager) {
    this.mediaRecorder = null;
    this.audioContext = null;
    this.analyser = null;
    this.visualizerCanvas = visualizerCanvas;
    this.canvasContext = visualizerCanvas ? visualizerCanvas.getContext('2d') : null;
    this.audioChunks = [];
    this.isRecording = false;
    this.audioStream = null;
    this.silenceDetectionInterval = null;
    this.silenceTimeout = 40000; // 40 seconds of silence (same as main app)
    this.silenceThreshold = 0.01; // Threshold for determining silence
    this.lastAudioLevel = 0;
    this.lastNonSilenceTime = 0;
    this.storageManager = storageManager;
    
    // Event callbacks
    this.onRecordingStart = null;
    this.onRecordingStop = null;
    this.onSilenceDetected = null;
    this.onAudioProcessed = null;
    this.onError = null;
    
    // Bind methods
    this.drawVisualizer = this.drawVisualizer.bind(this);
    this.checkSilence = this.checkSilence.bind(this);
    
    // Set up visualizer if canvas exists
    if (this.canvasContext) {
      this.setupVisualizer();
    }
    
    // Add extension specific event listener for popup closing
    window.addEventListener('beforeunload', this.handleExtensionClosing.bind(this));
  }
  
  /**
   * Handle extension popup closing
   * With our collapsed popup approach, this method won't actually get called
   * when collapsing, only when actually closing
   */
  async handleExtensionClosing() {
    console.log('Extension popup closing');
    
    // Check if we should stop recording
    // If we are in viewMode=mini or a detached window, don't stop recording
    const isMiniMode = window.location.search.includes('viewMode=mini');
    
    if (isMiniMode) {
      console.log('Popup is in mini mode, keeping recording active');
      // Don't stop recording or release resources when in mini mode
      return;
    }
    
    // Only reach here if the popup is actually closing
    if (this.isRecording) {
      console.log('Recording was in progress - stopping due to popup closing');
      try {
        // Explicitly stop the recording
        this.stopRecording();
      } catch (error) {
        console.error('Error stopping recording on popup close:', error);
      }
      
      // Reset recording state in storage
      await this.storageManager.setRecordingInProgress(false);
    }
    
    // Clean up resources when actually closing
    this.releaseAudioResources();
  }
  
  /**
   * Restore from previous state
   */
  async restoreFromState() {
    const wasRecording = await this.storageManager.getRecordingInProgress();
    if (wasRecording) {
      console.log('Restoring previous recording session');
      await this.startRecording();
    }
  }
  
  /**
   * Set up the visualizer canvas
   */
  setupVisualizer() {
    const dpr = window.devicePixelRatio || 1;
    const rect = this.visualizerCanvas.getBoundingClientRect();
    
    // Set canvas size accounting for device pixel ratio
    this.visualizerCanvas.width = rect.width * dpr;
    this.visualizerCanvas.height = rect.height * dpr;
    this.canvasContext.scale(dpr, dpr);
    
    // Clear canvas with background
    this.clearVisualizer();
  }
  
  /**
   * Clears the visualizer with a static pattern instead of blank background
   */
  clearVisualizer() {
    if (!this.canvasContext || !this.visualizerCanvas) return;
    
    const width = this.visualizerCanvas.width / window.devicePixelRatio;
    const height = this.visualizerCanvas.height / window.devicePixelRatio;
    
    // Background
    this.canvasContext.fillStyle = '#111827';
    this.canvasContext.fillRect(0, 0, width, height);
    
    // Draw a static line to indicate it's not recording
    this.canvasContext.beginPath();
    this.canvasContext.moveTo(0, height / 2);
    this.canvasContext.lineTo(width, height / 2);
    this.canvasContext.strokeStyle = '#2563eb';
    this.canvasContext.lineWidth = 1;
    this.canvasContext.stroke();
  }
  
  /**
   * Initialize audio context and media stream
   */
  async setupAudioSession() {
    if (!this.audioContext) {
      this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
    }
    
    try {
      // Request microphone access directly with better error handling
      this.audioStream = await navigator.mediaDevices.getUserMedia({
        audio: {
          echoCancellation: true,
          noiseSuppression: true,
          autoGainControl: true
        },
        video: false
      });
      
      // Set up analyser node for visualization and silence detection
      this.analyser = this.audioContext.createAnalyser();
      this.analyser.fftSize = 256;
      const source = this.audioContext.createMediaStreamSource(this.audioStream);
      source.connect(this.analyser);
      
      return true;
    } catch (error) {
      console.error('Error accessing microphone:', error);
      
      // Provide more user-friendly error messages
      if (error.name === 'NotAllowedError' || error.name === 'PermissionDeniedError') {
        if (this.onError) {
          this.onError(new Error('Microphone access was denied. Please click the microphone icon in your browser\'s address bar and select "Allow", then try again.'));
        }
      } else if (error.name === 'NotFoundError' || error.name === 'DevicesNotFoundError') {
        if (this.onError) {
          this.onError(new Error('No microphone found. Please connect a microphone and try again.'));
        }
      } else {
        if (this.onError) {
          this.onError(new Error('Failed to access microphone: ' + error.message + '. Please ensure microphone permissions are granted and try again.'));
        }
      }
      return false;
    }
  }
  
  /**
   * Start recording audio
   */
  async startRecording() {
    if (this.isRecording) return false;
    
    // Always set up a fresh audio session
    // This ensures we have clean state for each recording
    const success = await this.setupAudioSession();
    if (!success) return false;
    
    // Reset audio chunks
    this.audioChunks = [];
    
    // Create media recorder
    this.mediaRecorder = new MediaRecorder(this.audioStream);
    
    // Handle data available event
    this.mediaRecorder.ondataavailable = (event) => {
      if (event.data.size > 0) {
        this.audioChunks.push(event.data);
      }
    };
    
    // Start recording
    this.mediaRecorder.start();
    this.isRecording = true;
    
    // Start visualizer animation
    if (this.canvasContext) {
      this.animationFrame = requestAnimationFrame(this.drawVisualizer);
    }
    
    // Start silence detection
    this.lastNonSilenceTime = Date.now();
    this.silenceDetectionInterval = setInterval(this.checkSilence, 1000);
    
    // Fire callback if defined
    if (typeof this.onRecordingStart === 'function') {
      this.onRecordingStart();
    }
    
    return true;
  }
  
  /**
   * Stop recording audio
   */
  stopRecording() {
    if (!this.isRecording) {
      console.log("Not recording, nothing to stop");
      return;
    }
    
    if (!this.mediaRecorder) {
      console.log("No media recorder, resetting state");
      this.isRecording = false;
      return;
    }
    
    console.log(`Stopping recording. Media recorder state: ${this.mediaRecorder.state}`);
    
    try {
      // Only stop if the state is "recording"
      if (this.mediaRecorder.state === "recording") {
        this.mediaRecorder.stop();
        console.log("Media recorder stopped");
      } else {
        console.log(`Cannot stop media recorder in state: ${this.mediaRecorder.state}`);
      }
    } catch (err) {
      console.error("Error stopping media recorder:", err);
    }
    
    // IMPORTANT: Update internal state BEFORE stopping animation 
    // so drawVisualizer knows to stop
    this.isRecording = false;
    
    // Stop visualizer animation - do this TWICE to ensure it's stopped
    if (this.animationFrame) {
      cancelAnimationFrame(this.animationFrame);
      this.animationFrame = null;
      console.log("Animation frame cancelled");
    }
    
    // Force clear the visualizer to show static line
    this.clearVisualizer();
    console.log("Visualizer cleared to static line");
    
    // Double-check no animation is running by cancelling again
    if (this.animationFrame) {
      cancelAnimationFrame(this.animationFrame);
      this.animationFrame = null;
      console.log("Second animation frame cancellation");
    }
    
    // Clear silence detection interval
    if (this.silenceDetectionInterval) {
      clearInterval(this.silenceDetectionInterval);
      this.silenceDetectionInterval = null;
    }
    
    // Process recorded audio after a brief delay to ensure all data is collected
    setTimeout(() => {
      // Check if we have any audio chunks
      if (this.audioChunks.length === 0) {
        console.log("No audio chunks recorded, skipping callback");
        return;
      }
      
      const audioBlob = new Blob(this.audioChunks, { type: 'audio/webm' });
      console.log(`Created audio blob: ${audioBlob.size} bytes`);
      
      // Save last audio blob
      if (this.storageManager) {
        this.storageManager.setLastAudioBlob(audioBlob);
      }
      
      // Fire callback with audio blob if defined
      if (typeof this.onRecordingStop === 'function') {
        this.onRecordingStop(audioBlob);
      } else {
        console.log("No onRecordingStop callback registered");
      }
    }, 300);
  }
  
  /**
   * Draw audio visualizer
   */
  drawVisualizer() {
    // IMPORTANT: Strict check to immediately stop if not recording
    if (!this.isRecording) {
      // Cancel any future animation frames to ensure visualizer stops
      if (this.animationFrame) {
        cancelAnimationFrame(this.animationFrame);
        this.animationFrame = null;
      }
      
      // Show the static line
      this.clearVisualizer();
      console.log("drawVisualizer: immediately returned due to !isRecording");
      return;
    }
    
    // Second check for required resources
    if (!this.analyser || !this.canvasContext) {
      // If we don't have an analyser, clear the visualizer
      this.clearVisualizer();
      console.log("drawVisualizer: no analyser or context");
      
      // Cancel any future animation frames
      if (this.animationFrame) {
        cancelAnimationFrame(this.animationFrame);
        this.animationFrame = null;
      }
      return;
    }
    
    const bufferLength = this.analyser.frequencyBinCount;
    const dataArray = new Uint8Array(bufferLength);
    this.analyser.getByteFrequencyData(dataArray);
    
    // Calculate audio level (used for silence detection too)
    let sum = 0;
    for (let i = 0; i < bufferLength; i++) {
      sum += dataArray[i];
    }
    this.lastAudioLevel = sum / (bufferLength * 255); // Normalize to 0-1
    
    // Clear canvas
    const width = this.visualizerCanvas.width / window.devicePixelRatio;
    const height = this.visualizerCanvas.height / window.devicePixelRatio;
    this.canvasContext.fillStyle = '#111827';
    this.canvasContext.fillRect(0, 0, width, height);
    
    // Draw bars - with reduced sensitivity and more subtle appearance
    const barWidth = width / bufferLength * 2.5;
    let x = 0;
    
    for (let i = 0; i < bufferLength; i++) {
      // Further reduce the bar height to half of the original height by using 0.35 scaling
      const barHeight = (dataArray[i] / 255) * height * 0.35; 
      
      // Use a more subtle gradient with lower saturation and lightness
      const hue = 220 - (i / bufferLength * 40);
      this.canvasContext.fillStyle = `hsl(${hue}, 70%, 50%)`;
      
      // Draw smaller bars with some spacing
      this.canvasContext.fillRect(x, height - barHeight, barWidth - 1, barHeight);
      x += barWidth + 1;
    }
    
    // IMPORTANT: Schedule next frame ONLY if we're definitely still recording
    // This check is critical to ensure animation stops when recording stops
    if (this.isRecording) {
      this.animationFrame = requestAnimationFrame(this.drawVisualizer);
    } else {
      // Double check - if we're no longer recording, make sure to draw static line
      this.clearVisualizer();
      console.log("drawVisualizer: detected !isRecording at end of function");
    }
  }
  
  /**
   * Check for silence to auto-stop recording
   */
  checkSilence() {
    if (!this.isRecording) return;
    
    // If audio level is above threshold, update last non-silence time
    if (this.lastAudioLevel > this.silenceThreshold) {
      this.lastNonSilenceTime = Date.now();
      return;
    }
    
    // If silence exceeds timeout, stop recording
    const silenceDuration = Date.now() - this.lastNonSilenceTime;
    if (silenceDuration > this.silenceTimeout) {
      console.log('Silence detected for', silenceDuration, 'ms. Auto-stopping recording.');
      this.stopRecording();
      
      // Fire callback if defined
      if (typeof this.onSilenceDetected === 'function') {
        this.onSilenceDetected();
      }
    }
  }
  
  /**
   * Clean up resources
   */
  cleanup() {
    if (this.isRecording) {
      this.stopRecording();
    }
    
    // Release all audio resources
    this.releaseAudioResources();
  }
  
  /**
   * Release all audio resources
   * Helper method to fully release microphone access
   */
  releaseAudioResources() {
    console.log("FORCEFULLY releasing ALL audio resources");
    
    // Stop and release all audio tracks
    try {
      // 1. Stop current recording if needed
      if (this.mediaRecorder && this.mediaRecorder.state === 'recording') {
        try {
          this.mediaRecorder.stop();
          console.log("Stopped active MediaRecorder");
        } catch (err) {
          console.warn("Error stopping MediaRecorder:", err);
        }
      }
      
      // 2. Stop all audio tracks with extra safety checks
      if (this.audioStream) {
        const tracks = this.audioStream.getTracks();
        console.log(`Found ${tracks.length} tracks to stop`);
        
        tracks.forEach(track => {
          try {
            track.stop();
            console.log(`Audio track (kind: ${track.kind}) stopped`);
          } catch (err) {
            console.warn(`Error stopping track (kind: ${track.kind}):`, err);
          }
        });
        
        // Explicitly null the stream
        this.audioStream = null;
      }
      
      // 3. Close the audio context with immediate error handling
      if (this.audioContext) {
        try {
          const closeContext = async () => {
            try {
              if (this.audioContext.state !== 'closed') {
                await this.audioContext.close();
                console.log("Audio context closed successfully");
              }
            } catch (err) {
              console.warn("Error during AudioContext.close():", err);
            }
          };
          
          // Execute close but don't wait for it
          closeContext();
        } catch (err) {
          console.warn("Error initiating context close:", err);
        }
        
        // Explicitly null the context regardless of close result
        this.audioContext = null;
      }
      
      // 4. Explicitly null all other audio components
      this.analyser = null;
      this.mediaRecorder = null;
      this.audioChunks = [];
      
      // 5. Reset recording state flag
      this.isRecording = false;
      
      // 6. Cancel any pending animation frames
      if (this.animationFrame) {
        cancelAnimationFrame(this.animationFrame);
        this.animationFrame = null;
      }
      
      // 7. Clear visualizer to empty state with static line
      this.clearVisualizer();
      console.log("Visualizer cleared as part of resource release");
      
      // 8. Clear any remaining intervals
      if (this.silenceDetectionInterval) {
        clearInterval(this.silenceDetectionInterval);
        this.silenceDetectionInterval = null;
      }
      
      console.log("Audio resources fully released - COMPLETE");
    } catch (err) {
      console.error("Error while releasing audio resources:", err);
      // Even on error, force-null everything
      this.audioStream = null;
      this.audioContext = null;
      this.analyser = null;
      this.mediaRecorder = null;
      this.isRecording = false;
    }
  }
  
  /**
   * Release microphone temporarily without fully closing context
   * Used for pausing between recordings
   */
  pauseMicrophone() {
    // Only stop the current tracks, but don't close context
    if (this.audioStream) {
      this.audioStream.getTracks().forEach(track => track.stop());
      this.audioStream = null;
    }
    
    // Stop visualizer animation when pausing
    if (this.animationFrame) {
      cancelAnimationFrame(this.animationFrame);
      this.animationFrame = null;
    }
    
    // Clear visualizer to inactive state with static line
    this.clearVisualizer();
    
    this.mediaRecorder = null;
    console.log("Microphone paused, context preserved");
  }
}