/**
 * TextRad Dictation Extension - Offscreen Document for Background Recording
 * 
 * This document maintains audio recording in the background even when the
 * popup is closed. It communicates with the service worker to send audio data.
 */

let mediaRecorder = null;
let audioChunks = [];
let audioContext = null;
let analyser = null;
let audioStream = null;
let silenceDetectionInterval = null;
let recording = false;
let lastAudioLevel = 0;
let lastNonSilenceTime = Date.now();
let silenceThreshold = 0.01;
let silenceTimeout = 40000; // 40 seconds (default)

// Set up message listener for commands from service worker
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Offscreen document received message:', message);
  
  if (message.target !== 'offscreen') return false;
  
  // Log that we're processing this message with details
  console.log(`Processing offscreen message: ${message.action}`, message);
  
  switch (message.action) {
    case 'START_RECORDING':
      console.log('Start recording request received in offscreen document');
      startRecording()
        .then(success => {
          console.log('startRecording result:', success);
          sendResponse({ success });
        })
        .catch(error => {
          console.error('Error starting recording:', error);
          sendResponse({ success: false, error: error.message });
        });
      return true; // Keep the message channel open
      
    case 'STOP_RECORDING':
      stopRecording()
        .then(audioBlob => {
          if (audioBlob) {
            // Convert to array buffer for transfer
            const reader = new FileReader();
            reader.onloadend = () => {
              sendResponse({ 
                success: true, 
                audioData: reader.result,
                audioSize: audioBlob.size,
                audioType: audioBlob.type
              });
            };
            reader.onerror = (err) => {
              sendResponse({ success: false, error: 'Failed to convert audio: ' + err });
            };
            reader.readAsArrayBuffer(audioBlob);
          } else {
            sendResponse({ success: false, error: 'No audio recorded' });
          }
        })
        .catch(error => {
          console.error('Error stopping recording:', error);
          sendResponse({ success: false, error: error.message });
        });
      return true; // Keep the message channel open
      
    case 'CHECK_STATUS':
      sendResponse({ 
        isRecording: recording,
        audioChunksCount: audioChunks.length,
        lastAudioLevel: lastAudioLevel
      });
      return false;
      
    case 'SET_SILENCE_TIMEOUT':
      if (message.timeout && typeof message.timeout === 'number') {
        silenceTimeout = message.timeout;
        sendResponse({ success: true });
      } else {
        sendResponse({ success: false, error: 'Invalid timeout value' });
      }
      return false;
  }

  return false;
});

// Set up audio recording
async function setupAudioSession() {
  if (!audioContext) {
    audioContext = new (window.AudioContext || window.webkitAudioContext)();
  }
  
  try {
    // First try to check if permission is already granted
    try {
      const permissionStatus = await navigator.permissions.query({ name: 'microphone' });
      if (permissionStatus.state === 'denied') {
        throw new Error('Microphone access is blocked. Please allow microphone access in your browser settings.');
      }
    } catch (permissionError) {
      // Permissions API may not be available, continue with getUserMedia
      console.log('Permissions API not available:', permissionError);
    }
    
    audioStream = await navigator.mediaDevices.getUserMedia({
      audio: {
        echoCancellation: true,
        noiseSuppression: true,
        autoGainControl: true
      },
      video: false
    });
    
    // Set up analyser node for visualization and silence detection
    analyser = audioContext.createAnalyser();
    analyser.fftSize = 256;
    const source = audioContext.createMediaStreamSource(audioStream);
    source.connect(analyser);
    
    // Notify service worker about successful setup
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'AUDIO_SESSION_READY'
    });
    
    return true;
  } catch (error) {
    console.error('Error accessing microphone:', error);
    
    // Notify service worker about error
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'RECORDING_ERROR',
      error: error.message || 'Microphone access failed'
    });
    
    return false;
  }
}

// Start recording audio
async function startRecording() {
  if (recording) return true; // Already recording
  
  // Always set up a fresh audio session
  const success = await setupAudioSession();
  if (!success) return false;
  
  // Reset audio chunks
  audioChunks = [];
  
  try {
    // Create media recorder
    mediaRecorder = new MediaRecorder(audioStream);
    
    // Handle data available event
    mediaRecorder.ondataavailable = (event) => {
      if (event.data.size > 0) {
        audioChunks.push(event.data);
        
        // Send chunk count update to service worker periodically
        if (audioChunks.length % 5 === 0) {
          chrome.runtime.sendMessage({
            target: 'service-worker',
            action: 'RECORDING_UPDATE',
            chunksCount: audioChunks.length,
            lastAudioLevel: lastAudioLevel
          });
        }
      }
    };
    
    // Start recording
    mediaRecorder.start(1000); // Capture in 1-second chunks
    recording = true;
    lastNonSilenceTime = Date.now(); // Reset silence detection timer
    
    // Start silence detection
    startSilenceDetection();
    
    // Notify service worker that recording has started
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'RECORDING_STARTED'
    });
    
    return true;
  } catch (error) {
    console.error('Error starting MediaRecorder:', error);
    
    // Notify service worker about error
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'RECORDING_ERROR',
      error: error.message || 'Failed to start recording'
    });
    
    return false;
  }
}

// Start silence detection
function startSilenceDetection() {
  // Clear any existing interval
  if (silenceDetectionInterval) {
    clearInterval(silenceDetectionInterval);
  }
  
  silenceDetectionInterval = setInterval(() => {
    if (!recording || !analyser) return;
    
    const bufferLength = analyser.frequencyBinCount;
    const dataArray = new Uint8Array(bufferLength);
    analyser.getByteFrequencyData(dataArray);
    
    // Calculate audio level
    let sum = 0;
    for (let i = 0; i < bufferLength; i++) {
      sum += dataArray[i];
    }
    lastAudioLevel = sum / (bufferLength * 255); // Normalize to 0-1
    
    // If audio level is above threshold, update last non-silence time
    if (lastAudioLevel > silenceThreshold) {
      lastNonSilenceTime = Date.now();
      return;
    }
    
    // If silence exceeds timeout, stop recording
    const silenceDuration = Date.now() - lastNonSilenceTime;
    if (silenceDuration > silenceTimeout) {
      console.log('Silence detected for', silenceDuration, 'ms. Auto-stopping recording.');
      
      // Notify service worker about silence detection
      chrome.runtime.sendMessage({
        target: 'service-worker',
        action: 'SILENCE_DETECTED'
      });
      
      // Stop recording and send audio to service worker
      stopRecording().then(audioBlob => {
        if (audioBlob) {
          sendAudioToServiceWorker(audioBlob);
        }
      });
    }
  }, 1000);
}

// Stop recording audio
async function stopRecording() {
  return new Promise((resolve, reject) => {
    console.log('Stop recording called, current state:', { recording, mediaRecorder: mediaRecorder?.state });
    
    if (!recording || !mediaRecorder) {
      recording = false;
      if (silenceDetectionInterval) {
        clearInterval(silenceDetectionInterval);
        silenceDetectionInterval = null;
      }
      resolve(null);
      return;
    }
    
    try {
      // Only stop if the state is "recording"
      if (mediaRecorder.state === "recording") {
        // Set up the onstop handler to resolve with the audio blob
        mediaRecorder.onstop = () => {
          console.log('MediaRecorder onstop event fired, chunks:', audioChunks.length);
          
          // Clear silence detection interval
          if (silenceDetectionInterval) {
            clearInterval(silenceDetectionInterval);
            silenceDetectionInterval = null;
          }
          
          // Update recording state
          recording = false;
          
          // Process recorded audio
          if (!audioChunks || audioChunks.length === 0) {
            console.warn('No audio chunks recorded');
            resolve(null);
            return;
          }
          
          const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
          console.log(`Created audio blob: ${audioBlob.size} bytes`);
          
          // Notify service worker recording has stopped
          chrome.runtime.sendMessage({
            target: 'service-worker',
            action: 'RECORDING_STOPPED',
            duration: (Date.now() - lastNonSilenceTime) / 1000,
            blobSize: audioBlob.size
          });
          
          resolve(audioBlob);
        };
        
        // Set up onerror handler
        mediaRecorder.onerror = (event) => {
          console.error('MediaRecorder error:', event);
          recording = false;
          reject(new Error('MediaRecorder error: ' + (event.error?.message || 'Unknown error')));
        };
        
        // Stop recording
        mediaRecorder.stop();
        console.log('MediaRecorder stop called');
      } else {
        console.warn(`Cannot stop media recorder in state: ${mediaRecorder.state}`);
        recording = false;
        
        // Clear silence detection interval
        if (silenceDetectionInterval) {
          clearInterval(silenceDetectionInterval);
          silenceDetectionInterval = null;
        }
        
        // Try to resolve with any audio we have
        if (audioChunks && audioChunks.length > 0) {
          const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
          console.log(`Created audio blob from ${audioChunks.length} chunks: ${audioBlob.size} bytes`);
          resolve(audioBlob);
        } else {
          resolve(null);
        }
      }
    } catch (err) {
      console.error("Error stopping media recorder:", err);
      recording = false;
      
      // Clear silence detection interval
      if (silenceDetectionInterval) {
        clearInterval(silenceDetectionInterval);
        silenceDetectionInterval = null;
      }
      
      reject(err);
    }
  });
}

// Release audio resources
function releaseAudioResources() {
  console.log("Releasing audio resources");
  
  try {
    // Stop all audio tracks
    if (audioStream) {
      audioStream.getTracks().forEach(track => {
        try {
          track.stop();
          console.log(`Audio track (kind: ${track.kind}) stopped`);
        } catch (err) {
          console.warn(`Error stopping track (kind: ${track.kind}):`, err);
        }
      });
      
      audioStream = null;
    }
    
    // Close the audio context
    if (audioContext) {
      try {
        audioContext.close().then(() => {
          console.log("Audio context closed successfully");
        }).catch(err => {
          console.warn("Error during AudioContext.close():", err);
        });
      } catch (err) {
        console.warn("Error closing audio context:", err);
      }
      
      audioContext = null;
    }
    
    // Reset other audio components
    analyser = null;
    mediaRecorder = null;
    audioChunks = [];
    recording = false;
    
    // Clear any remaining intervals
    if (silenceDetectionInterval) {
      clearInterval(silenceDetectionInterval);
      silenceDetectionInterval = null;
    }
    
    console.log("Audio resources fully released");
  } catch (err) {
    console.error("Error while releasing audio resources:", err);
    
    // Force-null everything on error
    audioStream = null;
    audioContext = null;
    analyser = null;
    mediaRecorder = null;
    recording = false;
  }
}

// Send audio data to service worker
function sendAudioToServiceWorker(audioBlob) {
  if (!audioBlob || audioBlob.size === 0) {
    console.warn('No audio blob to send to service worker');
    return;
  }
  
  // Convert blob to array buffer for transfer
  const reader = new FileReader();
  reader.onload = () => {
    const arrayBuffer = reader.result;
    
    // Send the audio data to the service worker
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'AUDIO_RECORDED',
      audioData: arrayBuffer,
      audioSize: audioBlob.size,
      audioType: audioBlob.type
    });
  };
  
  reader.onerror = (error) => {
    console.error('Error reading audio blob:', error);
    
    chrome.runtime.sendMessage({
      target: 'service-worker',
      action: 'RECORDING_ERROR',
      error: 'Failed to process recorded audio'
    });
  };
  
  reader.readAsArrayBuffer(audioBlob);
}

// Log that the offscreen document is ready
console.log('TextRad Dictation offscreen document ready for background recording');

// Notify service worker that offscreen document is ready
chrome.runtime.sendMessage({
  target: 'service-worker',
  action: 'OFFSCREEN_DOCUMENT_READY'
});