/**
 * TextRad Dictation Extension - Popup Script
 * 
 * Responsibilities:
 * - User interface for recording and managing dictations
 * - Communication with service worker (background script)
 * - Audio recording and visualization
 * - Displaying transcription results
 */

// Import AudioRecorder and TranscriptionService
import { ExtensionAudioRecorder } from '../lib/audio-recorder.js';
import { ExtensionTranscriptionService } from '../lib/transcription-service.js';
import { StorageManager } from '../lib/storage-manager.js';

// UI Elements
let recordButton;
let visualizerCanvas;
let recordingStatus;
// Timer removed
let recordingPanel;
let transcriptionPanel;
let transcriptionText;
let newDictationButton;
let copyButton;
let sendButton;
let settingsButton;
let settingsPanel;
let closeSettingsButton;
let saveSettingsButton;
let appStatusPanel;
let appStatusIndicator;
let appStatusText;
let openTextRadButton;
let loadingOverlay;
let loadingText;
let loginRequiredMessage;
let loginButton;
let toastNotification;
let toastMessage;

// New template selection UI elements
let templateSelectionPanel;
let formatSelector;
let formatSearchInput;
let formatDropdownList;
let formatNoResults;
let availableFormats = [];
let filteredFormats = [];
let selectedFormatIndex = -1;
let templateLoading;

// Transcription panel elements for collapse functionality
let templateError;
let templateErrorMessage;
let generateReportPanel;
let generateButton;
let generateButtonText;
let generationProgress;
let generationStatus;

// Main/Mini UI Elements
let expandedContainer;
let collapsedContainer;
let collapseButton;
let expandButton;
let miniRecordButton;
let closeButton;
let collapsedStatus;

// Settings Elements
let autoSendSetting;
let silenceTimeoutSetting;
let silenceTimeoutValue;

// Service instances
let audioRecorder;
let transcriptionService;
let storageManager;

// State tracking
let isRecording = false;
let isTranscribing = false;
let recordingStartTime = 0;
let recordingTimer = null;
let currentTranscription = '';
let textradTabs = [];
let selectedTabId = null;

// Template/format state
let selectedFormat = null;
let isGeneratingReport = false;
let currentReportContent = null;

// Login status monitoring
let loginCheckInterval = null;
let lastLoginCheck = 0;

// Flag to determine whether to append or replace transcription
// true = append to existing transcription (set by main record button)
// false = start fresh (set by New Dictation button)
// Default to undefined so we know when it's not been explicitly set
let recordMoreButtonClicked = undefined;

/**
 * Initialize popup
 */
async function initialize() {
  console.log('TextRad Dictation Extension popup initialized');
  
  // Initialize services
  storageManager = new StorageManager();
  await storageManager.initialize();
  
  // Get UI elements
  getUIElements();
  
  // Set up event listeners
  setupEventListeners();

  // Update initial container state
  updateContainerState();
  
  // Initialize audio recorder and transcription service
  initializeServices();
  
  // Disable recording buttons until login status is verified
  if (recordButton) {
    recordButton.disabled = true;
    recordButton.classList.add('disabled');
  }
  if (miniRecordButton) {
    miniRecordButton.disabled = true;
    miniRecordButton.classList.add('disabled');
  }
  
  // No need to move buttons anymore - using simpler approach
  
  // Start with not-logged-in state by default
  updateAppStatus('not-logged-in');
  
  // Check for TextRad tabs
  findTextRadTabs();
  
  // Load settings
  loadSettings();
  
  // Reset record mode flags to ensure clean start
  recordMoreButtonClicked = undefined;
  
  // Check URL for mini mode parameter
  const urlParams = new URLSearchParams(window.location.search);
  if (urlParams.has('viewMode') && urlParams.get('viewMode') === 'mini') {
    // Switch to mini mode immediately from URL param
    expandedContainer.classList.add('hidden');
    collapsedContainer.classList.remove('hidden');
    document.body.classList.add('mini-mode');
    document.documentElement.classList.add('mini-mode');
    
    // Force window size to exactly match mini UI
    window.resizeTo(150, 40);
    
    updateMiniStatus();
  } else {
    // Otherwise, load saved view mode
    chrome.storage.local.get(['viewMode'], (result) => {
      if (result.viewMode === 'mini') {
        // Switch to mini mode
        expandedContainer.classList.add('hidden');
        collapsedContainer.classList.remove('hidden');
        document.body.classList.add('mini-mode');
        document.documentElement.classList.add('mini-mode');
        
        // Force window size to exactly match mini UI
        window.resizeTo(140, 32);
        
        updateMiniStatus();
      } else {
        // Default to expanded mode
        expandedContainer.classList.remove('hidden');
        collapsedContainer.classList.add('hidden');
        document.body.classList.remove('mini-mode');
        document.documentElement.classList.remove('mini-mode');
      }
    });
  }
  
  // Check state of recording from service worker
  const extensionState = await chrome.runtime.sendMessage({ action: 'GET_STATE' });
  console.log('Extension state:', extensionState);
  
  // Check for any stale recording state and reset it
  if (extensionState && extensionState.state && extensionState.state.recording) {
    console.log('Found stale recording state, resetting it');
    
    // Reset recording state in the service worker
    chrome.runtime.sendMessage({ 
      action: 'SET_TRANSCRIPTION', 
      transcription: '' // Empty string to clear transcription
    });
    
    // Also reset recording state in storage
    await storageManager.setRecordingInProgress(false);
    
    // Show default recording UI
    isRecording = false;
    showRecordingUI();
  } else if (extensionState && extensionState.state && extensionState.state.transcribing) {
    console.log('Found stale transcribing state, resetting it');
    
    // Reset the state in the service worker
    chrome.runtime.sendMessage({ 
      action: 'SET_TRANSCRIPTION', 
      transcription: '' // Empty string to clear transcription
    });
    
    // Show default recording UI
    isRecording = false;
    showRecordingUI();
  } else if (extensionState && extensionState.state && extensionState.state.lastTranscription) {
    // If we have a previous transcription, show it
    console.log('Found previous transcription, showing it');
    currentTranscription = extensionState.state.lastTranscription;
    showTranscriptionUI(currentTranscription);
    // Wait for TextRad tabs to be detected before loading templates
    setTimeout(() => {
      loadAvailableFormats();
    }, 1000);
  }
  
  // Check extension state
  checkState();
  
  // Start continuous login monitoring
  startLoginMonitoring();
}

/**
 * Get UI elements
 */
function getUIElements() {
  // Main UI elements
  recordButton = document.getElementById('record-button');
  visualizerCanvas = null; // Removing visualizer
  recordingStatus = document.getElementById('recording-status');
  // Timer element removed
  recordingPanel = document.getElementById('recording-panel');
  transcriptionPanel = document.getElementById('transcription-panel');
  transcriptionText = document.getElementById('transcription-text');
  newDictationButton = document.getElementById('new-dictation-button');
  copyButton = document.getElementById('copy-button');
  sendButton = document.getElementById('send-button');
  
  // App status elements
  appStatusPanel = document.getElementById('app-status-panel');
  appStatusIndicator = document.getElementById('app-status-indicator').querySelector('.status-dot');
  appStatusText = document.getElementById('app-status-text');
  openTextRadButton = document.getElementById('open-textrad-button');
  
  // Login required message elements
  loginRequiredMessage = document.getElementById('login-required-message');
  loginButton = document.getElementById('login-button');
  
  // Settings elements
  settingsButton = document.getElementById('settings-button');
  settingsPanel = document.getElementById('settings-panel');
  closeSettingsButton = document.getElementById('close-settings-button');
  saveSettingsButton = document.getElementById('save-settings-button');
  autoSendSetting = document.getElementById('auto-send-setting');
  silenceTimeoutSetting = document.getElementById('silence-timeout-setting');
  silenceTimeoutValue = document.getElementById('silence-timeout-value');
  
  // Loading elements
  loadingOverlay = document.getElementById('loading-overlay');
  loadingText = document.getElementById('loading-text');
  
  // Toast notification elements
  toastNotification = document.getElementById('toast-notification');
  toastMessage = document.getElementById('toast-message');
  
  // Main/Mini UI elements
  expandedContainer = document.getElementById('expanded-container');
  collapsedContainer = document.getElementById('collapsed-container');
  collapseButton = document.getElementById('collapse-button');
  expandButton = document.getElementById('expand-button');
  miniRecordButton = document.getElementById('mini-record-button');
  closeButton = document.getElementById('close-button');
  collapsedStatus = document.getElementById('collapsed-status');

  // New template selection elements
  templateSelectionPanel = document.getElementById('template-selection-panel');
  formatSelector = document.getElementById('format-selector');
  formatSearchInput = document.getElementById('format-search-input');
  formatDropdownList = document.getElementById('format-dropdown-list');
  formatNoResults = document.getElementById('format-no-results');
  templateLoading = document.getElementById('template-loading');
  templateError = document.getElementById('template-error');
  templateErrorMessage = document.getElementById('template-error-message');
  generateReportPanel = document.getElementById('generate-report-panel');
  generateButton = document.getElementById('generate-button');
  generateButtonText = document.getElementById('generate-button-text');
  generationProgress = document.getElementById('generation-progress');
  generationStatus = document.getElementById('generation-status');
}

/**
 * Set up event listeners
 */
function setupEventListeners() {
  // Record button
  recordButton.addEventListener('click', toggleRecording);
  
  // New dictation button
  newDictationButton.addEventListener('click', startNewDictation);
  
  // Copy button
  copyButton.addEventListener('click', copyTranscription);
  
  // Send button
  sendButton.addEventListener('click', sendTranscription);

  // Format selector (searchable dropdown)
  if (formatSearchInput) {
    formatSearchInput.addEventListener('input', handleFormatSearch);
    formatSearchInput.addEventListener('keydown', handleFormatKeydown);
    formatSearchInput.addEventListener('focus', showFormatDropdown);
    formatSearchInput.addEventListener('blur', hideFormatDropdownDelayed);
  }

  // Generate report button
  if (generateButton) {
    generateButton.addEventListener('click', generateReport);
  }
  
  // Open TextRad button
  openTextRadButton.addEventListener('click', openTextRad);
  
  // Login button (if present)
  if (loginButton) {
    loginButton.addEventListener('click', openTextRad);
  }
  
  // Refresh status button
  const refreshButton = document.getElementById('refresh-status-button');
  if (refreshButton) {
    refreshButton.addEventListener('click', () => {
      console.log('Main refresh button clicked - performing aggressive login recheck');
      
      // Force clear all cached state
      textradTabs = [];
      selectedTabId = null;
      lastLoginCheck = 0;
      
      // Reset to not-logged-in state first
      updateAppStatus('not-logged-in');
      
      // Show loading indicator
      showLoading('Refreshing login status...');
      
      // Force a fresh check with delay to ensure caches are cleared
      setTimeout(() => {
        findTextRadTabs();
      }, 500);
    });
  }
  
  // Settings button
  settingsButton.addEventListener('click', toggleSettings);
  
  // Close settings button
  closeSettingsButton.addEventListener('click', toggleSettings);
  
  // Save settings button
  saveSettingsButton.addEventListener('click', saveSettings);
  
  // Silence timeout slider
  silenceTimeoutSetting.addEventListener('input', updateSilenceTimeoutValue);
  
  // Collapse button to switch to mini UI
  collapseButton.addEventListener('click', toggleViewMode);
  
  // Mini record button in mini mode
  miniRecordButton.addEventListener('click', toggleRecording);
  
  // Expand button to go back to full view
  expandButton.addEventListener('click', toggleViewMode);
  
  // Close button - actually closes the popup
  closeButton.addEventListener('click', closePopup);

  // Close dropdown when clicking outside
  document.addEventListener('click', (event) => {
    if (formatDropdownList && !formatDropdownList.classList.contains('hidden')) {
      const formatContainer = document.getElementById('format-selector-container');
      const isClickInsideDropdown = formatContainer && formatContainer.contains(event.target);
      if (!isClickInsideDropdown) {
        hideFormatDropdown();
      }
    }
  });

  // Add beforeunload event listener to handle potential popup closing
  window.addEventListener('beforeunload', handlePopupClosing);
  
  // Add unload event listener to clean up monitoring
  window.addEventListener('unload', () => {
    if (loginCheckInterval) {
      clearInterval(loginCheckInterval);
      loginCheckInterval = null;
    }
  });
  
  // Handle fullscreen/mini toggle via keyboard shortcut (Escape key)
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
      toggleViewMode();
      e.preventDefault();
    }
  });
}

/**
 * Initialize audio recorder and transcription service
 */
function initializeServices() {
  // Initialize audio recorder without visualizer
  audioRecorder = new ExtensionAudioRecorder(null, storageManager);
  
  // Set audio recorder event handlers
  audioRecorder.onRecordingStart = handleRecordingStart;
  audioRecorder.onRecordingStop = handleRecordingStop;
  audioRecorder.onSilenceDetected = handleSilenceDetected;
  audioRecorder.onError = handleRecordingError;
  
  // Initialize transcription service
  transcriptionService = new ExtensionTranscriptionService(storageManager);
  
  // Set transcription service event handlers
  transcriptionService.onTranscriptionStart = handleTranscriptionStart;
  transcriptionService.onTranscriptionComplete = handleTranscriptionComplete;
  transcriptionService.onTranscriptionError = handleTranscriptionError;
  
  // Initialize Firebase services
  transcriptionService.initializeFirebaseServices();
}

/**
 * Start continuous login monitoring
 */
function startLoginMonitoring() {
  // Clear any existing interval
  if (loginCheckInterval) {
    clearInterval(loginCheckInterval);
  }
  
  // Check login status every 10 seconds (reduced frequency to prevent interference)
  loginCheckInterval = setInterval(() => {
    // Only check if we're not currently performing a login check
    const now = Date.now();
    if (now - lastLoginCheck > 8000) { // Prevent rapid fire checks (increased from 3s to 8s)
      console.log('Periodic login check...');
      findTextRadTabs();
    }
    
    // Also ensure login message is visible if user is not logged in
    // BUT don't interfere if we have active transcription or user is currently using the extension
    if (!isUserLoggedIn()) {
      const loginMsg = document.getElementById('login-required-message');
      const transcriptionPanel = document.getElementById('transcription-panel');
      const hasActiveTranscription = transcriptionPanel && !transcriptionPanel.classList.contains('hidden');
      
      // Only force login message if no active transcription is shown
      if (loginMsg && loginMsg.classList.contains('hidden') && !hasActiveTranscription) {
        console.log('User not logged in but login message is hidden - fixing...');
        updateAppStatus('not-logged-in');
      }
    }
  }, 10000); // Reduced frequency from 5s to 10s
  
  // Also check when popup gains focus
  window.addEventListener('focus', () => {
    console.log('Popup gained focus, checking login status...');
    findTextRadTabs();
  });
  
  // Check when popup becomes visible
  document.addEventListener('visibilitychange', () => {
    if (!document.hidden) {
      console.log('Popup became visible, checking login status...');
      findTextRadTabs();
    }
  });
}

/**
 * Check for TextRad tabs (with forced refresh)
 */
function findTextRadTabs() {
  // Update last check time
  lastLoginCheck = Date.now();
  
  // Clear cached tabs to force fresh login check
  textradTabs = [];
  selectedTabId = null;
  
  updateAppStatus('connecting');
  
  // Show loading while checking
  showLoading('Checking TextRad connection...');
  
  // Set a timeout to hide the loading indicator if it takes too long
  const loadingTimeout = setTimeout(() => {
    console.log('Loading timeout reached, hiding loading indicator');
    hideLoading();
  }, 5000); // 5 seconds timeout
  
  chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
    // Always hide loading indicator
    hideLoading();
    
    // Clear the timeout
    clearTimeout(loadingTimeout);
    
    if (chrome.runtime.lastError) {
      console.error('Error finding TextRad tabs:', chrome.runtime.lastError);
      updateAppStatus('disconnected');
      return;
    }
    
    if (response && response.success) {
      // Check if there's a login error message
      if (response.error && response.error.includes('not logged in')) {
        console.log('User not logged in to TextRad');
        textradTabs = [];
        updateAppStatus('not-logged-in');
        return;
      }
      
      if (response.tabs && response.tabs.length > 0) {
        // Store tabs with login information
        textradTabs = response.tabs.map(tab => ({
          ...tab,
          loggedIn: response.loggedInTabs && response.loggedInTabs[tab.id] === true
        }));
        
        // Check if we have any logged-in tabs
        const loggedInTabs = textradTabs.filter(tab => tab.loggedIn);
        
        if (loggedInTabs.length > 0) {
          console.log(`Found ${loggedInTabs.length} logged-in TextRad tabs`);
          console.log('Tab details:', loggedInTabs.map(tab => ({ id: tab.id, url: tab.url, title: tab.title })));
          selectedTabId = loggedInTabs[0].id;
          console.log('Selected tab ID for template loading:', selectedTabId);
          updateAppStatus('connected');
        } else {
          // We found TextRad tabs, but user isn't logged in
          console.log('TextRad tabs found but user not logged in');
          textradTabs = [];
          updateAppStatus('not-logged-in');
        }
      } else {
        // No TextRad tabs found
        textradTabs = [];
        if (response.hasTextRadTabs && response.notLoggedInCount > 0) {
          // We found TextRad tabs, but user isn't logged in
          console.log('TextRad tabs found but user not logged in');
          updateAppStatus('not-logged-in');
        } else {
          // No TextRad tabs at all
          console.log('No TextRad tabs found');
          updateAppStatus('disconnected');
        }
      }
    } else {
      console.log('Find TextRad tabs failed or returned no success');
      textradTabs = [];
      updateAppStatus('disconnected');
    }
  });
}

/**
 * Update app connection status
 */
function updateAppStatus(status) {
  // Ensure UI elements are available
  if (!appStatusIndicator) {
    appStatusIndicator = document.getElementById('app-status-indicator')?.querySelector('.status-dot');
  }
  if (!appStatusText) {
    appStatusText = document.getElementById('app-status-text');
  }
  
  if (appStatusIndicator) {
    appStatusIndicator.className = 'status-dot';
  }
  
  // Ensure elements exist before accessing them
  // Hide login required message by default (except for not-logged-in status)
  const msgElement = document.getElementById('login-required-message');
  if (msgElement) {
    if (status !== 'not-logged-in') {
      msgElement.classList.add('hidden');
    }
    loginRequiredMessage = msgElement; // Ensure the variable is assigned
  }
  
  // Ensure login button is defined
  if (!loginButton) {
    loginButton = document.getElementById('login-button');
  }
  
  // Disable record buttons by default until we confirm login status
  if (recordButton) {
    recordButton.disabled = true;
  }
  if (miniRecordButton) {
    miniRecordButton.disabled = true;
  }
  
  switch (status) {
    case 'connected':
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('connected');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Connected to TextRad';
      }
      // Enable recording when connected and logged in
      if (recordButton) {
        recordButton.disabled = false;
        recordButton.classList.remove('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = false;
        miniRecordButton.classList.remove('disabled');
      }
      break;
      
    case 'connecting':
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('connecting');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Connecting...';
      }
      break;
      
    case 'not-logged-in':
      // Special case - user needs to log in
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('disconnected');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Login required';
      }
      
      // Always show login message prominently
      const loginMsg = document.getElementById('login-required-message');
      if (loginMsg) {
        loginMsg.classList.remove('hidden');
        loginRequiredMessage = loginMsg; // Ensure the variable is assigned
        
        // Make sure it's visible and prominent
        loginMsg.style.display = 'block !important';
        loginMsg.style.visibility = 'visible';
        loginMsg.style.opacity = '1';
        
        // Force it to be at the top of the UI
        const recordingPanel = document.getElementById('recording-panel');
        if (recordingPanel && recordingPanel.parentNode) {
          recordingPanel.parentNode.insertBefore(loginMsg, recordingPanel);
        }
        
        // Scroll into view
        setTimeout(() => {
          loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }, 100);
      }
      
      // Make sure recording UI is fully visible but disabled
      const recordingPanelElement = recordingPanel || document.getElementById('recording-panel');
      if (recordingPanelElement) {
        recordingPanelElement.classList.remove('hidden');
      }
      
      const transcriptionPanelElement = transcriptionPanel || document.getElementById('transcription-panel');
      if (transcriptionPanelElement) {
        // Only hide transcription if it doesn't have content or if this is an explicit logout
        const transcriptionText = document.getElementById('transcription-text');
        const hasContent = transcriptionText && transcriptionText.value && transcriptionText.value.trim().length > 0;
        
        // Don't hide transcription panel if user has content unless it's a fresh login check
        if (!hasContent) {
          transcriptionPanelElement.classList.add('hidden');
        }
      }
      
      // Ensure login button is set up
      const loginBtn = document.getElementById('login-button');
      if (loginBtn && !loginBtn.hasEventListener) {
        loginBtn.addEventListener('click', openTextRad);
        loginBtn.hasEventListener = true;
        loginButton = loginBtn;
      }
      
      // Ensure refresh button is set up
      const refreshBtn = document.getElementById('refresh-status-button');
      if (refreshBtn && !refreshBtn.hasEventListener) {
        refreshBtn.addEventListener('click', () => {
          console.log('Refresh button clicked - performing aggressive login recheck');
          
          // Force clear all cached state
          textradTabs = [];
          selectedTabId = null;
          lastLoginCheck = 0;
          
          // Reset to not-logged-in state first
          updateAppStatus('not-logged-in');
          
          showLoading('Refreshing login status...');
          
          // Force a fresh check with delay to ensure caches are cleared
          setTimeout(() => {
            findTextRadTabs();
          }, 500);
        });
        refreshBtn.hasEventListener = true;
      }
      
      // Disable recording when not logged in
      if (recordButton) {
        recordButton.disabled = true;
        recordButton.classList.add('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = true;
        miniRecordButton.classList.add('disabled');
      }
      break;
      
    case 'disconnected':
    default:
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('disconnected');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Not connected';
      }
      // Disable recording when disconnected
      if (recordButton) {
        recordButton.disabled = true;
        recordButton.classList.add('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = true;
        miniRecordButton.classList.add('disabled');
      }
      break;
  }
  
  // Update mini mode status
  updateMiniStatus();
}

/**
 * Open TextRad web app
 */
function openTextRad() {
  // Open TextRad in a new tab (use root URL to ensure login page is shown if needed)
  chrome.tabs.create({ url: 'https://textrad.in/' });
  
  // Show a helpful message to the user
  const msgElement = document.getElementById('login-required-message');
  if (msgElement) {
    if (!msgElement.querySelector('.login-instruction')) {
      const instruction = document.createElement('p');
      instruction.className = 'login-instruction mt-1 text-green-300';
      instruction.textContent = 'TextRad opened in a new tab. Please log in there and then try again.';
      msgElement.querySelector('div').appendChild(instruction);
    }
  }
}

/**
 * Copy transcription to clipboard
 */
function copyTranscription() {
  // Check if we have a transcription
  if (!transcriptionText.value) {
    alert('No transcription to copy');
    return;
  }

  // Ensure currentTranscription is updated with the textarea value
  // This ensures send will work even after copy
  currentTranscription = transcriptionText.value;

  // Determine if we're copying a generated report
  const isCopyingReport = currentReportContent !== null;

  // Copy to clipboard
  navigator.clipboard.writeText(transcriptionText.value).then(() => {
    // Show success message
    const originalText = copyButton.innerHTML;
    copyButton.innerHTML = '<i class="fas fa-check mr-1"></i>Copied!';

    // Enhanced feedback for report copying
    if (isCopyingReport) {
      showToast('Report copied to clipboard! Ready for new dictation.', 3000);

      // Auto-reset after copying a generated report (with slight delay for better UX)
      setTimeout(() => {
        startNewDictation();
      }, 1500);
    } else {
      showToast('Transcription copied to clipboard!', 2000);
    }

    // Reset button text after 2 seconds
    setTimeout(() => {
      copyButton.innerHTML = originalText;
    }, 2000);
  }).catch(err => {
    console.error('Could not copy text: ', err);
    alert('Failed to copy: ' + err.message);
  });
}

/**
 * Check extension state
 */
async function checkState() {
  chrome.runtime.sendMessage({ action: 'GET_STATE' }, async (response) => {
    if (chrome.runtime.lastError) {
      console.error('Error getting state:', chrome.runtime.lastError);
      return;
    }

    if (response && response.state) {
      const state = response.state;

      // Check recording state
      if (state.recording) {
        // We were recording, but popup was closed
        showRecordingUI();
      } else if (state.transcribing) {
        // We were transcribing
        showTranscribingUI();
      } else if (state.lastTranscription) {
        // We have a transcription
        showTranscriptionUI(state.lastTranscription);
        // Wait for TextRad tabs to be detected before loading templates
        setTimeout(() => {
          loadAvailableFormats();
        }, 1000);
      }
    }
  });
}

/**
 * Toggle recording state
 */
function toggleRecording() {
  // First check if we're connected to TextRad and logged in
  if (!isUserLoggedIn()) {
    console.log('Cannot record: User not logged in to TextRad');
    updateAppStatus('not-logged-in');
    
    // Show prominent message to user
    showToast('Please log in to TextRad first', 3000);
    
    // Ensure login message is visible
    const loginMsg = document.getElementById('login-required-message');
    if (loginMsg) {
      loginMsg.classList.remove('hidden');
      loginMsg.style.display = 'block';
      loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    return;
  }
  
  if (isRecording) {
    stopRecording();
  } else {
    // For the main record button, we want to append to existing text
    // Set the flag to indicate we want to append
    recordMoreButtonClicked = true;
    
    // Store current transcription before starting new recording
    if (transcriptionText && transcriptionText.value) {
      const previousTranscription = transcriptionText.value;
      currentTranscription = previousTranscription;
      console.log('Main record button clicked - storing transcription for append:', 
                previousTranscription.substring(0, 50) + '...');
    }
    
    startRecording();
  }
}

/**
 * Check if user is logged in to TextRad
 */
function isUserLoggedIn() {
  // If we have textradTabs with explicitly logged-in status, then the user is logged in
  const hasLoggedInTabs = textradTabs && textradTabs.length > 0 && 
                         textradTabs.some(tab => tab.loggedIn === true);
  
  // Additional check: if we're currently showing transcription content, 
  // don't consider user as logged out (prevents flickering)
  if (!hasLoggedInTabs) {
    const transcriptionText = document.getElementById('transcription-text');
    const hasActiveContent = transcriptionText && transcriptionText.value && transcriptionText.value.trim().length > 0;
    const transcriptionPanel = document.getElementById('transcription-panel');
    const isTranscriptionVisible = transcriptionPanel && !transcriptionPanel.classList.contains('hidden');
    
    // If user has active transcription content visible, assume they're logged in to prevent UI disruption
    if (hasActiveContent && isTranscriptionVisible) {
      console.log('User has active transcription content, assuming logged in to prevent UI disruption');
      return true;
    }
  }
  
  return hasLoggedInTabs;
}

/**
 * Start recording (behavior depends on which button initiated it)
 */
async function startRecording() {
  try {
    // Perform fresh login check right before recording
    console.log('Checking login status before starting recording...');
    
    // Force a fresh login check
    const now = Date.now();
    lastLoginCheck = now;
    
    const loginCheckResponse = await new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
        resolve(response);
      });
    });
    
    // Validate login response
    if (!loginCheckResponse || !loginCheckResponse.success || 
        !loginCheckResponse.tabs || loginCheckResponse.tabs.length === 0) {
      updateAppStatus('not-logged-in');
      throw new Error('You must be logged in to TextRad to record and transcribe audio');
    }
    
    // Check if user is actually logged in to any tabs
    const hasLoggedInTabs = loginCheckResponse.tabs.some(tab => 
      loginCheckResponse.loggedInTabs && loginCheckResponse.loggedInTabs[tab.id] === true
    );
    
    if (!hasLoggedInTabs) {
      updateAppStatus('not-logged-in');
      throw new Error('You must be logged in to TextRad to record and transcribe audio');
    }
    
    // Update tabs with fresh login data
    textradTabs = loginCheckResponse.tabs.map(tab => ({
      ...tab,
      loggedIn: loginCheckResponse.loggedInTabs && loginCheckResponse.loggedInTabs[tab.id] === true
    }));
    
    console.log('Login check passed, proceeding with recording');
    
    // Update UI
    recordButton.disabled = true;
    recordingStatus.textContent = 'Starting...';
    
    // The recordMoreButtonClicked flag is set in toggleRecording (normal record button)
    // and explicitly set to false in startNewDictation
    // So we don't need to do anything else with the flag here
    console.log('Starting recording with append mode:', recordMoreButtonClicked ? 'true' : 'false');
    
    /* 
     * Background recording has been disabled - we are not using it anymore
     * Only use direct recording through the audioRecorder
     */
    
    // If background recording fails, fall back to local recording
    console.log('Falling back to local audio recording');
    const success = await audioRecorder.startRecording();
    
    if (!success) {
      throw new Error('Failed to start recording');
    }
  } catch (error) {
    console.error('Error starting recording:', error);
    
    // If this is a login error, update UI appropriately
    if (error.message.includes('logged in')) {
      updateAppStatus('not-logged-in');
    } else {
      recordingStatus.textContent = 'Error: ' + error.message;
    }
    
    recordButton.disabled = false;
  }
}

/**
 * Send a message with timeout
 */
function sendMessageWithTimeout(message, timeout = 3000) {
  return new Promise((resolve, reject) => {
    const timer = setTimeout(() => {
      reject(new Error(`Message timeout after ${timeout}ms`));
    }, timeout);
    
    chrome.runtime.sendMessage(message, (response) => {
      clearTimeout(timer);
      if (chrome.runtime.lastError) {
        reject(chrome.runtime.lastError);
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * Stop recording
 */
async function stopRecording() {
  try {
    // Update UI
    recordButton.disabled = true;
    recordingStatus.textContent = 'Stopping...';
    
    /* 
     * Background recording has been disabled - we are not using it anymore
     * Only use direct recording through the audioRecorder
     */
    
    // Fall back to local recording if background recording fails
    console.log('Falling back to local audio recorder stop');
    audioRecorder.stopRecording();
    
    // Ensure recording in progress flag is cleared
    if (storageManager) {
      storageManager.setRecordingInProgress(false);
    }
  } catch (error) {
    console.error('Error stopping recording:', error);
    recordingStatus.textContent = 'Error: ' + error.message;
    recordButton.disabled = false;
  }
}

/**
 * Start a brand new dictation (clear previous text)
 */
function startNewDictation() {
  // First check if we're connected to TextRad and logged in
  if (!isUserLoggedIn()) {
    console.log('Cannot start new dictation: User not logged in to TextRad');
    updateAppStatus('not-logged-in');
    
    // Show prominent message to user
    showToast('Please log in to TextRad first', 3000);
    
    // Ensure login message is visible
    const loginMsg = document.getElementById('login-required-message');
    if (loginMsg) {
      loginMsg.classList.remove('hidden');
      loginMsg.style.display = 'block';
      loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }
    
    return;
  }
  
  // EXPLICITLY Reset flag to indicate we're starting fresh
  recordMoreButtonClicked = false;
  
  // Clear the current transcription (local and in services)
  // IMPORTANT: Set both to empty strings to ensure complete clearing
  currentTranscription = '';

  // Reset template state
  resetTemplateState();
  
  // Reset transcription in all services
  transcriptionService.resetTranscription();
  transcriptionService.setCurrentTranscription('');
  
  // Reset the DirectTranscriptionService if it's being imported
  try {
    // This is an async import but we'll just fire it and let it complete
    import('../lib/direct-transcription-service.js').then(module => {
      const DirectTranscriptionService = module.DirectTranscriptionService;
      const directService = new DirectTranscriptionService();
      directService.resetTranscription();
      directService.setCurrentTranscription('');
    }).catch(err => {
      console.error('Error resetting DirectTranscriptionService:', err);
    });
  } catch (err) {
    console.error('Error importing DirectTranscriptionService:', err);
  }
  
  // Explicitly clear the UI text area
  if (transcriptionText) {
    transcriptionText.value = '';
  }
  
  // Also reset the stored value in the service worker and disable append mode
  chrome.runtime.sendMessage({
    action: 'NEW_DICTATION'
  });
  
  console.log('New Dictation clicked - cleared previous transcription');
  
  // Hide transcription panel
  transcriptionPanel.classList.add('hidden');
  
  // Show recording panel
  document.getElementById('recording-panel').classList.remove('hidden');
  
  // Make sure the full UI is shown before starting a new recording
  const fullRecordingUI = document.getElementById('full-recording-ui');
  if (fullRecordingUI) {
    fullRecordingUI.classList.remove('hidden');
  }
  appStatusPanel.classList.remove('hidden');
  
  // DO NOT start recording automatically - let user press the mic button
  // Just reset the UI to recording-ready state
  recordingStatus.textContent = 'Ready to record';
  recordButton.disabled = false;
  recordButton.classList.remove('recording');
}

/**
 * Send transcription to TextRad
 */
function sendTranscription() {
  // Check if we have a transcription
  if (!currentTranscription) {
    alert('No transcription to send');
    return;
  }
  
  // Check the connection state first
  updateAppStatus('connecting');
  
  // Find TextRad tabs with verified login state
  chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
    if (chrome.runtime.lastError) {
      console.error('Error finding TextRad tabs:', chrome.runtime.lastError);
      hideLoading();
      updateAppStatus('disconnected');
      alert('Error connecting to TextRad: ' + chrome.runtime.lastError.message);
      return;
    }
    
    // Check for login issues
    if (response && response.error && response.error.includes('not logged in')) {
      console.log('User not logged in to TextRad');
      hideLoading();
      updateAppStatus('not-logged-in');
      return;
    }
    
    // Check if we have logged-in TextRad tabs
    if (response && response.success && response.tabs && response.tabs.length > 0) {
      // Store tabs with login information
      textradTabs = response.tabs.map(tab => ({
        ...tab,
        loggedIn: response.loggedInTabs && response.loggedInTabs[tab.id] === true
      }));
      
      // Only use logged-in tabs
      const loggedInTabs = textradTabs.filter(tab => tab.loggedIn);
      if (loggedInTabs.length > 0) {
        selectedTabId = loggedInTabs[0].id;
        updateAppStatus('connected');
        
        // Show loading state
        showLoading('Sending to TextRad...');
        
        // Send transcription to TextRad tab
        chrome.runtime.sendMessage({
          action: 'SEND_TRANSCRIPTION_TO_TAB',
          tabId: selectedTabId,
          transcription: currentTranscription
        }, (response) => {
        // Hide loading state
        hideLoading();
        
        if (chrome.runtime.lastError) {
          console.error('Error sending transcription:', chrome.runtime.lastError);
          showToast('Error: ' + chrome.runtime.lastError.message, 3000);
          return;
        }
        
        if (response && response.success) {
          // Success - show subtle toast notification
          showToast('✓ Sent to TextRad', 1500);
          
          // Update button to show temporary success state
          const originalHTML = sendButton.innerHTML;
          sendButton.innerHTML = '<i class="fas fa-check"></i>';
          sendButton.classList.add('bg-green-700');
          
          // Reset button after a delay
          setTimeout(() => {
            sendButton.innerHTML = originalHTML;
            sendButton.classList.remove('bg-green-700');
          }, 1500);
        } else if (response && response.error && response.error.includes('not logged in')) {
          // Login error
          updateAppStatus('not-logged-in');
        } else {
          // Other error
          showToast('Error: ' + (response?.error || 'Unknown error'), 3000);
        }
        });
      } else {
        // We found TextRad tabs, but user isn't logged in to any
        hideLoading();
        updateAppStatus('not-logged-in');
      }
    } else {
      // No TextRad tabs found or not logged in
      hideLoading();
      
      if (response && response.hasTextRadTabs && response.notLoggedInCount > 0) {
        // Found TextRad tabs but user isn't logged in
        updateAppStatus('not-logged-in');
      } else {
        // No TextRad tabs at all
        updateAppStatus('disconnected');
        
        // Ask if user wants to open TextRad
        if (confirm('No TextRad app detected. Open TextRad in a new tab?')) {
          openTextRad();
        }
      }
    }
  });
}

/**
 * Handle recording start
 */
function handleRecordingStart() {
  console.log('Recording started');
  
  // Update state
  isRecording = true;
  recordingStartTime = Date.now();
  
  // Update UI
  showRecordingUI();
  
  // Start recording timer
  startRecordingTimer();
}

/**
 * Handle recording stop
 */
function handleRecordingStop(audioBlob) {
  console.log('Recording stopped, blob size:', audioBlob.size);
  
  // Update state
  isRecording = false;
  
  // Stop recording timer
  stopRecordingTimer();
  
  // Update UI
  showTranscribingUI();
  
  // Check login status again before transcribing
  if (!isUserLoggedIn()) {
    console.log('Cannot transcribe: User not logged in to TextRad');
    handleTranscriptionError(new Error('You must be logged in to TextRad to transcribe audio'));
    updateAppStatus('not-logged-in');
    return;
  }
  
  // Start transcription
  if (audioBlob && audioBlob.size > 0) {
    transcriptionService.transcribeAudio(audioBlob)
      .catch(error => {
        // Extra error handling for login issues
        if (error.message.includes('logged in') || error.message.includes('login')) {
          updateAppStatus('not-logged-in');
        }
      });
  } else {
    handleTranscriptionError(new Error('No audio recorded'));
  }
}

/**
 * Handle silence detected
 */
function handleSilenceDetected() {
  console.log('Silence detected, auto-stopping recording');
  
  // Update UI
  recordingStatus.textContent = 'Silence detected, stopping...';
  
  // We don't need to call stopRecording() here as the audioRecorder
  // will automatically call onRecordingStop which we handle above
}

/**
 * Handle recording error
 */
function handleRecordingError(error) {
  console.error('Recording error:', error);
  
  // Update state
  isRecording = false;
  
  // Stop recording timer
  stopRecordingTimer();
  
  // Update UI
  recordingStatus.textContent = 'Error: ' + error.message;
  recordButton.disabled = false;
  recordButton.classList.remove('recording');
}

/**
 * Handle transcription start
 */
function handleTranscriptionStart() {
  console.log('Transcription started');
  
  // Update state
  isTranscribing = true;
  
  // Update UI
  recordingStatus.textContent = 'Transcribing...';
}

/**
 * Handle transcription complete
 */
function handleTranscriptionComplete(text) {
  console.log('Transcription complete:', text.substring(0, 50) + '...');
  
  // Update state
  isTranscribing = false;
  
  // Check if we need to append to previous transcription
  let finalText = text;
  const previousTranscription = currentTranscription || '';
  
  console.log('recordMoreButtonClicked flag:', recordMoreButtonClicked);
  console.log('Has previous transcription:', previousTranscription.trim() !== '');
  
  // Only append if we explicitly stored a previous transcription AND the recordMoreButtonClicked flag is true
  if (recordMoreButtonClicked === true && previousTranscription && previousTranscription.trim() !== '') {
    // Append new transcription to the previous one with a space in between
    finalText = previousTranscription + ' ' + text;
    console.log('Appended to previous transcription, result:', finalText.substring(0, 50) + '...');
  } else {
    console.log('Not appending - using just the new transcription');
    // When recordMoreButtonClicked is explicitly false (set by New Dictation), 
    // make sure we use only the new transcription
    finalText = text;
    currentTranscription = '';
  }
  
  // Always reset the recordMoreButtonClicked flag after use
  recordMoreButtonClicked = false;
  
  // Update current transcription
  currentTranscription = finalText;
  
  // Reset the service's stored transcription to avoid duplication
  transcriptionService.setCurrentTranscription('');
  
  // Update UI
  showTranscriptionUI(finalText);

  // Load templates after transcription is complete
  loadAvailableFormats();

  // Notify background script
  chrome.runtime.sendMessage({
    action: 'SET_TRANSCRIPTION',
    transcription: finalText
  });
  
  // Check if auto-send is enabled
  if (autoSendSetting.checked && textradTabs.length > 0) {
    sendTranscription();
  }
}

/**
 * Handle transcription error
 */
function handleTranscriptionError(error) {
  console.error('Transcription error:', error);
  
  // Update state
  isTranscribing = false;
  
  // Different UI handling based on error type
  if (error.message && (
      error.message.includes('audio') || 
      error.message.includes('speech') || 
      error.message.includes('short') ||
      error.message.includes('empty') ||
      error.message.includes('recording'))) {
    // Audio-related error - don't show login prompt
    recordingStatus.textContent = 'Recording issue: ' + error.message;
  } else if (error.message && error.message.includes('timed out')) {
    // Timeout error
    recordingStatus.textContent = 'Timeout: ' + error.message;
  } else if (error.message && (
      error.message.includes('login') || 
      error.message.includes('logged in'))) {
    // Login-related error - update login status
    recordingStatus.textContent = 'Login required';
    updateAppStatus('not-logged-in');
  } else {
    // Generic error
    recordingStatus.textContent = 'Transcription error: ' + error.message;
  }
  
  recordButton.disabled = false;
  hideLoading();
}

/**
 * Start recording timer (simplified, just tracks start time)
 */
function startRecordingTimer() {
  // We still track recording start time for other purposes
  // but we no longer display it
  recordingStartTime = Date.now();
}

/**
 * Stop recording timer
 */
function stopRecordingTimer() {
  if (recordingTimer) {
    clearInterval(recordingTimer);
    recordingTimer = null;
  }
}

/**
 * Show recording UI - Simple approach that always works
 */
function showRecordingUI() {
  // Update record button
  recordButton.disabled = false;
  recordButton.classList.add('recording');
  recordButton.querySelector('i').className = 'fas fa-stop';
  
  // Update status
  recordingStatus.textContent = 'Recording...';
  
  // Hide transcription panel
  transcriptionPanel.classList.add('hidden');
  
  // Ensure recording panel is visible
  document.getElementById('recording-panel').classList.remove('hidden');
  
  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show transcribing UI
 */
function showTranscribingUI() {
  // Update record button
  recordButton.disabled = true;
  recordButton.classList.remove('recording');
  recordButton.querySelector('i').className = 'fas fa-microphone';
  
  // Update status
  recordingStatus.textContent = 'Transcribing...';
  
  // Show loading
  showLoading('Transcribing audio...');
  
  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show transcription UI
 */
function showTranscriptionUI(text) {
  // Hide loading
  hideLoading();
  
  // Update record button
  recordButton.disabled = false;
  recordButton.classList.remove('recording');
  recordButton.querySelector('i').className = 'fas fa-microphone';
  
  // Update status
  recordingStatus.textContent = 'Ready to record';
  
  // Button is already in its fixed position - no need to move it
  
  // Show recording panel again
  document.getElementById('recording-panel').classList.remove('hidden');
  
  // Show other UI elements
  appStatusPanel.classList.remove('hidden');
  settingsButton.classList.remove('hidden');
  
  // Update transcription text
  transcriptionText.value = text;
  
  // Show transcription panel
  transcriptionPanel.classList.remove('hidden');
  updateContainerState();
  
  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show loading overlay
 */
function showLoading(message) {
  loadingText.textContent = message || 'Processing...';
  loadingOverlay.classList.remove('hidden');
}

/**
 * Hide loading overlay
 */
function hideLoading() {
  loadingOverlay.classList.add('hidden');
}

/**
 * Show toast notification
 * @param {string} message - Message to display
 * @param {number} duration - How long to show in ms (default 2000ms)
 */
function showToast(message, duration = 2000) {
  if (!toastNotification || !toastMessage) return;
  
  // Set message
  toastMessage.textContent = message;
  
  // Show toast
  toastNotification.classList.remove('hidden');
  
  // Use setTimeout to ensure the transition works
  setTimeout(() => {
    toastNotification.classList.add('show');
  }, 10);
  
  // Auto hide after duration
  setTimeout(() => {
    // Start fade out
    toastNotification.classList.remove('show');
    toastNotification.classList.add('hide');
    
    // Actually hide after transition completes
    setTimeout(() => {
      toastNotification.classList.remove('hide');
      toastNotification.classList.add('hidden');
    }, 300);
  }, duration);
}

/**
 * Toggle settings panel
 */
function toggleSettings() {
  if (settingsPanel.classList.contains('hidden')) {
    // Show settings
    settingsPanel.classList.remove('hidden');
  } else {
    // Hide settings
    settingsPanel.classList.add('hidden');
  }
}

/**
 * Toggle between expanded and mini modes
 */
function toggleViewMode() {
  // Check current state
  const isExpanded = !expandedContainer.classList.contains('hidden');
  
  if (isExpanded) {
    // Switch to mini mode
    expandedContainer.classList.add('hidden');
    collapsedContainer.classList.remove('hidden');
    document.body.classList.add('mini-mode');
    document.documentElement.classList.add('mini-mode');
    
    // Also notify the background service worker that we're in mini mode
    chrome.runtime.sendMessage({
      action: 'SET_VIEW_MODE',
      mode: 'mini'
    });
    
    // Save in local storage to restore state if popup is closed
    chrome.storage.local.set({ viewMode: 'mini' });
    
    // Force a resize to minimum size
    window.resizeTo(150, 40);
  } else {
    // Switch to expanded mode
    expandedContainer.classList.remove('hidden');
    collapsedContainer.classList.add('hidden');
    document.body.classList.remove('mini-mode');
    document.documentElement.classList.remove('mini-mode');
    
    // Also notify the background service worker that we're in expanded mode
    chrome.runtime.sendMessage({
      action: 'SET_VIEW_MODE',
      mode: 'expanded'
    });
    
    // Save in local storage
    chrome.storage.local.set({ viewMode: 'expanded' });
    
    // Restore normal size
    window.resizeTo(450, 350);
  }
  
  // Update mini mode status indicator based on current state
  updateMiniStatus();
}

/**
 * Handle popup closing
 */
function handlePopupClosing(event) {
  console.log('Popup is closing (beforeunload event)');
  
  // If recording is in progress, we need to do something
  if (isRecording) {
    console.log('Recording was in progress - notifying service worker');
    
    // Notify the background script to maintain recording state
    try {
      chrome.runtime.sendMessage({ 
        action: 'MAINTAIN_RECORDING',
        inProgress: true
      });
    } catch (error) {
      console.error('Failed to notify background script about recording:', error);
    }
  }
}

/**
 * Actually close the popup (when close button is clicked)
 */
function closePopup() {
  // Explicitly stop any recording first
  if (isRecording) {
    stopRecording();
  }
  
  // Then close the window
  window.close();
}

/**
 * Update mini mode status indicator based on current state
 */
function updateMiniStatus() {
  // Remove all classes first
  collapsedStatus.className = 'status-dot';
  
  if (isRecording) {
    // If recording, show red blinking status
    collapsedStatus.classList.add('recording');
    
    // Also update button
    miniRecordButton.classList.add('recording');
    miniRecordButton.querySelector('i').className = 'fas fa-stop';
  } else if (isTranscribing) {
    // If transcribing, show blue blinking status
    collapsedStatus.classList.add('transcribing');
    miniRecordButton.querySelector('i').className = 'fas fa-microphone';
    miniRecordButton.classList.remove('recording');
  } else {
    // Normal state - just show connection status
    miniRecordButton.querySelector('i').className = 'fas fa-microphone';
    miniRecordButton.classList.remove('recording');
    
    // Use the same status as in the app status panel
    if (appStatusIndicator && appStatusIndicator.classList.contains('connected')) {
      collapsedStatus.classList.add('connected');
    } else if (appStatusIndicator && appStatusIndicator.classList.contains('connecting')) {
      collapsedStatus.classList.add('connecting');
    } else {
      collapsedStatus.classList.add('disconnected');
    }
  }
}

/**
 * Update silence timeout value display
 */
function updateSilenceTimeoutValue() {
  const value = silenceTimeoutSetting.value;
  silenceTimeoutValue.textContent = `${value}s`;
}

// Removed transparency functions - using fixed opacity

/**
 * Load settings from storage
 */
async function loadSettings() {
  const settings = await storageManager.getSettings();
  
  // Apply settings
  autoSendSetting.checked = settings.autoSend;
  silenceTimeoutSetting.value = Math.floor(settings.silenceTimeout / 1000);
  
  // Update displayed values
  updateSilenceTimeoutValue();
  
  // Apply settings to services
  if (audioRecorder) {
    audioRecorder.silenceTimeout = settings.silenceTimeout;
  }
}

/**
 * Save settings to storage
 */
async function saveSettings() {
  // Get settings from UI
  const settings = {
    autoSend: autoSendSetting.checked,
    silenceTimeout: parseInt(silenceTimeoutSetting.value) * 1000
  };
  
  // Save settings
  await storageManager.setSettings(settings);
  
  // Fixed transparency - no need to apply anything
  
  // Apply settings to services
  if (audioRecorder) {
    audioRecorder.silenceTimeout = settings.silenceTimeout;
  }
  
  // Hide settings panel
  settingsPanel.classList.add('hidden');
  
  // Show confirmation
  alert('Settings saved');
}

/**
 * Load available formats from TextRad web app
 */
async function loadAvailableFormats() {
  try {
    console.log('Loading available formats...', { selectedTabId, textradTabs: textradTabs.length });

    // Show loading state
    if (templateLoading) {
      templateLoading.classList.remove('hidden');
    }
    if (templateError) {
      templateError.classList.add('hidden');
    }

    // Ensure we have a valid tab ID
    if (!selectedTabId && textradTabs.length > 0) {
      selectedTabId = textradTabs[0].id;
      console.log('Using first available TextRad tab:', selectedTabId);
    }

    if (!selectedTabId) {
      throw new Error('No TextRad tab available. Please ensure TextRad web app is open and you are logged in.');
    }

    console.log('Sending LOAD_FORMATS message to tab:', selectedTabId);

    // Load formats via background script
    const response = await chrome.runtime.sendMessage({
      action: 'LOAD_FORMATS',
      tabId: selectedTabId
    });

    if (response && response.success) {
      availableFormats = response.formats;
      populateFormatSelector(response.formats);
      console.log('Loaded', response.formats.length, 'formats');

      // Show template selection panel
      if (templateSelectionPanel) {
        templateSelectionPanel.classList.remove('hidden');
      }
    } else {
      throw new Error(response?.error || 'Failed to load formats');
    }
  } catch (error) {
    console.error('Error loading formats:', error);
    showTemplateError(error.message);
  } finally {
    // Hide loading state
    if (templateLoading) {
      templateLoading.classList.add('hidden');
    }
  }
}

/**
 * Populate the format selector dropdown
 */
function populateFormatSelector(formats) {
  if (!formatSearchInput || !formatDropdownList) return;

  // Store available formats
  availableFormats = formats;
  filteredFormats = [...formats];

  // Clear search input
  formatSearchInput.value = '';
  formatSearchInput.placeholder = `Type to search ${formats.length} formats...`;

  // Clear hidden selector value
  if (formatSelector) {
    formatSelector.value = '';
  }

  // Render dropdown items
  renderFormatDropdown();

  console.log('Populated format selector with', formats.length, 'formats');
}

/**
 * Handle format selection change (for direct programmatic selection)
 */
function handleFormatSelection(formatId = null) {
  if (!formatId && formatSelector) {
    formatId = formatSelector.value;
  }

  if (formatId) {
    // Find the selected format
    selectedFormat = availableFormats.find(f => f.id === formatId);

    if (selectedFormat) {
      console.log('Format selected:', selectedFormat.name);

      // Update UI
      formatSearchInput.value = selectedFormat.name;
      formatSelector.value = formatId;
      hideFormatDropdown();

      // Show generate report panel
      if (generateReportPanel) {
        generateReportPanel.classList.remove('hidden');
        updateContainerState();
      }
    } else {
      console.error('Selected format not found:', formatId);
    }
  } else {
    // No format selected
    selectedFormat = null;

    // Hide generate report panel
    if (generateReportPanel) {
      generateReportPanel.classList.add('hidden');
    }
  }
}

/**
 * Render the format dropdown items
 */
function renderFormatDropdown() {
  if (!formatDropdownList || !formatNoResults) return;

  // Clear existing items
  formatDropdownList.innerHTML = '';
  formatNoResults.classList.add('hidden');

  if (filteredFormats.length === 0) {
    formatNoResults.classList.remove('hidden');
    formatDropdownList.appendChild(formatNoResults);
    return;
  }

  // Create dropdown items
  filteredFormats.forEach((format, index) => {
    const item = document.createElement('div');
    item.className = 'format-dropdown-item';
    item.textContent = format.name;
    item.dataset.formatId = format.id;
    item.dataset.index = index;

    // Add click handler
    item.addEventListener('mousedown', (e) => {
      e.preventDefault(); // Prevent input blur
      selectFormat(format.id);
    });

    formatDropdownList.appendChild(item);
  });

  // Highlight first item by default
  selectedFormatIndex = 0;
  updateHighlight();
}

/**
 * Handle search input
 */
function handleFormatSearch(event) {
  const query = event.target.value.toLowerCase().trim();

  if (query === '') {
    filteredFormats = [...availableFormats];
  } else {
    filteredFormats = availableFormats.filter(format =>
      format.name.toLowerCase().includes(query)
    );
  }

  selectedFormatIndex = filteredFormats.length > 0 ? 0 : -1;
  renderFormatDropdown();
  showFormatDropdown();
}

/**
 * Handle keyboard navigation
 */
function handleFormatKeydown(event) {
  if (!formatDropdownList.classList.contains('hidden')) {
    switch (event.key) {
      case 'ArrowDown':
        event.preventDefault();
        if (selectedFormatIndex < filteredFormats.length - 1) {
          selectedFormatIndex++;
          updateHighlight();
        }
        break;

      case 'ArrowUp':
        event.preventDefault();
        if (selectedFormatIndex > 0) {
          selectedFormatIndex--;
          updateHighlight();
        }
        break;

      case 'Enter':
        event.preventDefault();
        if (selectedFormatIndex >= 0 && selectedFormatIndex < filteredFormats.length) {
          selectFormat(filteredFormats[selectedFormatIndex].id);
        }
        break;

      case 'Escape':
        event.preventDefault();
        hideFormatDropdown();
        formatSearchInput.blur();
        break;
    }
  }
}

/**
 * Update highlighted item in dropdown
 */
function updateHighlight() {
  const items = formatDropdownList.querySelectorAll('.format-dropdown-item');
  items.forEach((item, index) => {
    if (index === selectedFormatIndex) {
      item.classList.add('highlighted');
    } else {
      item.classList.remove('highlighted');
    }
  });
}

/**
 * Select a format
 */
function selectFormat(formatId) {
  handleFormatSelection(formatId);
}

/**
 * Show format dropdown
 */
function showFormatDropdown() {
  if (formatDropdownList && filteredFormats.length > 0) {
    formatDropdownList.classList.remove('hidden');
    // Add expanded class for more space
    formatDropdownList.classList.add('format-dropdown-expanded');
    // Add container class to maintain height
    if (expandedContainer) {
      expandedContainer.classList.add('format-dropdown-active');
    }
    // Collapse transcription panel to make more space
    collapseTranscriptionPanel();
  }
}

/**
 * Hide format dropdown
 */
function hideFormatDropdown() {
  if (formatDropdownList) {
    formatDropdownList.classList.add('hidden');
    // Remove expanded class
    formatDropdownList.classList.remove('format-dropdown-expanded');
    // Remove container class to restore normal height
    if (expandedContainer) {
      expandedContainer.classList.remove('format-dropdown-active');
    }
    // Expand transcription panel back to normal
    expandTranscriptionPanel();
  }
}

/**
 * Hide format dropdown with delay (for blur events)
 */
function hideFormatDropdownDelayed() {
  setTimeout(() => {
    hideFormatDropdown();
  }, 150);
}

/**
 * Collapse transcription panel to make space for format dropdown
 */
function collapseTranscriptionPanel() {
  if (transcriptionPanel && !transcriptionPanel.classList.contains('hidden')) {
    transcriptionPanel.classList.add('transcription-panel-collapsed');
  }
}

/**
 * Expand transcription panel back to normal size
 */
function expandTranscriptionPanel() {
  if (transcriptionPanel) {
    transcriptionPanel.classList.remove('transcription-panel-collapsed');
  }
}

/**
 * Update container size based on current state
 */
function updateContainerState() {
  if (!expandedContainer) return;

  // Remove all state classes
  expandedContainer.classList.remove('has-transcription', 'has-report', 'report-expanded');

  // Check if transcription is visible and has content
  const hasTranscription = transcriptionPanel &&
                          !transcriptionPanel.classList.contains('hidden') &&
                          transcriptionText &&
                          transcriptionText.value.trim() !== '';

  // Check if generate report panel is visible
  const hasReport = generateReportPanel &&
                   !generateReportPanel.classList.contains('hidden');

  // Apply appropriate classes
  if (hasTranscription) {
    expandedContainer.classList.add('has-transcription');
  }

  if (hasReport) {
    expandedContainer.classList.add('has-report');
  }
}

/**
 * Expand report area for better viewing
 */
function expandReportArea() {
  if (expandedContainer && generateReportPanel) {
    expandedContainer.classList.add('report-expanded');
    generateReportPanel.classList.add('report-expanded');
    expandedContainer.classList.add('report-mode');

    // Directly set textarea height to override Tailwind classes
    if (transcriptionText) {
      transcriptionText.style.height = '280px';
      transcriptionText.style.minHeight = '280px';
      transcriptionText.style.transition = 'height 0.3s ease-in-out';
    }
  }
}

/**
 * Collapse report area back to normal
 */
function collapseReportArea() {
  if (expandedContainer && generateReportPanel) {
    expandedContainer.classList.remove('report-expanded', 'report-mode');
    generateReportPanel.classList.remove('report-expanded');

    // Reset textarea height back to normal
    if (transcriptionText) {
      transcriptionText.style.height = '';
      transcriptionText.style.minHeight = '';
    }
  }
}

/**
 * Generate report using selected format
 */
async function generateReport() {
  if (!currentTranscription || !selectedFormat) {
    alert('Please ensure you have transcription text and a selected format');
    return;
  }

  if (isGeneratingReport) {
    console.log('Report generation already in progress');
    return;
  }

  try {
    console.log('Generating report...', {
      formatId: selectedFormat.id,
      formatName: selectedFormat.name,
      transcriptionLength: currentTranscription.length
    });

    // Update UI to show generation in progress
    isGeneratingReport = true;
    updateGenerationUI(true);

    // CRITICAL: Ensure TextRad tabs are active and services ready before report generation
    console.log('🔄 Ensuring TextRad tabs are active and ready...');
    try {
      updateGenerationStatus('Activating TextRad services...');

      const activationResponse = await chrome.runtime.sendMessage({
        action: 'ACTIVATE_TEXTRAD_TABS'
      });

      if (activationResponse && activationResponse.success) {
        console.log(`✅ Successfully activated ${activationResponse.activatedTabs} TextRad tabs`);
        updateGenerationStatus('TextRad services ready. Generating report...');

        // Wait a moment for services to fully initialize
        await new Promise(resolve => setTimeout(resolve, 2000));
      } else {
        console.warn('Tab activation failed, proceeding anyway...');
        updateGenerationStatus('Tab activation warning. Attempting report generation...');
      }
    } catch (activationError) {
      console.warn('Tab activation error:', activationError);
      updateGenerationStatus('Proceeding with report generation...');
    }

    // Call background script to generate report with retry logic
    let reportResponse = null;
    let lastError = null;
    const maxRetries = 2;

    for (let attempt = 0; attempt <= maxRetries; attempt++) {
      try {
        if (attempt > 0) {
          updateGenerationStatus(`Retrying... (attempt ${attempt + 1}/${maxRetries + 1})`);
          console.log(`🔄 Retrying report generation (attempt ${attempt + 1}/${maxRetries + 1})`);

          // Wait longer between retries
          await new Promise(resolve => setTimeout(resolve, 3000));

          // Re-activate tabs before retry
          try {
            await chrome.runtime.sendMessage({ action: 'ACTIVATE_TEXTRAD_TABS' });
            await new Promise(resolve => setTimeout(resolve, 2000));
          } catch (retryActivationError) {
            console.warn('Retry tab activation failed:', retryActivationError);
          }
        }

        updateGenerationStatus('Generating report...');
        const response = await chrome.runtime.sendMessage({
          action: 'GENERATE_REPORT',
          transcription: currentTranscription,
          formatId: selectedFormat.id,
          tabId: selectedTabId
        });

        if (response && response.success) {
          reportResponse = response;
          break; // Success, exit retry loop
        } else {
          lastError = new Error(response?.error || 'Failed to generate report');

          // Check if this is a retryable error
          const errorMsg = lastError.message.toLowerCase();
          if (attempt < maxRetries && (
              errorMsg.includes('not available') ||
              errorMsg.includes('timeout') ||
              errorMsg.includes('suspended') ||
              errorMsg.includes('services not ready'))) {
            console.warn(`Attempt ${attempt + 1} failed with retryable error:`, lastError.message);
            continue; // Try again
          } else {
            throw lastError; // Non-retryable error or max retries reached
          }
        }
      } catch (error) {
        lastError = error;
        const errorMsg = error.message.toLowerCase();

        // Check if this is a retryable error
        if (attempt < maxRetries && (
            errorMsg.includes('not available') ||
            errorMsg.includes('timeout') ||
            errorMsg.includes('suspended') ||
            errorMsg.includes('services not ready') ||
            errorMsg.includes('timed out'))) {
          console.warn(`Attempt ${attempt + 1} failed with retryable error:`, error.message);
          continue; // Try again
        } else {
          throw error; // Non-retryable error or max retries reached
        }
      }
    }

    if (reportResponse && reportResponse.success) {
      // Update transcription with generated report
      currentReportContent = reportResponse.report;
      currentTranscription = reportResponse.report;

      // Expand report area for better viewing
      expandReportArea();

      // Update UI to show the generated report
      transcriptionText.value = reportResponse.report;
      updateUIForGeneratedReport();

      console.log('Report generated successfully, length:', reportResponse.report.length);
      showToast('Report generated successfully!', 2000);
    } else {
      throw lastError || new Error('Report generation failed after multiple attempts');
    }
  } catch (error) {
    console.error('Error generating report:', error);
    showToast('Failed to generate report: ' + error.message, 3000);
  } finally {
    // Reset UI state
    isGeneratingReport = false;
    updateGenerationUI(false);
  }
}

/**
 * Update generation UI state
 */
function updateGenerationUI(generating) {
  if (!generateButton || !generateButtonText || !generationProgress) return;

  if (generating) {
    generateButton.disabled = true;
    generateButtonText.textContent = 'Generating...';
    generationProgress.classList.remove('hidden');
    if (generationStatus) {
      generationStatus.textContent = 'Processing with TextRad AI...';
    }
  } else {
    generateButton.disabled = false;
    generateButtonText.textContent = 'Generate Report';
    generationProgress.classList.add('hidden');
  }
}

/**
 * Update generation status message
 */
function updateGenerationStatus(statusMessage) {
  if (generationStatus) {
    generationStatus.textContent = statusMessage;
    console.log('Generation status:', statusMessage);
  }
}

/**
 * Update UI after report is generated
 */
function updateUIForGeneratedReport() {
  // Update transcription panel header to indicate this is now a report
  const transcriptionHeader = transcriptionPanel.querySelector('h2');
  if (transcriptionHeader) {
    transcriptionHeader.innerHTML = '<i class="fas fa-file-medical mr-1"></i>Generated Report';
  }

  // Update copy button text
  if (copyButton) {
    copyButton.title = 'Copy Generated Report';
  }

  // Hide template selection and generate panels (report is complete)
  if (templateSelectionPanel) {
    templateSelectionPanel.classList.add('hidden');
  }
  if (generateReportPanel) {
    generateReportPanel.classList.add('hidden');
  }

  // Show a "New Dictation" option to reset the workflow
  showReportCompletionActions();
}

/**
 * Show actions available after report generation
 */
function showReportCompletionActions() {
  // The existing "New Dictation" button will reset everything
  // We could add additional actions here if needed
  console.log('Report generation complete - New Dictation button available for reset');
}

/**
 * Show template error message
 */
function showTemplateError(message) {
  if (templateError && templateErrorMessage) {
    templateErrorMessage.textContent = message;
    templateError.classList.remove('hidden');
  }

  console.error('Template error:', message);
}

/**
 * Reset template selection state (called by New Dictation)
 */
function resetTemplateState() {
  // Clear template-related state
  selectedFormat = null;
  currentReportContent = null;

  // Collapse report area if expanded
  collapseReportArea();

  // Reset UI elements
  if (formatSearchInput) {
    formatSearchInput.value = '';
    formatSearchInput.placeholder = 'Type to search formats...';
  }
  if (formatSelector) {
    formatSelector.value = '';
  }
  if (formatDropdownList) {
    formatDropdownList.innerHTML = '';
    formatDropdownList.classList.add('hidden');
  }
  availableFormats = [];
  filteredFormats = [];
  selectedFormatIndex = -1;

  // Hide template panels
  if (templateSelectionPanel) {
    templateSelectionPanel.classList.add('hidden');
  }
  if (generateReportPanel) {
    generateReportPanel.classList.add('hidden');
  }

  // Update container state after changes
  updateContainerState();
  if (templateError) {
    templateError.classList.add('hidden');
  }

  // Reset transcription panel header
  const transcriptionHeader = transcriptionPanel.querySelector('h2');
  if (transcriptionHeader) {
    transcriptionHeader.innerHTML = 'Transcription';
  }

  // Reset copy button
  if (copyButton) {
    copyButton.title = 'Copy Text';
  }

  console.log('Template state reset');
}

// Listen for messages from background script (mini extension triggers)
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Popup received message:', message);

  if (message.action === 'MINI_EXTENSION_TRIGGER_RECORDING' && message.fromMiniExtension) {
    console.log('Triggering recording from mini extension...');

    // Wait for popup to be fully initialized
    setTimeout(() => {
      // Trigger the same recording function as the mini record button
      if (recordButton && !recordButton.disabled) {
        console.log('Clicking record button from mini extension trigger...');
        recordButton.click();
        sendResponse({ success: true, message: 'Recording triggered from mini extension' });
      } else if (miniRecordButton && !miniRecordButton.disabled) {
        console.log('Clicking mini record button from mini extension trigger...');
        miniRecordButton.click();
        sendResponse({ success: true, message: 'Recording triggered from mini extension via mini button' });
      } else {
        console.error('No record button available or all disabled');
        sendResponse({ success: false, message: 'No record button available' });
      }
    }, 300);

    return true; // Keep message channel open for async response
  }
});

// Initialize popup when DOM is loaded
document.addEventListener('DOMContentLoaded', initialize);