/**
 * TextRad Dictation Extension - Popup Script
 * 
 * Responsibilities:
 * - User interface for recording and managing dictations
 * - Communication with service worker (background script)
 * - Audio recording and visualization
 * - Displaying transcription results
 */

// Import AudioRecorder and TranscriptionService
import { ExtensionAudioRecorder } from '../lib/audio-recorder.js';
import { ExtensionTranscriptionService } from '../lib/transcription-service.js';
import { StorageManager } from '../lib/storage-manager.js';
import { ExtensionReportStorage } from '../lib/extension-report-storage.js';
import { HTMLFormatter } from '../lib/html-formatter.js';

// UI Elements
let recordButton;
let visualizerCanvas;
let recordingStatus;
// Timer removed
let recordingPanel;
let transcriptionPanel;
let transcriptionText;
let newDictationButton;
let copyButton;
let settingsButton;
let settingsPanel;
let closeSettingsButton;
let saveSettingsButton;
let appStatusPanel;
let appStatusIndicator;
let appStatusText;
let openTextRadButton;
let loadingOverlay;
let loadingText;
let loginRequiredMessage;
let loginButton;
let toastNotification;
let toastMessage;

// HTML preview elements
let toggleHtmlPreview;
let htmlPreviewContainer;
let htmlPreviewContent;
let previewToggleText;
let isShowingHtmlPreview = false;

// New template selection UI elements
let templateSelectionPanel;
let formatSelector;
let formatSearchInput;
let formatDropdownList;
let formatNoResults;
let availableFormats = [];
let filteredFormats = [];
let selectedFormatIndex = -1;
let templateLoading;

// Quick generate UI elements
let quickGenerateContainer;
let quickGenerateButton;
let quickGenerateButtonText;
let quickGenerationProgress;
let quickGenerationStatus;

// Transcription panel elements for collapse functionality
let templateError;
let templateErrorMessage;
let generateReportPanel;
let generateButton;
let generateButtonText;
let generationProgress;
let generationStatus;

// App page warning elements
let appPageWarning;
let openAppPageButton;

// Main/Mini UI Elements
let expandedContainer;
let collapsedContainer;
let collapseButton;
let expandButton;
let miniRecordButton;
let closeButton;
let collapsedStatus;

// Settings Elements
let autoSendSetting;
let silenceTimeoutSetting;
let silenceTimeoutValue;
let transcriptionProviderSetting;
let extensionSizeSetting;

// Service instances
let audioRecorder;
let transcriptionService;
let storageManager;
let reportStorage;

// State tracking
let isRecording = false;
let isTranscribing = false;
let recordingStartTime = 0;
let recordingTimer = null;
let currentTranscription = '';
let textradTabs = [];
let selectedTabId = null;

// Template/format state
let selectedFormat = null;
let isGeneratingReport = false;
let currentReportContent = null;

// Dynamic message rotation for report generation
let messageRotationInterval = null;
let currentMessageIndex = 0;
const generationMessages = [
  'Analyzing transcription...',
  'Processing medical information...',
  'Extracting key findings...',
  'Structuring report sections...',
  'Applying format template...',
  'Generating professional report...',
  'Finalizing medical documentation...'
];

// Format caching to avoid repeated DOM extraction
let formatCache = new Map();
const FORMAT_CACHE_TTL = 300000; // 5 minutes

// Login status monitoring
let loginCheckInterval = null;
let lastLoginCheck = 0;

// Reset operation tracking
let isResetInProgress = false;

// Operation state management for performance optimization
let activeOperations = new Set();
let tabInfoCache = null;
let tabInfoCacheTime = 0;
const TAB_CACHE_TTL = 30000; // 30 seconds

// Login status cache
let loginStatusCache = new Map();
const LOGIN_CACHE_TTL = 30000; // 30 seconds

/**
 * Check Firebase auth directly from popup by reading localStorage
 * This helps detect login status even when DOM-based detection fails (e.g., on landing page)
 */
async function checkFirebaseAuthDirectly() {
  try {
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tabs[0]) {
      return false;
    }

    // Execute script to check Firebase auth in localStorage
    const result = await chrome.scripting.executeScript({
      target: { tabId: tabs[0].id },
      func: () => {
        try {
          // Check for Firebase auth user in localStorage
          // This key pattern is used by Firebase Auth SDK
          const firebaseKeys = Object.keys(localStorage).filter(key =>
            key.startsWith('firebase:authUser:')
          );

          if (firebaseKeys.length > 0) {
            const authData = localStorage.getItem(firebaseKeys[0]);
            return authData && authData !== 'null';
          }

          return false;
        } catch (e) {
          return false;
        }
      }
    });

    return result && result[0] && result[0].result;
  } catch (error) {
    console.error('Error checking Firebase auth:', error);
    return false;
  }
}

// Tab responsiveness tracking
function markTabAsUnresponsive(tabId) {
  const tabKey = `unresponsive_${tabId}`;
  localStorage.setItem(tabKey, Date.now().toString());
  console.log(`🚫 Marked tab ${tabId} as unresponsive`);

  // Invalidate cache if it contains this tab
  if (tabInfoCache && tabInfoCache.tabs.some(tab => tab.id === tabId)) {
    console.log('📋 Invalidating tab cache due to unresponsive tab');
    tabInfoCache = null;
    tabInfoCacheTime = 0;
  }
}

// Request deduplication
let pendingTabRequest = null;

// Flag to determine whether to append or replace transcription
// true = append to existing transcription (set by main record button)
// false = start fresh (set by New Dictation button)
// Default to undefined so we know when it's not been explicitly set
let recordMoreButtonClicked = undefined;

/**
 * Initialize popup
 */
async function initialize() {
  console.log('TextRad Dictation Extension popup initialized');

  // Initialize services
  storageManager = new StorageManager();
  await storageManager.initialize();

  // Initialize report storage
  reportStorage = new ExtensionReportStorage();
  try {
    await reportStorage.initialize();
    console.log('Report storage initialized successfully');
  } catch (error) {
    console.warn('Report storage initialization failed:', error);
  }

  // Get UI elements
  getUIElements();

  // Set up event listeners
  setupEventListeners();

  // Update initial container state
  updateContainerState();

  // Initialize audio recorder and transcription service
  initializeServices();

  // Disable recording buttons until login status is verified
  if (recordButton) {
    recordButton.disabled = true;
    recordButton.classList.add('disabled');
  }
  if (miniRecordButton) {
    miniRecordButton.disabled = true;
    miniRecordButton.classList.add('disabled');
  }

  // No need to move buttons anymore - using simpler approach

  // Start with not-logged-in state by default
  updateAppStatus('not-logged-in');

  // Check for TextRad tabs
  findTextRadTabs();

  // Check app page status (show warning if needed)
  checkAppPageStatus();

  // Check if mini extension is already deployed on current tab
  checkMiniExtensionDeployment();

  // Apply default extension size before loading settings
  applyExtensionSize();

  // Load settings
  loadSettings();

  // Reset record mode flags to ensure clean start
  recordMoreButtonClicked = undefined;

  // Check URL for mini mode parameter
  const urlParams = new URLSearchParams(window.location.search);
  if (urlParams.has('viewMode') && urlParams.get('viewMode') === 'mini') {
    // Switch to mini mode immediately from URL param
    expandedContainer.classList.add('hidden');
    collapsedContainer.classList.remove('hidden');
    document.body.classList.add('mini-mode');
    document.documentElement.classList.add('mini-mode');

    // Force window size to exactly match mini UI
    window.resizeTo(150, 40);

    updateMiniStatus();
  } else {
    // Otherwise, load saved view mode
    chrome.storage.local.get(['viewMode'], (result) => {
      if (result.viewMode === 'mini') {
        // Switch to mini mode
        expandedContainer.classList.add('hidden');
        collapsedContainer.classList.remove('hidden');
        document.body.classList.add('mini-mode');
        document.documentElement.classList.add('mini-mode');

        // Force window size to exactly match mini UI
        window.resizeTo(140, 32);

        updateMiniStatus();
      } else {
        // Default to expanded mode
        expandedContainer.classList.remove('hidden');
        collapsedContainer.classList.add('hidden');
        document.body.classList.remove('mini-mode');
        document.documentElement.classList.remove('mini-mode');
      }
    });
  }

  // Check state of recording from service worker
  const extensionState = await chrome.runtime.sendMessage({ action: 'GET_STATE' });
  console.log('Extension state:', extensionState);

  // Check for any stale recording state and reset it
  if (extensionState && extensionState.state && extensionState.state.recording) {
    console.log('Found stale recording state, resetting it');

    // Reset recording state in the service worker
    chrome.runtime.sendMessage({
      action: 'SET_TRANSCRIPTION',
      transcription: '' // Empty string to clear transcription
    });

    // Also reset recording state in storage
    await storageManager.setRecordingInProgress(false);

    // Show default recording UI
    isRecording = false;
    showRecordingUI();
  } else if (extensionState && extensionState.state && extensionState.state.transcribing) {
    console.log('Found stale transcribing state, resetting it');

    // Reset the state in the service worker
    chrome.runtime.sendMessage({
      action: 'SET_TRANSCRIPTION',
      transcription: '' // Empty string to clear transcription
    });

    // Show default recording UI
    isRecording = false;
    showRecordingUI();
  } else if (extensionState && extensionState.state && extensionState.state.lastTranscription) {
    // If we have a previous transcription, show it
    console.log('Found previous transcription, showing it');
    currentTranscription = extensionState.state.lastTranscription;
    showTranscriptionUI(currentTranscription);
    // Wait for TextRad tabs to be detected before loading templates
    setTimeout(() => {
      loadAvailableFormats();
    }, 1000);
  }

  // Check extension state
  checkState();

  // Start continuous login monitoring
  startLoginMonitoring();

  // Check for any pending reset operations from interrupted close
  await checkPendingResetOperation();
}

/**
 * Get UI elements
 */
function getUIElements() {
  // Main UI elements
  recordButton = document.getElementById('record-button');
  visualizerCanvas = null; // Removing visualizer
  recordingStatus = document.getElementById('recording-status');
  // Timer element removed
  recordingPanel = document.getElementById('recording-panel');
  transcriptionPanel = document.getElementById('transcription-panel');
  transcriptionText = document.getElementById('transcription-text');
  newDictationButton = document.getElementById('new-dictation-button');
  copyButton = document.getElementById('copy-button');

  // App status elements
  appStatusPanel = document.getElementById('app-status-panel');
  appStatusIndicator = document.getElementById('app-status-indicator').querySelector('.status-dot');
  appStatusText = document.getElementById('app-status-text');
  openTextRadButton = document.getElementById('open-textrad-button');

  // Login required message elements
  loginRequiredMessage = document.getElementById('login-required-message');
  loginButton = document.getElementById('login-button');

  // Settings elements
  settingsButton = document.getElementById('settings-button');
  settingsPanel = document.getElementById('settings-panel');
  closeSettingsButton = document.getElementById('close-settings-button');
  saveSettingsButton = document.getElementById('save-settings-button');
  autoSendSetting = document.getElementById('auto-send-setting');
  silenceTimeoutSetting = document.getElementById('silence-timeout-setting');
  silenceTimeoutValue = document.getElementById('silence-timeout-value');
  transcriptionProviderSetting = document.getElementById('transcription-provider-setting');

  // Extension size now uses radio buttons instead of dropdown
  extensionSizeSetting = null; // Deprecated - now using radio buttons

  // Loading elements
  loadingOverlay = document.getElementById('loading-overlay');
  loadingText = document.getElementById('loading-text');

  // Toast notification elements
  toastNotification = document.getElementById('toast-notification');
  toastMessage = document.getElementById('toast-message');

  // HTML preview elements
  toggleHtmlPreview = document.getElementById('toggle-html-preview');
  htmlPreviewContainer = document.getElementById('html-preview-container');
  htmlPreviewContent = document.getElementById('html-preview-content');
  previewToggleText = document.getElementById('preview-toggle-text');

  // Main/Mini UI elements
  expandedContainer = document.getElementById('expanded-container');
  collapsedContainer = document.getElementById('collapsed-container');
  collapseButton = document.getElementById('collapse-button');
  expandButton = document.getElementById('expand-button');
  miniRecordButton = document.getElementById('mini-record-button');
  closeButton = document.getElementById('close-button');
  collapsedStatus = document.getElementById('collapsed-status');

  // New template selection elements
  templateSelectionPanel = document.getElementById('template-selection-panel');
  formatSelector = document.getElementById('format-selector');
  formatSearchInput = document.getElementById('format-search-input');
  formatDropdownList = document.getElementById('format-dropdown-list');
  formatNoResults = document.getElementById('format-no-results');
  templateLoading = document.getElementById('template-loading');
  templateError = document.getElementById('template-error');
  templateErrorMessage = document.getElementById('template-error-message');
  generateReportPanel = document.getElementById('generate-report-panel');
  generateButton = document.getElementById('generate-button');
  generateButtonText = document.getElementById('generate-button-text');
  generationProgress = document.getElementById('generation-progress');
  generationStatus = document.getElementById('generation-status');

  // Quick generate elements
  quickGenerateContainer = document.getElementById('quick-generate-container');
  quickGenerateButton = document.getElementById('quick-generate-button');
  quickGenerateButtonText = document.getElementById('quick-generate-button-text');
  quickGenerationProgress = document.getElementById('quick-generation-progress');
  quickGenerationStatus = document.getElementById('quick-generation-status');

  // App page warning elements
  appPageWarning = document.getElementById('app-page-warning');
  openAppPageButton = document.getElementById('open-app-page-button');
}

/**
 * Set up event listeners
 */
function setupEventListeners() {
  // Record button
  recordButton.addEventListener('click', toggleRecording);

  // New dictation button
  newDictationButton.addEventListener('click', startNewDictation);

  // Copy button - now also auto-saves to web app
  copyButton.addEventListener('click', copyAndSaveReport);

  // HTML preview toggle
  if (toggleHtmlPreview) {
    toggleHtmlPreview.addEventListener('click', toggleHtmlPreviewMode);
  }

  // Format selector (searchable dropdown)
  if (formatSearchInput) {
    formatSearchInput.addEventListener('input', handleFormatSearch);
    formatSearchInput.addEventListener('keydown', handleFormatKeydown);
    formatSearchInput.addEventListener('click', handleFormatInputClick);
  }

  // Generate report button
  if (generateButton) {
    generateButton.addEventListener('click', generateReport);
  }

  // Quick generate report button
  if (quickGenerateButton) {
    quickGenerateButton.addEventListener('click', generateQuickReport);
  }

  // Open TextRad button
  openTextRadButton.addEventListener('click', openTextRad);

  // Open TextRAD App button (app page warning)
  if (openAppPageButton) {
    openAppPageButton.addEventListener('click', openTextRadApp);
  }

  // Deploy Mini Extension button
  const deployMiniButton = document.getElementById('deploy-mini-button');
  if (deployMiniButton) {
    deployMiniButton.addEventListener('click', deployMiniExtension);
  }

  // Login button (if present)
  if (loginButton) {
    loginButton.addEventListener('click', openTextRad);
  }

  // Refresh status button
  const refreshButton = document.getElementById('refresh-status-button');
  if (refreshButton) {
    refreshButton.addEventListener('click', () => {
      console.log('Main refresh button clicked - performing aggressive login recheck');

      // Force clear all cached state
      textradTabs = [];
      selectedTabId = null;
      lastLoginCheck = 0;

      // Reset to not-logged-in state first
      updateAppStatus('not-logged-in');

      // Show loading indicator
      showLoading('Refreshing login status...');

      // Force a fresh check with delay to ensure caches are cleared
      setTimeout(() => {
        findTextRadTabs();
      }, 500);
    });
  }

  // Settings button
  settingsButton.addEventListener('click', toggleSettings);

  // Close settings button
  closeSettingsButton.addEventListener('click', toggleSettings);

  // Save settings button
  saveSettingsButton.addEventListener('click', saveSettings);

  // Silence timeout slider
  silenceTimeoutSetting.addEventListener('input', updateSilenceTimeoutValue);

  // Extension size radio buttons - apply immediately for instant preview
  document.querySelectorAll('.extension-size-radio').forEach(radio => {
    radio.addEventListener('change', () => {
      applyExtensionSize();
      // Auto-save the size setting for persistence
      saveExtensionSizeOnly();
    });
  });

  // Domain list toggle button
  const domainListToggle = document.getElementById('domain-list-toggle');
  const domainListContainer = document.getElementById('domain-list-container');
  const domainListChevron = document.getElementById('domain-list-chevron');

  if (domainListToggle) {
    domainListToggle.addEventListener('click', () => {
      const isHidden = domainListContainer.classList.contains('hidden');
      if (isHidden) {
        domainListContainer.classList.remove('hidden');
        domainListChevron.classList.add('rotated');
      } else {
        domainListContainer.classList.add('hidden');
        domainListChevron.classList.remove('rotated');
      }
    });
  }

  // Collapse button to switch to mini UI
  collapseButton.addEventListener('click', toggleViewMode);

  // Mini record button in mini mode
  miniRecordButton.addEventListener('click', toggleRecording);

  // Expand button to go back to full view
  expandButton.addEventListener('click', toggleViewMode);

  // Close button - actually closes the popup
  closeButton.addEventListener('click', closePopup);

  // Close dropdown when clicking outside
  document.addEventListener('click', (event) => {
    if (formatDropdownList && !formatDropdownList.classList.contains('hidden')) {
      const formatContainer = document.getElementById('format-selector-container');
      const isClickInsideDropdown = formatContainer && formatContainer.contains(event.target);
      if (!isClickInsideDropdown) {
        hideFormatDropdown();
      }
    }
  });

  // Add beforeunload event listener to handle potential popup closing
  window.addEventListener('beforeunload', handlePopupClosing);

  // Add document click handler to close format dropdown when clicking outside
  document.addEventListener('click', (event) => {
    if (formatDropdownList && !formatDropdownList.classList.contains('hidden')) {
      // Check if click is outside the format selector area
      const formatContainer = document.getElementById('format-selector-container');
      // Don't close if clicking on the input itself (handled by input click handler)
      if (formatContainer && !formatContainer.contains(event.target) && event.target !== formatSearchInput) {
        hideFormatDropdown();
        isSelectingFormat = false;
      }
    }
  });

  // Add unload event listener to clean up monitoring
  window.addEventListener('unload', () => {
    if (loginCheckInterval) {
      clearInterval(loginCheckInterval);
      loginCheckInterval = null;
    }
  });

  // Add auto-resize functionality to transcription textarea
  if (transcriptionText) {
    transcriptionText.addEventListener('input', () => {
      autoResizeTextarea(transcriptionText);
    });
    transcriptionText.addEventListener('keyup', () => {
      autoResizeTextarea(transcriptionText);
    });
  }

  // Handle fullscreen/mini toggle via keyboard shortcut (Escape key)
  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
      toggleViewMode();
      e.preventDefault();
    }
  });
}

/**
 * Initialize audio recorder and transcription service
 */
function initializeServices() {
  // Initialize audio recorder without visualizer
  audioRecorder = new ExtensionAudioRecorder(null, storageManager);

  // Set audio recorder event handlers
  audioRecorder.onRecordingStart = handleRecordingStart;
  audioRecorder.onRecordingStop = handleRecordingStop;
  audioRecorder.onSilenceDetected = handleSilenceDetected;
  audioRecorder.onError = handleRecordingError;

  // Initialize transcription service
  transcriptionService = new ExtensionTranscriptionService(storageManager);

  // Set transcription service event handlers
  transcriptionService.onTranscriptionStart = handleTranscriptionStart;
  transcriptionService.onTranscriptionComplete = handleTranscriptionComplete;
  transcriptionService.onTranscriptionError = handleTranscriptionError;

  // Initialize Firebase services
  transcriptionService.initializeFirebaseServices();
}

/**
 * Operation state management functions
 */
function startOperation(operationType) {
  activeOperations.add(operationType);
  console.log(`🔄 Started operation: ${operationType}`);
}

function endOperation(operationType) {
  activeOperations.delete(operationType);
  console.log(`✅ Completed operation: ${operationType}`);
}

function hasActiveOperations() {
  return activeOperations.size > 0;
}

function shouldSkipPeriodicChecks() {
  return hasActiveOperations() || isGeneratingReport;
}

/**
 * Start continuous login monitoring
 */
function startLoginMonitoring() {
  // Clear any existing interval
  if (loginCheckInterval) {
    clearInterval(loginCheckInterval);
  }

  // Check login status every 10 seconds (reduced frequency to prevent interference)
  loginCheckInterval = setInterval(() => {
    // Skip periodic checks during active operations
    if (shouldSkipPeriodicChecks()) {
      console.log('⏸️ Skipping periodic check - active operation in progress');
      return;
    }

    // Only check if we're not currently performing a login check
    const now = Date.now();
    if (now - lastLoginCheck > 8000) { // Prevent rapid fire checks (increased from 3s to 8s)
      console.log('Periodic login check (silent)...');
      findTextRadTabsSilent();
    }

    // Also ensure login message is visible if user is not logged in
    // BUT don't interfere if we have active transcription or user is currently using the extension
    if (!isUserLoggedIn()) {
      const loginMsg = document.getElementById('login-required-message');
      const transcriptionPanel = document.getElementById('transcription-panel');
      const hasActiveTranscription = transcriptionPanel && !transcriptionPanel.classList.contains('hidden');

      // Only force login message if no active transcription is shown
      if (loginMsg && loginMsg.classList.contains('hidden') && !hasActiveTranscription) {
        console.log('User not logged in but login message is hidden - fixing...');
        updateAppStatus('not-logged-in');
      }
    }
  }, 10000); // Reduced frequency from 5s to 10s

  // Also check when popup gains focus
  window.addEventListener('focus', () => {
    console.log('Popup gained focus, checking login status...');
    findTextRadTabs();
  });

  // Check when popup becomes visible
  document.addEventListener('visibilitychange', () => {
    if (!document.hidden) {
      console.log('Popup became visible, checking login status...');
      findTextRadTabs();
    }
  });
}

/**
 * Check for TextRad tabs (with forced refresh)
 */
/**
 * Check if user needs to navigate to /app/ page
 * Show warning banner if logged in but not on app page
 */
async function checkAppPageStatus() {
  if (!appPageWarning) return;

  try {
    // Query all TextRAD tabs
    const tabs = await chrome.tabs.query({});
    const textradTabs = tabs.filter(tab =>
      tab.url && (
        tab.url.includes('textrad.in') ||
        tab.url.includes('textrad.com') ||
        tab.url.includes('localhost')
      )
    );

    let hasAppPage = false;
    let hasLoggedInNonAppPage = false;

    // Check each TextRAD tab for app page status
    for (const tab of textradTabs) {
      try {
        const response = await chrome.tabs.sendMessage(tab.id, { action: 'GET_PAGE_STATUS' });

        if (response && response.isLoggedIn) {
          if (response.isAppPage) {
            hasAppPage = true;
            break; // Found an app page, no need to show warning
          } else {
            hasLoggedInNonAppPage = true;
          }
        }
      } catch (error) {
        // Tab might not have content script, ignore
        console.log(`Could not check app page status for tab ${tab.id}`);
      }
    }

    // Show warning if user is logged in but not on /app/ page
    if (hasLoggedInNonAppPage && !hasAppPage) {
      appPageWarning.classList.remove('hidden');
    } else {
      appPageWarning.classList.add('hidden');
    }
  } catch (error) {
    console.error('Error checking app page status:', error);
  }
}

function findTextRadTabs(forceRefresh = false) {
  // Update last check time
  lastLoginCheck = Date.now();

  // Check if we can use cached tab information
  const now = Date.now();
  if (!forceRefresh && tabInfoCache && (now - tabInfoCacheTime) < TAB_CACHE_TTL) {
    console.log('📋 Using cached tab information');
    textradTabs = tabInfoCache.tabs || [];
    selectedTabId = tabInfoCache.selectedTabId || null;

    if (textradTabs.length > 0) {
      updateAppStatus('connected');
      hideLoading();
      return;
    }
  }

  // Request deduplication - if there's already a pending request, wait for it
  if (pendingTabRequest) {
    console.log('🔄 Tab discovery already in progress, waiting for completion...');
    return pendingTabRequest;
  }

  // Start tab discovery operation
  startOperation('findTextRadTabs');

  // Clear cached tabs to force fresh login check, but preserve selectedTabId if we're not forcing refresh
  // and it's currently valid (helps prevent race conditions)
  const previousSelectedTabId = selectedTabId;
  textradTabs = [];

  // Only clear selectedTabId on forced refresh or if we don't have a valid one
  if (forceRefresh || !selectedTabId) {
    selectedTabId = null;
  } else {
    console.log('🔒 Preserving selectedTabId during tab discovery to prevent race conditions:', selectedTabId);
  }

  updateAppStatus('connecting');

  // Show loading while checking
  showLoading('Checking TextRad connection...');

  // Set a timeout to hide the loading indicator if it takes too long
  const loadingTimeout = setTimeout(() => {
    console.log('Loading timeout reached, hiding loading indicator');
    hideLoading();
  }, 5000); // 5 seconds timeout

  // Create and store the pending request promise
  pendingTabRequest = new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
      // Always hide loading indicator
      hideLoading();

      // Clear the timeout
      clearTimeout(loadingTimeout);

      if (chrome.runtime.lastError) {
        console.error('Error finding TextRad tabs:', chrome.runtime.lastError);
        updateAppStatus('disconnected');
        return;
      }

      if (response && response.success) {
        // Check if there's a login error message
        if (response.error && response.error.includes('not logged in')) {
          console.log('User not logged in to TextRad');
          textradTabs = [];
          updateAppStatus('not-logged-in');
          return;
        }

        if (response.tabs && response.tabs.length > 0) {
          // Store tabs with login information and filter out any known unresponsive tabs
          textradTabs = response.tabs.map(tab => ({
            ...tab,
            loggedIn: response.loggedInTabs && response.loggedInTabs[tab.id] === true
          })).filter(tab => {
            // Filter out tabs that we know are unresponsive from previous attempts
            const tabKey = `unresponsive_${tab.id}`;
            const lastFailTime = localStorage.getItem(tabKey);
            if (lastFailTime) {
              const timeSinceLastFail = Date.now() - parseInt(lastFailTime);
              // Remove tabs that have been unresponsive for more than 2 minutes
              if (timeSinceLastFail < 120000) {
                console.log(`🚫 Filtering out recently unresponsive tab ${tab.id}`);
                return false;
              } else {
                // Clear the flag if enough time has passed
                localStorage.removeItem(tabKey);
              }
            }
            return true;
          });

          // Check if we have any logged-in tabs
          const loggedInTabs = textradTabs.filter(tab => tab.loggedIn);

          if (loggedInTabs.length > 0) {
            console.log(`Found ${loggedInTabs.length} logged-in TextRad tabs`);
            console.log('Tab details:', loggedInTabs.map(tab => ({ id: tab.id, url: tab.url, title: tab.title })));
            selectedTabId = loggedInTabs[0].id;
            console.log('Selected tab ID for template loading:', selectedTabId);
            updateAppStatus('connected');
          } else {
            // We found TextRad tabs, but user isn't logged in
            console.log('TextRad tabs found but user not logged in');
            textradTabs = [];
            updateAppStatus('not-logged-in');
          }
        } else {
          // No TextRad tabs found
          textradTabs = [];
          if (response.hasTextRadTabs && response.notLoggedInCount > 0) {
            // We found TextRad tabs, but user isn't logged in
            console.log('TextRad tabs found but user not logged in');
            updateAppStatus('not-logged-in');
          } else {
            // No TextRad tabs at all
            console.log('No TextRad tabs found');
            updateAppStatus('disconnected');
          }
        }
      } else {
        console.log('Find TextRad tabs failed or returned no success');
        textradTabs = [];
        updateAppStatus('disconnected');
      }

      // Update cache with current results
      tabInfoCache = {
        tabs: [...textradTabs],
        selectedTabId: selectedTabId
      };
      tabInfoCacheTime = Date.now();

      // End the tab discovery operation
      endOperation('findTextRadTabs');

      // Clear pending request and resolve promise
      pendingTabRequest = null;
      resolve();
    });
  });

  return pendingTabRequest;
}

/**
 * Silent version of findTextRadTabs for background periodic checks
 * This version doesn't show loading indicators or update UI unless status actually changes
 */
function findTextRadTabsSilent() {
  // Update last check time
  lastLoginCheck = Date.now();

  // Check if we can use cached tab information
  const now = Date.now();
  if (tabInfoCache && (now - tabInfoCacheTime) < TAB_CACHE_TTL) {
    console.log('📋 Using cached tab information (silent check)');
    textradTabs = tabInfoCache.tabs || [];
    selectedTabId = tabInfoCache.selectedTabId || null;

    if (textradTabs.length > 0) {
      // Only update status if it's not already connected
      if (document.getElementById('app-status-text')?.textContent !== 'Connected to TextRad') {
        updateAppStatus('connected');
      }
      return;
    }
  }

  // Request deduplication - if there's already a pending request, wait for it
  if (pendingTabRequest) {
    console.log('🔄 Tab discovery already in progress, skipping silent check...');
    return pendingTabRequest;
  }

  // Store current login status for comparison
  const wasLoggedIn = isUserLoggedIn();

  // Clear cached tabs to force fresh login check
  textradTabs = [];

  // Don't show loading or update status to connecting during silent checks
  console.log('🔍 Performing silent login verification...');

  // Create and store the pending request promise
  pendingTabRequest = new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {

      if (chrome.runtime.lastError) {
        console.error('Error finding TextRad tabs (silent):', chrome.runtime.lastError);
        // Only update status if it changed
        if (wasLoggedIn) {
          updateAppStatus('disconnected');
        }
        pendingTabRequest = null;
        resolve();
        return;
      }

      if (response && response.success) {
        // Check if there's a login error message
        if (response.error && response.error.includes('not logged in')) {
          console.log('User not logged in to TextRad (silent check)');
          textradTabs = [];
          // Only update status if user was previously logged in
          if (wasLoggedIn) {
            updateAppStatus('not-logged-in');
          }
          pendingTabRequest = null;
          resolve();
          return;
        }

        if (response.tabs && response.tabs.length > 0) {
          // Store tabs with login information
          textradTabs = response.tabs.map(tab => ({
            ...tab,
            loggedIn: tab.loggedIn !== false // Default to true unless explicitly false
          }));

          // Update cache with new tab information
          tabInfoCache = {
            tabs: textradTabs,
            selectedTabId: selectedTabId
          };
          tabInfoCacheTime = now;

          console.log(`🔍 Silent check found ${textradTabs.length} logged-in TextRad tabs`);

          // Only update UI if status actually changed
          if (!wasLoggedIn) {
            updateAppStatus('connected');
            console.log('📱 Status changed from not-logged-in to connected');
          }
        } else {
          console.log('🔍 Silent check: No logged-in TextRad tabs found');
          textradTabs = [];
          // Only update status if user was previously logged in
          if (wasLoggedIn) {
            updateAppStatus('not-logged-in');
            console.log('📱 Status changed from connected to not-logged-in');
          }
        }
      } else {
        console.log('🔍 Silent check: Failed to get response from background script');
        textradTabs = [];
        // Only update status if user was previously logged in
        if (wasLoggedIn) {
          updateAppStatus('disconnected');
        }
      }

      // Clear pending request and resolve promise
      pendingTabRequest = null;
      resolve();
    });
  });

  return pendingTabRequest;
}

/**
 * Check login status by triggering tab discovery
 * This function was missing and causing ReferenceError in loadAvailableFormats
 */
function checkLoginStatus() {
  console.log('Checking login status via tab discovery...');
  return findTextRadTabs();
}

/**
 * Update app connection status
 */
function updateAppStatus(status) {
  // Ensure UI elements are available
  if (!appStatusIndicator) {
    appStatusIndicator = document.getElementById('app-status-indicator')?.querySelector('.status-dot');
  }
  if (!appStatusText) {
    appStatusText = document.getElementById('app-status-text');
  }

  if (appStatusIndicator) {
    appStatusIndicator.className = 'status-dot';
  }

  // Ensure elements exist before accessing them
  // Hide login required message by default (except for not-logged-in status)
  const msgElement = document.getElementById('login-required-message');
  if (msgElement) {
    if (status !== 'not-logged-in') {
      msgElement.classList.add('hidden');
    }
    loginRequiredMessage = msgElement; // Ensure the variable is assigned
  }

  // Ensure login button is defined
  if (!loginButton) {
    loginButton = document.getElementById('login-button');
  }

  // Disable record buttons by default until we confirm login status
  if (recordButton) {
    recordButton.disabled = true;
  }
  if (miniRecordButton) {
    miniRecordButton.disabled = true;
  }

  switch (status) {
    case 'connected':
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('connected');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Connected to TextRad';
      }
      // Enable recording when connected and logged in
      if (recordButton) {
        recordButton.disabled = false;
        recordButton.classList.remove('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = false;
        miniRecordButton.classList.remove('disabled');
      }
      break;

    case 'connecting':
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('connecting');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Connecting...';
      }
      break;

    case 'not-logged-in':
      // Special case - check if user might actually be logged in via Firebase auth
      // This handles the case where user is on landing page (DOM detection fails)
      // but Firebase auth exists in localStorage
      (async () => {
        const hasFirebaseAuth = await checkFirebaseAuthDirectly();

        if (hasFirebaseAuth) {
          // User IS logged in - check if they're already on app page
          const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
          const currentTab = tabs[0];
          const isOnAppPage = currentTab && currentTab.url && currentTab.url.includes('/app');

          if (isOnAppPage) {
            // User is logged in AND on app page - something else is wrong
            // Don't show app warning (they're already where they need to be)
            console.log('User is logged in and on app page - treating as disconnected');

            if (appStatusIndicator) {
              appStatusIndicator.classList.add('disconnected');
            }
            if (appStatusText) {
              appStatusText.textContent = 'Not connected';
            }

            // Hide both login message and app warning
            const loginMsg = document.getElementById('login-required-message');
            if (loginMsg) {
              loginMsg.classList.add('hidden');
            }
            const appWarning = document.getElementById('app-page-warning');
            if (appWarning) {
              appWarning.classList.add('hidden');
            }

            return;
          }

          // User IS logged in but NOT on app page - show app page warning
          console.log('Firebase auth detected despite not-logged-in status - showing app page warning');

          if (appStatusIndicator) {
            appStatusIndicator.classList.add('connecting');
          }
          if (appStatusText) {
            appStatusText.textContent = 'Please open TextRAD App';
          }

          // Hide login message
          const loginMsg = document.getElementById('login-required-message');
          if (loginMsg) {
            loginMsg.classList.add('hidden');
          }

          // Show app page warning
          const appWarning = document.getElementById('app-page-warning');
          if (appWarning) {
            appWarning.classList.remove('hidden');
            appPageWarning = appWarning;

            // Make sure it's visible
            appWarning.style.display = 'block';
            appWarning.style.visibility = 'visible';
            appWarning.style.opacity = '1';
          }

          // Setup open app page button
          const openAppBtn = document.getElementById('open-app-page-button');
          if (openAppBtn && !openAppBtn.hasEventListener) {
            openAppBtn.addEventListener('click', () => {
              chrome.tabs.create({ url: 'https://textrad.in/app/' });
            });
            openAppBtn.hasEventListener = true;
            openAppPageButton = openAppBtn;
          }

          return; // Exit early - don't show login message
        }

        // User genuinely not logged in - show login message
        if (appStatusIndicator) {
          appStatusIndicator.classList.add('disconnected');
        }
        if (appStatusText) {
          appStatusText.textContent = 'Login required';
        }

        // Always show login message prominently
        const loginMsg = document.getElementById('login-required-message');
        if (loginMsg) {
          loginMsg.classList.remove('hidden');
          loginRequiredMessage = loginMsg; // Ensure the variable is assigned

          // Make sure it's visible and prominent
          loginMsg.style.display = 'block !important';
          loginMsg.style.visibility = 'visible';
          loginMsg.style.opacity = '1';

          // Force it to be at the top of the UI
          const recordingPanel = document.getElementById('recording-panel');
          if (recordingPanel && recordingPanel.parentNode) {
            recordingPanel.parentNode.insertBefore(loginMsg, recordingPanel);
          }

          // Scroll into view
          setTimeout(() => {
            loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
          }, 100);
        }
      })();

      // Make sure recording UI is fully visible but disabled
      const recordingPanelElement = recordingPanel || document.getElementById('recording-panel');
      if (recordingPanelElement) {
        recordingPanelElement.classList.remove('hidden');
      }

      const transcriptionPanelElement = transcriptionPanel || document.getElementById('transcription-panel');
      if (transcriptionPanelElement) {
        // Only hide transcription if it doesn't have content or if this is an explicit logout
        const transcriptionText = document.getElementById('transcription-text');
        const hasContent = transcriptionText && transcriptionText.value && transcriptionText.value.trim().length > 0;

        // Don't hide transcription panel if user has content unless it's a fresh login check
        if (!hasContent) {
          transcriptionPanelElement.classList.add('hidden');
        }
      }

      // Ensure login button is set up
      const loginBtn = document.getElementById('login-button');
      if (loginBtn && !loginBtn.hasEventListener) {
        loginBtn.addEventListener('click', openTextRad);
        loginBtn.hasEventListener = true;
        loginButton = loginBtn;
      }

      // Ensure refresh button is set up
      const refreshBtn = document.getElementById('refresh-status-button');
      if (refreshBtn && !refreshBtn.hasEventListener) {
        refreshBtn.addEventListener('click', () => {
          console.log('Refresh button clicked - performing aggressive login recheck');

          // Force clear all cached state
          textradTabs = [];
          selectedTabId = null;
          lastLoginCheck = 0;

          // Reset to not-logged-in state first
          updateAppStatus('not-logged-in');

          showLoading('Refreshing login status...');

          // Force a fresh check with delay to ensure caches are cleared
          setTimeout(() => {
            findTextRadTabs();
          }, 500);
        });
        refreshBtn.hasEventListener = true;
      }

      // Disable recording when not logged in
      if (recordButton) {
        recordButton.disabled = true;
        recordButton.classList.add('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = true;
        miniRecordButton.classList.add('disabled');
      }
      break;

    case 'disconnected':
    default:
      if (appStatusIndicator) {
        appStatusIndicator.classList.add('disconnected');
      }
      if (appStatusText) {
        appStatusText.textContent = 'Not connected';
      }
      // Disable recording when disconnected
      if (recordButton) {
        recordButton.disabled = true;
        recordButton.classList.add('disabled');
      }
      if (miniRecordButton) {
        miniRecordButton.disabled = true;
        miniRecordButton.classList.add('disabled');
      }
      break;
  }

  // Update mini mode status
  updateMiniStatus();
}

/**
 * Open TextRad web app
 */
/**
 * Deploy mini extension on current active tab
 */
async function deployMiniExtension() {
  try {
    // Get current active tab
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!activeTab) {
      throw new Error('No active tab found');
    }

    const currentDomain = new URL(activeTab.url).hostname;
    console.log('Checking deployment preference for domain:', currentDomain);

    // Check if we have a preference for this domain
    const result = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
    const preferences = result.miniExtensionDomainPreferences || {};

    if (preferences[currentDomain]) {
      // Domain has preference, deploy directly
      console.log(`Domain ${currentDomain} has preference: ${preferences[currentDomain]}`);
      await performDeployment(activeTab, preferences[currentDomain]);
    } else {
      // New domain, show preference dialog
      console.log(`New domain ${currentDomain}, showing preference dialog`);
      await showDeploymentPreferenceDialog(activeTab, currentDomain);
    }
  } catch (error) {
    console.error('Error in deployMiniExtension:', error);
    alert(`Failed to deploy mini extension: ${error.message}\n\nNote: Some pages (like chrome:// URLs) cannot have extensions injected.`);
  }
}

/**
 * Show deployment preference dialog for new domain
 */
async function showDeploymentPreferenceDialog(activeTab, domain) {
  return new Promise((resolve) => {
    const dialog = document.getElementById('deploy-preference-dialog');
    const domainNameEl = document.getElementById('deploy-domain-name');
    const cancelBtn = document.getElementById('deploy-preference-cancel');
    const confirmBtn = document.getElementById('deploy-preference-confirm');

    // Set domain name in dialog
    domainNameEl.textContent = domain;

    // Show dialog
    dialog.classList.remove('hidden');

    // Handle cancel
    const handleCancel = () => {
      dialog.classList.add('hidden');
      cleanup();
      resolve(false);
    };

    // Handle confirm
    const handleConfirm = async () => {
      const selectedPreference = document.querySelector('input[name="deploy-preference"]:checked').value;
      console.log(`User selected preference: ${selectedPreference} for ${domain}`);

      // Store preference
      const result = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
      const preferences = result.miniExtensionDomainPreferences || {};
      preferences[domain] = selectedPreference;
      await chrome.storage.local.set({ miniExtensionDomainPreferences: preferences });

      dialog.classList.add('hidden');
      cleanup();

      // Perform deployment with selected preference
      await performDeployment(activeTab, selectedPreference);
      resolve(true);
    };

    // Cleanup function
    const cleanup = () => {
      cancelBtn.removeEventListener('click', handleCancel);
      confirmBtn.removeEventListener('click', handleConfirm);
    };

    // Add event listeners
    cancelBtn.addEventListener('click', handleCancel);
    confirmBtn.addEventListener('click', handleConfirm);
  });
}

/**
 * Perform the actual deployment
 */
async function performDeployment(activeTab, preference, silent = false) {
  const deployButton = document.getElementById('deploy-mini-button');
  const deployButtonText = document.getElementById('deploy-mini-button-text');
  const deployPanel = document.getElementById('deploy-mini-panel');

  try {
    // Disable button during deployment (only if not silent/auto)
    if (!silent) {
      deployButton.disabled = true;
      deployButtonText.textContent = 'Deploying...';
    }

    console.log(`Deploying mini extension to tab: ${activeTab.id} (silent: ${silent})`);

    // First, try to inject content script (will fail silently if already injected)
    try {
      await chrome.scripting.executeScript({
        target: { tabId: activeTab.id },
        files: ['content/content-script.js']
      });
      console.log('Content script injected successfully');

      // Wait for content script to initialize and set up listeners
      // Increased timeout to ensure listeners are ready
      await new Promise(resolve => setTimeout(resolve, 1000));
    } catch (scriptError) {
      // Content script might already be loaded, that's okay
      console.log('Content script injection skipped (may already be loaded):', scriptError.message);

      // Still wait a bit to ensure existing script is ready
      await new Promise(resolve => setTimeout(resolve, 300));
    }

    // Send message to deploy mini extension with preference
    let response;
    try {
      response = await chrome.tabs.sendMessage(activeTab.id, {
        action: 'DEPLOY_MINI_EXTENSION',
        preference: preference
      });
      console.log('Received response from content script:', response);
    } catch (messageError) {
      console.error('Error sending message to content script:', messageError);
      throw new Error(`Message sending failed: ${messageError.message}`);
    }

    if (response && response.success) {
      console.log('Mini extension deployed successfully');

      // Only show success toast if NOT silent OR if it wasn't already deployed (meaning we actually did something)
      // If it's silent (auto-check) AND it was already deployed (no-op), then stay silent.
      if (!silent || !response.alreadyDeployed) {
        showToast(response.alreadyDeployed ? 'Mini extension already on this page' : 'Mini extension deployed successfully!');
      }

      // Hide the deploy panel after successful deployment
      deployPanel.classList.add('hidden');
    } else {
      const errorMsg = response?.error || 'Unknown error';
      throw new Error(`Failed to deploy mini extension: ${errorMsg}`);
    }
  } catch (error) {
    console.error('Error deploying mini extension:', error);

    if (!silent) {
      deployButtonText.textContent = 'Deploy Failed';
      alert(`Failed to deploy mini extension: ${error.message}\n\nNote: Some pages (like chrome:// URLs) cannot have extensions injected.`);

      // Reset button
      setTimeout(() => {
        deployButtonText.textContent = 'Deploy Mini Extension';
        deployButton.disabled = false;
      }, 2000);
    }
  }
}

/**
 * Check if mini extension is already deployed on current tab
 */
async function checkMiniExtensionDeployment() {
  const deployPanel = document.getElementById('deploy-mini-panel');

  try {
    // Get current active tab
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!activeTab) {
      return;
    }

    // Get domain from URL
    const currentDomain = new URL(activeTab.url).hostname;

    // Check if domain has a preference
    const result = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
    const preferences = result.miniExtensionDomainPreferences || {};
    const domainPreference = preferences[currentDomain];

    // If domain has "always" preference, hide deploy button (will auto-deploy)
    if (domainPreference === 'always') {
      console.log(`Domain ${currentDomain} has "always" preference, hiding deploy button and auto-deploying`);
      deployPanel.classList.add('hidden');

      // Auto-deploy silently
      performDeployment(activeTab, 'always', true);
      return;
    }

    // For other cases, check if mini extension is currently deployed
    try {
      const response = await chrome.tabs.sendMessage(activeTab.id, {
        action: 'CHECK_MINI_EXTENSION_DEPLOYED'
      });

      if (response && response.deployed) {
        console.log('Mini extension already deployed on this tab, hiding deploy button');
        deployPanel.classList.add('hidden');
      } else {
        console.log('Mini extension not deployed, showing deploy button');
        deployPanel.classList.remove('hidden');
      }
    } catch (messageError) {
      // Content script not loaded - show deploy button
      console.log('Content script not available, showing deploy button');
      deployPanel.classList.remove('hidden');
    }
  } catch (error) {
    // Error getting tab or domain - show deploy button as fallback
    console.log('Error checking deployment status, showing deploy button:', error);
    deployPanel.classList.remove('hidden');
  }
}

function openTextRad() {
  // Open TextRad in a new tab (use root URL to ensure login page is shown if needed)
  chrome.tabs.create({ url: 'https://textrad.in/' });

  // Show a helpful message to the user
  const msgElement = document.getElementById('login-required-message');
  if (msgElement) {
    if (!msgElement.querySelector('.login-instruction')) {
      const instruction = document.createElement('p');
      instruction.className = 'login-instruction mt-1 text-green-300';
      instruction.textContent = 'TextRad opened in a new tab. Please log in there and then try again.';
      msgElement.querySelector('div').appendChild(instruction);
    }
  }
}

/**
 * Open TextRAD App page (https://textrad.in/app/)
 * Navigates existing TextRAD tab in background or creates new one
 */
async function openTextRadApp() {
  try {
    // Query all tabs
    const tabs = await chrome.tabs.query({});

    // First, check if there's already an app page tab open
    const appTab = tabs.find(tab =>
      tab.url && (
        tab.url.includes('textrad.in/app') ||
        tab.url.includes('textrad.com/app') ||
        (tab.url.includes('localhost') && tab.url.includes('/app'))
      )
    );

    if (appTab) {
      // App page already open, just notify user
      showToast('TextRAD App page is already open');
      console.log('TextRAD app tab already exists');
    } else {
      // Check if there's any TextRAD tab (main site, etc.) that we can navigate
      const textradTab = tabs.find(tab =>
        tab.url && (
          tab.url.includes('textrad.in') ||
          tab.url.includes('textrad.com') ||
          (tab.url.includes('localhost') && !tab.url.includes('/app'))
        )
      );

      if (textradTab) {
        // Navigate existing TextRAD tab to /app/ (in background, no focus)
        await chrome.tabs.update(textradTab.id, { url: 'https://textrad.in/app/' });
        showToast('TextRAD tab navigated to App page');
        console.log('Navigated existing TextRAD tab to /app/');
      } else {
        // No TextRAD tabs found, create new one
        await chrome.tabs.create({ url: 'https://textrad.in/app/', active: false });
        showToast('TextRAD App opened in new tab');
        console.log('Opened new TextRAD app tab');
      }
    }

    // Re-check app page status after a short delay
    setTimeout(() => {
      checkAppPageStatus();
    }, 1500);
  } catch (error) {
    console.error('Error opening TextRAD app:', error);
    showToast('Failed to open TextRAD App');
  }
}

/**
 * Copy transcription to clipboard, inject into web app, auto-save and reset
 */
async function copyAndSaveReport() {
  // Check if we have a transcription
  if (!transcriptionText.value) {
    alert('No transcription to copy');
    return;
  }

  // Ensure currentTranscription is updated with the textarea value
  // This ensures send will work even after copy
  currentTranscription = transcriptionText.value;

  // Determine if we're copying a generated report
  const isCopyingReport = currentReportContent !== null;

  // Get the content to copy (with appropriate formatting)
  const contentToCopy = HTMLFormatter.getOptimalCopyFormat(
    transcriptionText.value,
    isCopyingReport,
    selectedFormat ? selectedFormat.name : null
  );

  try {
    // Copy to clipboard (HTML or plain text depending on content)
    if (isCopyingReport && HTMLFormatter.isMarkdownFormatted(transcriptionText.value)) {
      // For reports, copy as HTML to preserve formatting
      await copyHTMLToClipboard(contentToCopy, transcriptionText.value);
    } else {
      // For regular transcriptions, copy as plain text
      await navigator.clipboard.writeText(contentToCopy);
    }

    // Enhanced button animation feedback - just animation, no text change
    copyButton.classList.add('copy-success-animation');

    // Enhanced feedback and auto-save for report copying
    if (isCopyingReport) {
      // First, inject report into web app's report panel
      await injectReportToWebApp();

      // Auto-save the report to web app
      const saveResult = await autoSaveReport();

      // Show success message for copy operation - adjust message based on save result
      if (saveResult) {
        showToast('Report copied with formatting and saved to My Reports!', 3000);
      } else {
        showToast('Report copied with formatting!', 3000);
      }

      // Provide immediate UI feedback by starting the collapse transition
      const expandedContainer = document.getElementById('expanded-container');
      if (expandedContainer) {
        // Add a brief visual feedback class
        expandedContainer.classList.add('state-transitioning');

        // Immediately remove report-related classes to start collapse transition
        expandedContainer.classList.remove('has-report', 'report-mode', 'report-expanded');
        console.log('🎛️ Starting UI collapse immediately after copy');

        // Remove transition class after animation
        setTimeout(() => {
          expandedContainer.classList.remove('state-transitioning');
        }, 400);
      }

      // Mark reset operation as in progress and persist it
      isResetInProgress = true;
      chrome.storage.local.set({
        resetInProgress: true,
        resetTimestamp: Date.now()
      });
      console.log('🔄 Reset operation started and persisted');

      // Auto-reset after copying a generated report (with shorter delay for better UX)
      setTimeout(() => {
        completeResetOperation();
      }, 800);
    }
    // No toast notification for regular transcriptions - just button animation

    // Reset button animation after 1.5 seconds
    setTimeout(() => {
      copyButton.classList.remove('copy-success-animation');
    }, 1500);
  } catch (err) {
    console.error('Could not copy text: ', err);

    // Show appropriate error message
    if (isCopyingReport) {
      showToast('Copy succeeded but failed to save to My Reports. Please check your login status.', 4000);
    } else {
      alert('Failed to copy: ' + err.message);
    }
  }
}

/**
 * Copy HTML content to clipboard with fallback to plain text
 */
async function copyHTMLToClipboard(htmlContent, plainTextFallback) {
  try {
    // Try to copy as HTML first (for rich text editors)
    const clipboardItem = new ClipboardItem({
      'text/html': new Blob([htmlContent], { type: 'text/html' }),
      'text/plain': new Blob([plainTextFallback], { type: 'text/plain' })
    });

    await navigator.clipboard.write([clipboardItem]);
  } catch (error) {
    console.warn('HTML copy failed, falling back to plain text:', error);
    // Fallback to plain text copy
    await navigator.clipboard.writeText(plainTextFallback);
  }
}

/**
 * Auto-save generated report to web app's "My Reports" section
 */
async function autoSaveReport() {
  if (!reportStorage) {
    console.warn('Report storage not initialized, cannot auto-save');
    return false;
  }

  // Check if user is authenticated
  if (!reportStorage.isUserAuthenticated()) {
    console.warn('User not authenticated, cannot auto-save report');
    // Skip login message for copy operations - just silently fail
    return false;
  }

  try {
    // Extract the actual format name from the first line of the report
    const reportContent = transcriptionText.value;
    let extractedFormatName = selectedFormat ? selectedFormat.name : 'Extension Report';

    // Try to extract title from first line (patterns: **TITLE** or # TITLE)
    const firstLine = reportContent.split('\n')[0].trim();
    if (firstLine) {
      // Check for markdown bold pattern: **TITLE**
      const boldMatch = firstLine.match(/^\*\*(.+?)\*\*$/);
      if (boldMatch) {
        extractedFormatName = boldMatch[1].trim();
      }
      // Check for markdown heading pattern: # TITLE
      else if (firstLine.startsWith('#')) {
        extractedFormatName = firstLine.replace(/^#+\s*/, '').trim();
      }
    }

    // Prepare report data
    const reportData = {
      content: reportContent,
      formatName: extractedFormatName,
      source: 'extension'
    };

    // Save report and update usage counter
    const result = await reportStorage.saveReportAndUpdateUsage(reportData);

    if (result.success) {
      console.log('Report auto-saved successfully:', result.reportId);
      return true;
    } else {
      console.warn('Failed to auto-save report:', result.reason);

      // Show specific error messages
      if (result.reason === 'limit_reached') {
        showToast('Report limit reached. Please upgrade your subscription.', 4000);
      } else if (result.reason === 'not_authenticated') {
        // Skip authentication message for copy operations - just silently fail
        console.warn('Authentication issue during auto-save, skipping error message');
      } else {
        showToast('Failed to save report to My Reports.', 3000);
      }

      return false;
    }
  } catch (error) {
    console.error('Error during auto-save:', error);
    showToast('Error saving report. Please try again later.', 3000);
    return false;
  }
}

/**
 * Check extension state
 */
async function checkState() {
  chrome.runtime.sendMessage({ action: 'GET_STATE' }, async (response) => {
    if (chrome.runtime.lastError) {
      console.error('Error getting state:', chrome.runtime.lastError);
      return;
    }

    if (response && response.state) {
      const state = response.state;

      // Check recording state
      if (state.recording) {
        // We were recording, but popup was closed
        showRecordingUI();
      } else if (state.transcribing) {
        // We were transcribing
        showTranscribingUI();
      } else if (state.lastTranscription) {
        // We have a transcription
        showTranscriptionUI(state.lastTranscription);
        // Wait for TextRad tabs to be detected before loading templates
        setTimeout(() => {
          loadAvailableFormats();
        }, 1000);
      }
    }
  });
}

/**
 * Toggle recording state
 */
function toggleRecording() {
  // First check if we're connected to TextRad and logged in
  if (!isUserLoggedIn()) {
    console.log('Cannot record: User not logged in to TextRad');
    updateAppStatus('not-logged-in');

    // Show prominent message to user
    showToast('Please log in to TextRad first', 3000);

    // Ensure login message is visible
    const loginMsg = document.getElementById('login-required-message');
    if (loginMsg) {
      loginMsg.classList.remove('hidden');
      loginMsg.style.display = 'block';
      loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }

    return;
  }

  if (isRecording) {
    stopRecording();
  } else {
    // For the main record button, we want to append to existing text
    // Set the flag to indicate we want to append
    recordMoreButtonClicked = true;

    // Store current transcription before starting new recording
    if (transcriptionText && transcriptionText.value) {
      const previousTranscription = transcriptionText.value;
      currentTranscription = previousTranscription;
      console.log('Main record button clicked - storing transcription for append:',
        previousTranscription.substring(0, 50) + '...');
    }

    startRecording();
  }
}

/**
 * Check if user is logged in to TextRad
 */
function isUserLoggedIn() {
  const now = Date.now();
  const cacheKey = 'login_status';

  // Check cache first
  if (loginStatusCache.has(cacheKey)) {
    const cached = loginStatusCache.get(cacheKey);
    if (now - cached.timestamp < LOGIN_CACHE_TTL) {
      return cached.value;
    }
    // Cache expired, remove it
    loginStatusCache.delete(cacheKey);
  }

  // Compute fresh login status
  // If we have textradTabs with explicitly logged-in status, then the user is logged in
  const hasLoggedInTabs = textradTabs && textradTabs.length > 0 &&
    textradTabs.some(tab => tab.loggedIn === true);

  let loginStatus = hasLoggedInTabs;

  // Additional check: if we're currently showing transcription content,
  // don't consider user as logged out (prevents flickering)
  if (!hasLoggedInTabs) {
    const transcriptionText = document.getElementById('transcription-text');
    const hasActiveContent = transcriptionText && transcriptionText.value && transcriptionText.value.trim().length > 0;
    const transcriptionPanel = document.getElementById('transcription-panel');
    const isTranscriptionVisible = transcriptionPanel && !transcriptionPanel.classList.contains('hidden');

    // If user has active transcription content visible, assume they're logged in to prevent UI disruption
    if (hasActiveContent && isTranscriptionVisible) {
      console.log('User has active transcription content, assuming logged in to prevent UI disruption');
      loginStatus = true;
    }
  }

  // Cache the result
  loginStatusCache.set(cacheKey, {
    value: loginStatus,
    timestamp: now
  });

  return loginStatus;
}

/**
 * Start recording (behavior depends on which button initiated it)
 */
async function startRecording() {
  try {
    // Perform fresh login check right before recording
    console.log('Checking login status before starting recording...');

    // Force a fresh login check
    const now = Date.now();
    lastLoginCheck = now;

    const loginCheckResponse = await new Promise((resolve) => {
      chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
        resolve(response);
      });
    });

    // Validate login response
    if (!loginCheckResponse || !loginCheckResponse.success ||
      !loginCheckResponse.tabs || loginCheckResponse.tabs.length === 0) {
      updateAppStatus('not-logged-in');
      throw new Error('You must be logged in to TextRad to record and transcribe audio');
    }

    // Check if user is actually logged in to any tabs
    const hasLoggedInTabs = loginCheckResponse.tabs.some(tab =>
      loginCheckResponse.loggedInTabs && loginCheckResponse.loggedInTabs[tab.id] === true
    );

    if (!hasLoggedInTabs) {
      updateAppStatus('not-logged-in');
      throw new Error('You must be logged in to TextRad to record and transcribe audio');
    }

    // Update tabs with fresh login data
    textradTabs = loginCheckResponse.tabs.map(tab => ({
      ...tab,
      loggedIn: loginCheckResponse.loggedInTabs && loginCheckResponse.loggedInTabs[tab.id] === true
    }));

    console.log('Login check passed, proceeding with recording');

    // Update UI
    recordButton.disabled = true;
    recordingStatus.textContent = 'Starting...';

    // The recordMoreButtonClicked flag is set in toggleRecording (normal record button)
    // and explicitly set to false in startNewDictation
    // So we don't need to do anything else with the flag here
    console.log('Starting recording with append mode:', recordMoreButtonClicked ? 'true' : 'false');

    /* 
     * Background recording has been disabled - we are not using it anymore
     * Only use direct recording through the audioRecorder
     */

    // If background recording fails, fall back to local recording
    console.log('Falling back to local audio recording');
    const success = await audioRecorder.startRecording();

    if (!success) {
      throw new Error('Failed to start recording');
    }
  } catch (error) {
    console.error('Error starting recording:', error);

    // If this is a login error, update UI appropriately
    if (error.message.includes('logged in')) {
      updateAppStatus('not-logged-in');
    } else {
      recordingStatus.textContent = 'Error: ' + error.message;
    }

    recordButton.disabled = false;
  }
}

/**
 * Send a message with timeout
 */
function sendMessageWithTimeout(message, timeout = 3000) {
  return new Promise((resolve, reject) => {
    const timer = setTimeout(() => {
      reject(new Error(`Message timeout after ${timeout}ms`));
    }, timeout);

    chrome.runtime.sendMessage(message, (response) => {
      clearTimeout(timer);
      if (chrome.runtime.lastError) {
        reject(chrome.runtime.lastError);
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * Stop recording
 */
async function stopRecording() {
  try {
    // Update UI
    recordButton.disabled = true;
    recordingStatus.textContent = 'Stopping...';

    /* 
     * Background recording has been disabled - we are not using it anymore
     * Only use direct recording through the audioRecorder
     */

    // Fall back to local recording if background recording fails
    console.log('Falling back to local audio recorder stop');
    audioRecorder.stopRecording();

    // Ensure recording in progress flag is cleared
    if (storageManager) {
      storageManager.setRecordingInProgress(false);
    }
  } catch (error) {
    console.error('Error stopping recording:', error);
    recordingStatus.textContent = 'Error: ' + error.message;
    recordButton.disabled = false;
  }
}

/**
 * Start a brand new dictation (clear previous text)
 */
function startNewDictation() {
  // First check if we're connected to TextRad and logged in
  if (!isUserLoggedIn()) {
    console.log('Cannot start new dictation: User not logged in to TextRad');
    updateAppStatus('not-logged-in');

    // Show prominent message to user
    showToast('Please log in to TextRad first', 3000);

    // Ensure login message is visible
    const loginMsg = document.getElementById('login-required-message');
    if (loginMsg) {
      loginMsg.classList.remove('hidden');
      loginMsg.style.display = 'block';
      loginMsg.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }

    return;
  }

  // EXPLICITLY Reset flag to indicate we're starting fresh
  recordMoreButtonClicked = false;

  // Clear the current transcription (local and in services)
  // IMPORTANT: Set both to empty strings to ensure complete clearing
  currentTranscription = '';
  currentReportContent = null;

  // Reset template state
  resetTemplateState();

  // Reset transcription in all services
  transcriptionService.resetTranscription();
  transcriptionService.setCurrentTranscription('');

  // Reset the DirectTranscriptionService if it's being imported
  try {
    // This is an async import but we'll just fire it and let it complete
    import('../lib/direct-transcription-service.js').then(module => {
      const DirectTranscriptionService = module.DirectTranscriptionService;
      const directService = new DirectTranscriptionService();
      directService.resetTranscription();
      directService.setCurrentTranscription('');
    }).catch(err => {
      console.error('Error resetting DirectTranscriptionService:', err);
    });
  } catch (err) {
    console.error('Error importing DirectTranscriptionService:', err);
  }

  // Explicitly clear the UI text area
  if (transcriptionText) {
    transcriptionText.value = '';
    // Reset textarea height
    autoResizeTextarea(transcriptionText);
  }

  // Also reset the stored value in the service worker and disable append mode
  chrome.runtime.sendMessage({
    action: 'NEW_DICTATION'
  });

  console.log('New Dictation clicked - cleared previous transcription');

  // Hide transcription panel
  transcriptionPanel.classList.add('hidden');

  // Show recording panel
  document.getElementById('recording-panel').classList.remove('hidden');

  // Reset container height by removing all state classes
  const expandedContainer = document.getElementById('expanded-container');
  if (expandedContainer) {
    expandedContainer.classList.remove('has-transcription', 'has-report', 'format-dropdown-active', 'report-mode', 'report-expanded');
    console.log('🎛️ Container state classes cleared for new dictation');
  }

  // Hide template and report generation panels
  const templatePanel = document.getElementById('template-selection-panel');
  const generatePanel = document.getElementById('generate-report-panel');
  if (templatePanel) templatePanel.classList.add('hidden');
  if (generatePanel) generatePanel.classList.add('hidden');

  // Make sure the full UI is shown before starting a new recording
  const fullRecordingUI = document.getElementById('full-recording-ui');
  if (fullRecordingUI) {
    fullRecordingUI.classList.remove('hidden');
  }
  appStatusPanel.classList.remove('hidden');

  // Force a layout recalculation to ensure smooth transition
  if (expandedContainer) {
    expandedContainer.offsetHeight; // Trigger reflow
  }

  // DO NOT start recording automatically - let user press the mic button
  // Just reset the UI to recording-ready state
  recordingStatus.textContent = 'Ready to record';
  recordButton.disabled = false;
  recordButton.classList.remove('recording');

  // Add visual feedback for completion
  console.log('✨ New dictation ready - UI reset complete');
}

/**
 * Smooth panel transition helper
 */
function smoothPanelTransition(element, show, callback = null) {
  if (!element) return;

  if (show) {
    // Show panel with animation
    element.style.display = 'block';
    element.classList.remove('hidden');
    // Force reflow
    element.offsetHeight;
    // Add smooth transition class if needed
    element.style.opacity = '1';
    element.style.transform = 'translateY(0)';
  } else {
    // Hide panel with animation
    element.style.opacity = '0';
    element.style.transform = 'translateY(-10px)';
    setTimeout(() => {
      element.classList.add('hidden');
      element.style.display = 'none';
      if (callback) callback();
    }, 300); // Match CSS transition duration
  }
}

/**
 * Complete the reset operation and clear persistence
 */
function completeResetOperation() {
  console.log('🔄 Completing reset operation...');

  // Call the original startNewDictation function
  startNewDictation();

  // Clear the reset state from storage and memory
  isResetInProgress = false;
  chrome.storage.local.remove(['resetInProgress', 'resetTimestamp']);

  console.log('✅ Reset operation completed and cleared');
}

/**
 * Check and handle any pending reset operations
 */
async function checkPendingResetOperation() {
  try {
    const result = await chrome.storage.local.get(['resetInProgress', 'resetTimestamp']);

    if (result.resetInProgress) {
      const now = Date.now();
      const timeSinceReset = now - (result.resetTimestamp || 0);

      console.log(`🔍 Found pending reset operation from ${timeSinceReset}ms ago`);

      // If reset was started recently (within 5 seconds), complete it
      if (timeSinceReset < 5000) {
        console.log('🔄 Completing interrupted reset operation...');
        completeResetOperation();
      } else {
        console.log('⚠️ Reset operation was too old, clearing it...');
        chrome.storage.local.remove(['resetInProgress', 'resetTimestamp']);
        isResetInProgress = false;
      }
    }
  } catch (error) {
    console.error('Error checking pending reset operation:', error);
  }
}

/**
 * Send transcription to TextRad
 */
function sendTranscription() {
  // Check if we have a transcription
  if (!currentTranscription) {
    alert('No transcription to send');
    return;
  }

  // Check the connection state first
  updateAppStatus('connecting');

  // Find TextRad tabs with verified login state
  chrome.runtime.sendMessage({ action: 'FIND_TEXTRAD_TABS' }, (response) => {
    if (chrome.runtime.lastError) {
      console.error('Error finding TextRad tabs:', chrome.runtime.lastError);
      hideLoading();
      updateAppStatus('disconnected');
      alert('Error connecting to TextRad: ' + chrome.runtime.lastError.message);
      return;
    }

    // Check for login issues
    if (response && response.error && response.error.includes('not logged in')) {
      console.log('User not logged in to TextRad');
      hideLoading();
      updateAppStatus('not-logged-in');
      return;
    }

    // Check if we have logged-in TextRad tabs
    if (response && response.success && response.tabs && response.tabs.length > 0) {
      // Store tabs with login information
      textradTabs = response.tabs.map(tab => ({
        ...tab,
        loggedIn: response.loggedInTabs && response.loggedInTabs[tab.id] === true
      }));

      // Only use logged-in tabs
      const loggedInTabs = textradTabs.filter(tab => tab.loggedIn);
      if (loggedInTabs.length > 0) {
        selectedTabId = loggedInTabs[0].id;
        updateAppStatus('connected');

        // Show loading state
        showLoading('Sending to TextRad...');

        // Send transcription to TextRad tab
        chrome.runtime.sendMessage({
          action: 'SEND_TRANSCRIPTION_TO_TAB',
          tabId: selectedTabId,
          transcription: currentTranscription
        }, (response) => {
          // Hide loading state
          hideLoading();

          if (chrome.runtime.lastError) {
            console.error('Error sending transcription:', chrome.runtime.lastError);
            showToast('Error: ' + chrome.runtime.lastError.message, 3000);
            return;
          }

          if (response && response.success) {
            // Success - show subtle toast notification
            showToast('✓ Sent to TextRad', 1500);

            // Update button to show temporary success state
            const originalHTML = sendButton.innerHTML;
            sendButton.innerHTML = '<i class="fas fa-check"></i>';
            sendButton.classList.add('bg-green-700');

            // Reset button after a delay
            setTimeout(() => {
              sendButton.innerHTML = originalHTML;
              sendButton.classList.remove('bg-green-700');
            }, 1500);
          } else if (response && response.error && response.error.includes('not logged in')) {
            // Login error
            updateAppStatus('not-logged-in');
          } else {
            // Other error
            showToast('Error: ' + (response?.error || 'Unknown error'), 3000);
          }
        });
      } else {
        // We found TextRad tabs, but user isn't logged in to any
        hideLoading();
        updateAppStatus('not-logged-in');
      }
    } else {
      // No TextRad tabs found or not logged in
      hideLoading();

      if (response && response.hasTextRadTabs && response.notLoggedInCount > 0) {
        // Found TextRad tabs but user isn't logged in
        updateAppStatus('not-logged-in');
      } else {
        // No TextRad tabs at all
        updateAppStatus('disconnected');

        // Ask if user wants to open TextRad
        if (confirm('No TextRad app detected. Open TextRad in a new tab?')) {
          openTextRad();
        }
      }
    }
  });
}

/**
 * Handle recording start
 */
function handleRecordingStart() {
  console.log('Recording started');

  // Update state
  isRecording = true;
  recordingStartTime = Date.now();

  // Update UI
  showRecordingUI();

  // Start recording timer
  startRecordingTimer();
}

/**
 * Handle recording stop
 */
function handleRecordingStop(audioBlob) {
  console.log('Recording stopped, blob size:', audioBlob.size);

  // Update state
  isRecording = false;

  // Stop recording timer
  stopRecordingTimer();

  // Update UI
  showTranscribingUI();

  // Check login status again before transcribing
  if (!isUserLoggedIn()) {
    console.log('Cannot transcribe: User not logged in to TextRad');
    handleTranscriptionError(new Error('You must be logged in to TextRad to transcribe audio'));
    updateAppStatus('not-logged-in');
    return;
  }

  // Start transcription
  if (audioBlob && audioBlob.size > 0) {
    transcriptionService.transcribeAudio(audioBlob)
      .catch(error => {
        // Extra error handling for login issues
        if (error.message.includes('logged in') || error.message.includes('login')) {
          updateAppStatus('not-logged-in');
        }
      });
  } else {
    handleTranscriptionError(new Error('No audio recorded'));
  }
}

/**
 * Handle silence detected
 */
function handleSilenceDetected() {
  console.log('Silence detected, auto-stopping recording');

  // Update UI
  recordingStatus.textContent = 'Silence detected, stopping...';

  // We don't need to call stopRecording() here as the audioRecorder
  // will automatically call onRecordingStop which we handle above
}

/**
 * Handle recording error
 */
function handleRecordingError(error) {
  console.error('Recording error:', error);

  // Update state
  isRecording = false;

  // Stop recording timer
  stopRecordingTimer();

  // Update UI
  recordingStatus.textContent = 'Error: ' + error.message;
  recordButton.disabled = false;
  recordButton.classList.remove('recording');
}

/**
 * Handle transcription start
 */
function handleTranscriptionStart() {
  console.log('Transcription started');

  // Update state
  isTranscribing = true;

  // Update UI
  recordingStatus.textContent = 'Transcribing...';
}

/**
 * Handle transcription complete
 */
function handleTranscriptionComplete(text) {
  console.log('Transcription complete:', text.substring(0, 50) + '...');

  // Update state
  isTranscribing = false;

  // Check if we need to append to previous transcription
  let finalText = text;
  const previousTranscription = currentTranscription || '';

  console.log('recordMoreButtonClicked flag:', recordMoreButtonClicked);
  console.log('Has previous transcription:', previousTranscription.trim() !== '');

  // Only append if we explicitly stored a previous transcription AND the recordMoreButtonClicked flag is true
  if (recordMoreButtonClicked === true && previousTranscription && previousTranscription.trim() !== '') {
    // Append new transcription to the previous one with a space in between
    finalText = previousTranscription + ' ' + text;
    console.log('Appended to previous transcription, result:', finalText.substring(0, 50) + '...');
  } else {
    console.log('Not appending - using just the new transcription');
    // When recordMoreButtonClicked is explicitly false (set by New Dictation), 
    // make sure we use only the new transcription
    finalText = text;
    currentTranscription = '';
  }

  // Always reset the recordMoreButtonClicked flag after use
  recordMoreButtonClicked = false;

  // Update current transcription
  currentTranscription = finalText;

  // Reset the service's stored transcription to avoid duplication
  transcriptionService.setCurrentTranscription('');

  // Update UI
  showTranscriptionUI(finalText);

  // Load templates after transcription is complete
  loadAvailableFormats();

  // Notify background script
  chrome.runtime.sendMessage({
    action: 'SET_TRANSCRIPTION',
    transcription: finalText
  });

  // Check if auto-send is enabled
  if (autoSendSetting.checked && textradTabs.length > 0) {
    sendTranscription();
  }
}

/**
 * Handle transcription error
 */
function handleTranscriptionError(error) {
  console.error('Transcription error:', error);

  // Update state
  isTranscribing = false;

  // Different UI handling based on error type
  if (error.message && (
    error.message.includes('audio') ||
    error.message.includes('speech') ||
    error.message.includes('short') ||
    error.message.includes('empty') ||
    error.message.includes('recording'))) {
    // Audio-related error - don't show login prompt
    recordingStatus.textContent = 'Recording issue: ' + error.message;
  } else if (error.message && error.message.includes('timed out')) {
    // Timeout error
    recordingStatus.textContent = 'Timeout: ' + error.message;
  } else if (error.message && (
    error.message.includes('login') ||
    error.message.includes('logged in'))) {
    // Login-related error - update login status
    recordingStatus.textContent = 'Login required';
    updateAppStatus('not-logged-in');
  } else {
    // Generic error
    recordingStatus.textContent = 'Transcription error: ' + error.message;
  }

  recordButton.disabled = false;
  hideLoading();
}

/**
 * Start recording timer (simplified, just tracks start time)
 */
function startRecordingTimer() {
  // We still track recording start time for other purposes
  // but we no longer display it
  recordingStartTime = Date.now();
}

/**
 * Stop recording timer
 */
function stopRecordingTimer() {
  if (recordingTimer) {
    clearInterval(recordingTimer);
    recordingTimer = null;
  }
}

/**
 * Show recording UI - Simple approach that always works
 */
function showRecordingUI() {
  // Update record button
  recordButton.disabled = false;
  recordButton.classList.add('recording');
  recordButton.querySelector('i').className = 'fas fa-stop';

  // Update status
  recordingStatus.textContent = 'Recording...';

  // Hide transcription panel
  transcriptionPanel.classList.add('hidden');

  // Ensure recording panel is visible
  document.getElementById('recording-panel').classList.remove('hidden');

  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show transcribing UI
 */
function showTranscribingUI() {
  // Update record button
  recordButton.disabled = true;
  recordButton.classList.remove('recording');
  recordButton.querySelector('i').className = 'fas fa-microphone';

  // Update status
  recordingStatus.textContent = 'Transcribing...';

  // Show loading
  showLoading('Transcribing audio...');

  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show transcription UI
 */
function showTranscriptionUI(text) {
  // Hide loading
  hideLoading();

  // Update record button
  recordButton.disabled = false;
  recordButton.classList.remove('recording');
  recordButton.querySelector('i').className = 'fas fa-microphone';

  // Update status
  recordingStatus.textContent = 'Ready to record';

  // Button is already in its fixed position - no need to move it

  // Show recording panel again
  document.getElementById('recording-panel').classList.remove('hidden');

  // Show other UI elements
  appStatusPanel.classList.remove('hidden');
  settingsButton.classList.remove('hidden');

  // Update transcription text
  transcriptionText.value = text;

  // Show transcription panel
  transcriptionPanel.classList.remove('hidden');

  // Show quick generate button when we have transcription
  if (quickGenerateContainer) {
    quickGenerateContainer.classList.remove('hidden');
  }

  updateContainerState();

  // Update mini mode status if it's visible
  updateMiniStatus();
}

/**
 * Show loading overlay
 */
function showLoading(message) {
  loadingText.textContent = message || 'Processing...';
  loadingOverlay.classList.remove('hidden');
}

/**
 * Hide loading overlay
 */
function hideLoading() {
  loadingOverlay.classList.add('hidden');
}

/**
 * Show toast notification
 * @param {string} message - Message to display
 * @param {number} duration - How long to show in ms (default 2000ms)
 */
function showToast(message, duration = 2000) {
  if (!toastNotification || !toastMessage) return;

  // Set message
  toastMessage.textContent = message;

  // Show toast
  toastNotification.classList.remove('hidden');

  // Use setTimeout to ensure the transition works
  setTimeout(() => {
    toastNotification.classList.add('show');
  }, 10);

  // Auto hide after duration
  setTimeout(() => {
    // Start fade out
    toastNotification.classList.remove('show');
    toastNotification.classList.add('hide');

    // Actually hide after transition completes
    setTimeout(() => {
      toastNotification.classList.remove('hide');
      toastNotification.classList.add('hidden');
    }, 300);
  }, duration);
}

/**
 * Toggle settings panel
 */
function toggleSettings() {
  if (settingsPanel.classList.contains('hidden')) {
    // Show settings
    settingsPanel.classList.remove('hidden');
  } else {
    // Hide settings
    settingsPanel.classList.add('hidden');
  }
}

/**
 * Toggle between expanded and mini modes
 */
function toggleViewMode() {
  // Check current state
  const isExpanded = !expandedContainer.classList.contains('hidden');

  if (isExpanded) {
    // Switch to mini mode
    expandedContainer.classList.add('hidden');
    collapsedContainer.classList.remove('hidden');
    document.body.classList.add('mini-mode');
    document.documentElement.classList.add('mini-mode');

    // Also notify the background service worker that we're in mini mode
    chrome.runtime.sendMessage({
      action: 'SET_VIEW_MODE',
      mode: 'mini'
    });

    // Save in local storage to restore state if popup is closed
    chrome.storage.local.set({ viewMode: 'mini' });

    // Force a resize to minimum size
    window.resizeTo(150, 40);
  } else {
    // Switch to expanded mode
    expandedContainer.classList.remove('hidden');
    collapsedContainer.classList.add('hidden');
    document.body.classList.remove('mini-mode');
    document.documentElement.classList.remove('mini-mode');

    // Also notify the background service worker that we're in expanded mode
    chrome.runtime.sendMessage({
      action: 'SET_VIEW_MODE',
      mode: 'expanded'
    });

    // Save in local storage
    chrome.storage.local.set({ viewMode: 'expanded' });

    // Restore normal size
    window.resizeTo(450, 350);
  }

  // Update mini mode status indicator based on current state
  updateMiniStatus();
}

/**
 * Handle popup closing
 */
function handlePopupClosing(event) {
  console.log('Popup is closing (beforeunload event)');

  // If recording is in progress, we need to do something
  if (isRecording) {
    console.log('Recording was in progress - notifying service worker');

    // Notify the background script to maintain recording state
    try {
      chrome.runtime.sendMessage({
        action: 'MAINTAIN_RECORDING',
        inProgress: true
      });
    } catch (error) {
      console.error('Failed to notify background script about recording:', error);
    }
  }

  // If reset operation is in progress, ensure it's persisted
  if (isResetInProgress) {
    console.log('🔄 Reset operation in progress during popup close - ensuring persistence');
    try {
      chrome.storage.local.set({
        resetInProgress: true,
        resetTimestamp: Date.now()
      });
    } catch (error) {
      console.error('Failed to persist reset state:', error);
    }
  }
}

/**
 * Actually close the popup (when close button is clicked)
 */
function closePopup() {
  // Explicitly stop any recording first
  if (isRecording) {
    stopRecording();
  }

  // Then close the window
  window.close();
}

/**
 * Update mini mode status indicator based on current state
 */
function updateMiniStatus() {
  // Remove all classes first
  collapsedStatus.className = 'status-dot';

  if (isRecording) {
    // If recording, show red blinking status
    collapsedStatus.classList.add('recording');

    // Also update button
    miniRecordButton.classList.add('recording');
    miniRecordButton.querySelector('i').className = 'fas fa-stop';
  } else if (isTranscribing) {
    // If transcribing, show blue blinking status
    collapsedStatus.classList.add('transcribing');
    miniRecordButton.querySelector('i').className = 'fas fa-microphone';
    miniRecordButton.classList.remove('recording');
  } else {
    // Normal state - just show connection status
    miniRecordButton.querySelector('i').className = 'fas fa-microphone';
    miniRecordButton.classList.remove('recording');

    // Use the same status as in the app status panel
    if (appStatusIndicator && appStatusIndicator.classList.contains('connected')) {
      collapsedStatus.classList.add('connected');
    } else if (appStatusIndicator && appStatusIndicator.classList.contains('connecting')) {
      collapsedStatus.classList.add('connecting');
    } else {
      collapsedStatus.classList.add('disconnected');
    }
  }
}

/**
 * Update silence timeout value display
 */
function updateSilenceTimeoutValue() {
  const value = silenceTimeoutSetting.value;
  silenceTimeoutValue.textContent = `${value}s`;
}

/**
 * Apply extension size scaling to CSS variables and body
 */
function applyExtensionSize() {
  // Default to 1.0 if element not available or invalid value
  let scaleValue = 1.0;

  // Get value from checked radio button
  const checkedRadio = document.querySelector('.extension-size-radio:checked');
  if (checkedRadio) {
    scaleValue = parseFloat(checkedRadio.value);
    // Validate the scale value
    if (isNaN(scaleValue) || scaleValue <= 0) {
      scaleValue = 1.0;
    }
  }

  document.documentElement.style.setProperty('--scale-multiplier', scaleValue.toString());

  const body = document.body;

  if (scaleValue !== 1.0) {
    body.classList.add('scaled');
  } else {
    body.classList.remove('scaled');
  }
}

// Removed transparency functions - using fixed opacity

/**
 * Load settings from storage
 */
async function loadSettings() {
  const settings = await storageManager.getSettings();

  // Apply settings
  autoSendSetting.checked = settings.autoSend;
  silenceTimeoutSetting.value = Math.floor(settings.silenceTimeout / 1000);
  transcriptionProviderSetting.value = settings.transcriptionProvider || 'auto';

  // Load extension size - default to 1.0 if not set
  const savedSize = settings.extensionSize || 1.0;

  // Set the appropriate radio button as checked
  const radioToCheck = document.querySelector(`.extension-size-radio[value="${savedSize}"]`);
  if (radioToCheck) {
    radioToCheck.checked = true;
  }

  console.log('Loaded extension size:', savedSize);

  // Update displayed values
  updateSilenceTimeoutValue();

  // Apply extension size immediately on load
  applyExtensionSize();

  // Load domain preferences for management
  await loadDomainList();

  // Apply settings to services
  if (audioRecorder) {
    audioRecorder.silenceTimeout = settings.silenceTimeout;
  }
}

/**
 * Load and display domain preferences
 */
async function loadDomainList() {
  const result = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
  const preferences = result.miniExtensionDomainPreferences || {};
  const domainList = document.getElementById('domain-list');

  // Clear existing list
  domainList.innerHTML = '';

  // Show ALL domains (both 'always' and 'session')
  const allDomains = Object.entries(preferences);

  if (allDomains.length === 0) {
    domainList.innerHTML = '<div class="text-xs text-gray-500 italic p-2">No domains with mini extension preferences</div>';
    return;
  }

  // Create domain items
  allDomains.forEach(([domain, preference]) => {
    const domainItem = document.createElement('div');
    domainItem.className = 'flex flex-col gap-1 p-2 bg-gray-700 rounded text-xs mb-1';

    const isAlways = preference === 'always';
    const badgeClass = isAlways ? 'bg-green-600' : 'bg-blue-600';
    const badgeText = isAlways ? 'Always' : 'Session';
    const changeText = isAlways ? 'Change to Session' : 'Change to Always';

    domainItem.innerHTML = `
      <div class="flex items-center justify-between">
        <span class="text-gray-200 font-medium">${domain}</span>
        <button class="remove-domain-btn text-red-400 hover:text-red-300" data-domain="${domain}">
          <i class="fas fa-times"></i>
        </button>
      </div>
      <div class="flex items-center gap-2">
        <span class="px-2 py-0.5 ${badgeClass} rounded text-xs">${badgeText}</span>
        <button class="change-domain-btn text-blue-400 hover:text-blue-300 underline" data-domain="${domain}" data-current="${preference}">
          ${changeText}
        </button>
      </div>
    `;
    domainList.appendChild(domainItem);
  });

  // Add event listeners to remove buttons
  domainList.querySelectorAll('.remove-domain-btn').forEach(btn => {
    btn.addEventListener('click', async (e) => {
      const domain = e.currentTarget.getAttribute('data-domain');
      await removeDomain(domain);
    });
  });

  // Add event listeners to change buttons
  domainList.querySelectorAll('.change-domain-btn').forEach(btn => {
    btn.addEventListener('click', async (e) => {
      const domain = e.currentTarget.getAttribute('data-domain');
      const currentPref = e.currentTarget.getAttribute('data-current');
      await changeDomainPreference(domain, currentPref);
    });
  });
}

/**
 * Change domain preference between 'always' and 'session'
 */
async function changeDomainPreference(domain, currentPref) {
  const result = await chrome.storage.local.get(['miniExtensionDomainPreferences']);
  const preferences = result.miniExtensionDomainPreferences || {};

  // Toggle preference
  const newPref = currentPref === 'always' ? 'session' : 'always';
  preferences[domain] = newPref;

  await chrome.storage.local.set({ miniExtensionDomainPreferences: preferences });

  console.log(`Changed ${domain} preference from ${currentPref} to ${newPref}`);

  // Reload the list
  await loadDomainList();

  const message = newPref === 'always'
    ? `${domain} will now always auto-deploy`
    : `${domain} will only auto-deploy this session`;
  showToast(message);
}

/**
 * Remove a domain from preferences
 */
async function removeDomain(domain) {
  const result = await chrome.storage.local.get(['miniExtensionDomainPreferences', 'miniExtensionDeployedDomains']);
  const preferences = result.miniExtensionDomainPreferences || {};
  const deployedDomains = result.miniExtensionDeployedDomains || {};

  // Remove from preferences
  delete preferences[domain];

  // Also clean up session data
  delete deployedDomains[domain];

  await chrome.storage.local.set({
    miniExtensionDomainPreferences: preferences,
    miniExtensionDeployedDomains: deployedDomains
  });

  console.log(`Removed domain ${domain} from preferences and session data`);

  // Reload the list
  await loadDomainList();

  showToast(`Removed ${domain} from auto-deploy`);
}

/**
 * Save settings to storage
 */
async function saveSettings() {
  // Get extension size from checked radio button
  const checkedRadio = document.querySelector('.extension-size-radio:checked');
  const extensionSize = checkedRadio ? parseFloat(checkedRadio.value) : 1.0;

  // Get settings from UI
  const settings = {
    autoSend: autoSendSetting.checked,
    silenceTimeout: parseInt(silenceTimeoutSetting.value) * 1000,
    transcriptionProvider: transcriptionProviderSetting.value,
    extensionSize: extensionSize
  };

  // Save settings
  await storageManager.setSettings(settings);

  // Apply extension size immediately
  applyExtensionSize();

  // Apply settings to services
  if (audioRecorder) {
    audioRecorder.silenceTimeout = settings.silenceTimeout;
  }

  // Hide settings panel
  settingsPanel.classList.add('hidden');

  // Show confirmation
  alert('Settings saved');
}

/**
 * Save only extension size setting (for immediate persistence)
 */
async function saveExtensionSizeOnly() {
  try {
    const checkedRadio = document.querySelector('.extension-size-radio:checked');
    const extensionSize = checkedRadio ? parseFloat(checkedRadio.value) : 1.0;

    const currentSettings = await storageManager.getSettings();
    currentSettings.extensionSize = extensionSize;
    await storageManager.setSettings(currentSettings);
    console.log('Extension size saved:', currentSettings.extensionSize);
  } catch (error) {
    console.error('Failed to save extension size:', error);
  }
}

/**
 * Load available formats from TextRad web app
 */
async function loadAvailableFormats() {
  try {
    console.log('Loading available formats...', { selectedTabId, textradTabs: textradTabs.length });

    // Show loading state
    if (templateLoading) {
      templateLoading.classList.remove('hidden');
    }
    if (templateError) {
      templateError.classList.add('hidden');
    }

    // Enhanced retry logic for race condition handling
    let retryCount = 0;
    const maxRetries = 3;

    while (!selectedTabId && retryCount < maxRetries) {
      console.log(`Template loading retry ${retryCount + 1}/${maxRetries} - attempting to find TextRad tabs...`);

      // Check if we have tabs but no selectedTabId (race condition)
      if (textradTabs.length > 0) {
        const loggedInTabs = textradTabs.filter(tab => tab.loggedIn);
        if (loggedInTabs.length > 0) {
          selectedTabId = loggedInTabs[0].id;
          console.log('Found logged-in tab during retry:', selectedTabId);
          break;
        }
      }

      // If no tabs available, trigger tab discovery and wait
      if (textradTabs.length === 0) {
        console.log('No tabs found, triggering fresh tab discovery...');
        await new Promise(resolve => {
          checkLoginStatus().then(() => {
            // Wait a bit for tab discovery to complete
            setTimeout(resolve, 500);
          });
        });
      }

      retryCount++;
      if (retryCount < maxRetries) {
        // Wait before retry
        await new Promise(resolve => setTimeout(resolve, 300));
      }
    }

    if (!selectedTabId) {
      throw new Error('No TextRad tab available after retries. Please ensure TextRad web app is open and you are logged in.');
    }

    console.log('Sending LOAD_FORMATS message to tab:', selectedTabId);

    // Load formats via background script
    const response = await chrome.runtime.sendMessage({
      action: 'LOAD_FORMATS',
      tabId: selectedTabId
    });

    if (response && response.success) {
      availableFormats = response.formats;
      populateFormatSelector(response.formats);
      console.log('Loaded', response.formats.length, 'formats');

      // Show template selection panel
      if (templateSelectionPanel) {
        templateSelectionPanel.classList.remove('hidden');
      }

      // Proactively warmup report generation service in background
      // This reduces latency when user clicks "Generate Report"
      warmupReportGenerationService();
    } else {
      throw new Error(response?.error || 'Failed to load formats');
    }
  } catch (error) {
    console.error('Error loading formats:', error);
    showTemplateError(error.message);
  } finally {
    // Hide loading state
    if (templateLoading) {
      templateLoading.classList.add('hidden');
    }
  }
}

/**
 * Populate the format selector dropdown
 */
function populateFormatSelector(formats) {
  if (!formatSearchInput || !formatDropdownList) return;

  // Store available formats
  availableFormats = formats;
  filteredFormats = [...formats];

  // Clear search input
  formatSearchInput.value = '';
  formatSearchInput.placeholder = `Type to search ${formats.length} formats...`;

  // Clear hidden selector value
  if (formatSelector) {
    formatSelector.value = '';
  }

  // Render dropdown items
  renderFormatDropdown();

  console.log('Populated format selector with', formats.length, 'formats');
}

/**
 * Handle format selection change (for direct programmatic selection)
 */
function handleFormatSelection(formatId = null) {
  if (!formatId && formatSelector) {
    formatId = formatSelector.value;
  }

  if (formatId) {
    // Find the selected format
    selectedFormat = availableFormats.find(f => f.id === formatId);

    if (selectedFormat) {
      console.log('Format selected:', selectedFormat.name);

      // Update UI
      formatSearchInput.value = selectedFormat.name;
      formatSelector.value = formatId;
      hideFormatDropdown();

      // Show generate report panel
      if (generateReportPanel) {
        generateReportPanel.classList.remove('hidden');
        updateContainerState();
      }
    } else {
      console.error('Selected format not found:', formatId);
    }
  } else {
    // No format selected
    selectedFormat = null;

    // Hide generate report panel
    if (generateReportPanel) {
      generateReportPanel.classList.add('hidden');
    }
  }
}

/**
 * Render the format dropdown items
 */
function renderFormatDropdown() {
  if (!formatDropdownList || !formatNoResults) return;

  // Clear existing items
  formatDropdownList.innerHTML = '';
  formatNoResults.classList.add('hidden');

  if (filteredFormats.length === 0) {
    formatNoResults.classList.remove('hidden');
    formatDropdownList.appendChild(formatNoResults);
    return;
  }

  // Create dropdown items
  filteredFormats.forEach((format, index) => {
    const item = document.createElement('div');
    item.className = 'format-dropdown-item';
    item.textContent = format.name;
    item.dataset.formatId = format.id;
    item.dataset.index = index;

    // Add click handlers - use both mousedown and click for reliability
    item.addEventListener('mousedown', (e) => {
      e.preventDefault(); // Prevent input blur
      isSelectingFormat = true;
    });

    item.addEventListener('click', (e) => {
      e.preventDefault();
      isSelectingFormat = true;
      selectFormat(format.id);
      // Reset flag after a short delay
      setTimeout(() => {
        isSelectingFormat = false;
      }, 100);
    });

    formatDropdownList.appendChild(item);
  });

  // Highlight first item by default
  selectedFormatIndex = 0;
  updateHighlight();
}

/**
 * Handle search input
 */
function handleFormatSearch(event) {
  const query = event.target.value.toLowerCase().trim();

  if (query === '') {
    filteredFormats = [...availableFormats];
  } else {
    filteredFormats = availableFormats.filter(format =>
      format.name.toLowerCase().includes(query)
    );
  }

  selectedFormatIndex = filteredFormats.length > 0 ? 0 : -1;
  renderFormatDropdown();
  showFormatDropdown();
}

/**
 * Handle keyboard navigation
 */
function handleFormatKeydown(event) {
  if (!formatDropdownList.classList.contains('hidden')) {
    switch (event.key) {
      case 'ArrowDown':
        event.preventDefault();
        if (selectedFormatIndex < filteredFormats.length - 1) {
          selectedFormatIndex++;
          updateHighlight();
        }
        break;

      case 'ArrowUp':
        event.preventDefault();
        if (selectedFormatIndex > 0) {
          selectedFormatIndex--;
          updateHighlight();
        }
        break;

      case 'Enter':
        event.preventDefault();
        if (selectedFormatIndex >= 0 && selectedFormatIndex < filteredFormats.length) {
          selectFormat(filteredFormats[selectedFormatIndex].id);
        }
        break;

      case 'Escape':
        event.preventDefault();
        hideFormatDropdown();
        formatSearchInput.blur();
        break;
    }
  }
}

/**
 * Handle format input click - always show dropdown if we have formats
 */
function handleFormatInputClick(e) {
  e.preventDefault();
  // Always show dropdown if we have formats (don't toggle)
  if (filteredFormats.length > 0) {
    showFormatDropdown();
  }
}


/**
 * Update highlighted item in dropdown
 */
function updateHighlight() {
  const items = formatDropdownList.querySelectorAll('.format-dropdown-item');
  items.forEach((item, index) => {
    if (index === selectedFormatIndex) {
      item.classList.add('highlighted');
    } else {
      item.classList.remove('highlighted');
    }
  });
}

/**
 * Select a format
 */
function selectFormat(formatId) {
  // Immediately hide dropdown to prevent flickering
  hideFormatDropdown();
  // Process the selection
  handleFormatSelection(formatId);
}

/**
 * Show format dropdown
 */
function showFormatDropdown() {
  if (formatDropdownList && filteredFormats.length > 0) {
    formatDropdownList.classList.remove('hidden');
    // Add expanded class for more space
    formatDropdownList.classList.add('format-dropdown-expanded');
    // Add container class to maintain height
    if (expandedContainer) {
      expandedContainer.classList.add('format-dropdown-active');
    }
    // Collapse transcription panel to make more space
    collapseTranscriptionPanel();
  }
}

/**
 * Hide format dropdown
 */
function hideFormatDropdown() {
  if (formatDropdownList) {
    formatDropdownList.classList.add('hidden');
    // Remove expanded class
    formatDropdownList.classList.remove('format-dropdown-expanded');
    // Remove container class to restore normal height
    if (expandedContainer) {
      expandedContainer.classList.remove('format-dropdown-active');
    }
    // Expand transcription panel back to normal
    expandTranscriptionPanel();
  }
}

// Flag to prevent dropdown from closing when clicking on items
let isSelectingFormat = false;

/**
 * Collapse transcription panel to make space for format dropdown
 */
function collapseTranscriptionPanel() {
  if (transcriptionPanel && !transcriptionPanel.classList.contains('hidden')) {
    transcriptionPanel.classList.add('transcription-panel-collapsed');
  }
}

/**
 * Expand transcription panel back to normal size
 */
function expandTranscriptionPanel() {
  if (transcriptionPanel) {
    transcriptionPanel.classList.remove('transcription-panel-collapsed');
  }
}

/**
 * Update container size based on current state
 */
function updateContainerState() {
  if (!expandedContainer) return;

  // Remove all state classes
  expandedContainer.classList.remove('has-transcription', 'has-report', 'report-expanded');

  // Check if transcription is visible and has content
  const hasTranscription = transcriptionPanel &&
    !transcriptionPanel.classList.contains('hidden') &&
    transcriptionText &&
    transcriptionText.value.trim() !== '';

  // Check if generate report panel is visible
  const hasReport = generateReportPanel &&
    !generateReportPanel.classList.contains('hidden');

  // Apply appropriate classes
  if (hasTranscription) {
    expandedContainer.classList.add('has-transcription');
  }

  if (hasReport) {
    expandedContainer.classList.add('has-report');
  }
}

/**
 * Expand report area for better viewing
 */
function expandReportArea() {
  if (expandedContainer && generateReportPanel) {
    expandedContainer.classList.add('report-expanded');
    generateReportPanel.classList.add('report-expanded');
    expandedContainer.classList.add('report-mode');

    // Directly set textarea height to override Tailwind classes
    if (transcriptionText) {
      transcriptionText.style.height = '280px';
      transcriptionText.style.minHeight = '280px';
      transcriptionText.style.transition = 'height 0.3s ease-in-out';
    }
  }
}

/**
 * Collapse report area back to normal
 */
function collapseReportArea() {
  if (expandedContainer && generateReportPanel) {
    expandedContainer.classList.remove('report-expanded', 'report-mode');
    generateReportPanel.classList.remove('report-expanded');

    // Reset textarea height back to normal
    if (transcriptionText) {
      transcriptionText.style.height = '';
      transcriptionText.style.minHeight = '';
    }
  }
}

/**
 * Generate report using selected format
 */
async function generateReport() {
  if (!currentTranscription || !selectedFormat) {
    alert('Please ensure you have transcription text and a selected format');
    return;
  }

  if (isGeneratingReport) {
    console.log('Report generation already in progress');
    return;
  }

  // Start report generation operation
  startOperation('generateReport');

  try {
    console.log('Generating report...', {
      formatId: selectedFormat.id,
      formatName: selectedFormat.name,
      transcriptionLength: currentTranscription.length
    });

    // Update UI to show generation in progress
    isGeneratingReport = true;
    updateGenerationUI(true);

    // Start dynamic message rotation for better user engagement
    startMessageRotation();

    // CRITICAL: Ensure TextRad tabs are active and services ready before report generation
    console.log('🔄 Ensuring TextRad tabs are active and ready...');
    try {
      // Message rotation will handle status updates automatically

      const activationResponse = await chrome.runtime.sendMessage({
        action: 'ACTIVATE_TEXTRAD_TABS'
      });

      if (activationResponse && activationResponse.success) {
        console.log(`✅ Successfully activated ${activationResponse.activatedTabs} TextRad tabs`);
        // Message rotation handles status - no need for static update

        // Optimized wait time - reduced from 2s to 1s for better performance
        // Only wait if tabs were actually activated (not already active)
        const waitTime = activationResponse.activatedTabs > 0 ? 1000 : 500;
        await new Promise(resolve => setTimeout(resolve, waitTime));
      } else {
        console.warn('Tab activation failed, proceeding anyway...');
        // Message rotation handles status - no need for static update
      }
    } catch (activationError) {
      console.warn('Tab activation error:', activationError);
      // Message rotation handles status - no need for static update
    }

    // Call background script to generate report with retry logic
    let reportResponse = null;
    let lastError = null;
    const maxRetries = 2;

    for (let attempt = 0; attempt <= maxRetries; attempt++) {
      try {
        if (attempt > 0) {
          // Show retry message briefly, then resume rotation
          if (generationStatus) {
            generationStatus.textContent = `Retrying... (attempt ${attempt + 1}/${maxRetries + 1})`;
          }
          console.log(`🔄 Retrying report generation (attempt ${attempt + 1}/${maxRetries + 1})`);

          // Wait longer between retries
          await new Promise(resolve => setTimeout(resolve, 3000));

          // Re-activate tabs before retry
          try {
            await chrome.runtime.sendMessage({ action: 'ACTIVATE_TEXTRAD_TABS' });
            await new Promise(resolve => setTimeout(resolve, 2000));
          } catch (retryActivationError) {
            console.warn('Retry tab activation failed:', retryActivationError);
          }
        }

        // Message rotation continues in background - no static status needed
        const response = await chrome.runtime.sendMessage({
          action: 'GENERATE_REPORT',
          transcription: currentTranscription,
          formatId: selectedFormat.id,
          tabId: selectedTabId
        });

        if (response && response.success) {
          reportResponse = response;
          break; // Success, exit retry loop
        } else {
          lastError = new Error(response?.error || 'Failed to generate report');

          // Check if this is a retryable error
          const errorMsg = lastError.message.toLowerCase();
          if (attempt < maxRetries && (
            errorMsg.includes('not available') ||
            errorMsg.includes('timeout') ||
            errorMsg.includes('suspended') ||
            errorMsg.includes('services not ready'))) {
            console.warn(`Attempt ${attempt + 1} failed with retryable error:`, lastError.message);
            continue; // Try again
          } else {
            throw lastError; // Non-retryable error or max retries reached
          }
        }
      } catch (error) {
        lastError = error;
        const errorMsg = error.message.toLowerCase();

        // Check if this is a retryable error
        if (attempt < maxRetries && (
          errorMsg.includes('not available') ||
          errorMsg.includes('timeout') ||
          errorMsg.includes('suspended') ||
          errorMsg.includes('services not ready') ||
          errorMsg.includes('timed out'))) {
          console.warn(`Attempt ${attempt + 1} failed with retryable error:`, error.message);
          continue; // Try again
        } else {
          throw error; // Non-retryable error or max retries reached
        }
      }
    }

    if (reportResponse && reportResponse.success) {
      // Update transcription with generated report
      currentReportContent = reportResponse.report;
      currentTranscription = reportResponse.report;

      // Expand report area for better viewing
      expandReportArea();

      // Update UI to show the generated report
      transcriptionText.value = reportResponse.report;
      updateUIForGeneratedReport();

      // Update HTML preview for formatted reports
      updateHtmlPreview();

      console.log('Report generated successfully, length:', reportResponse.report.length);
      showToast('Report generated successfully!', 2000);
    } else {
      throw lastError || new Error('Report generation failed after multiple attempts');
    }
  } catch (error) {
    console.error('Error generating report:', error);
    showToast('Failed to generate report: ' + error.message, 3000);
  } finally {
    // Stop message rotation
    stopMessageRotation();

    // Reset UI state
    isGeneratingReport = false;
    updateGenerationUI(false);

    // End report generation operation
    endOperation('generateReport');
  }
}

/**
 * Generate quick report using DirectTranscriptionService (no web app communication)
 */
async function generateQuickReport() {
  if (!currentTranscription) {
    alert('Please ensure you have transcription text');
    return;
  }

  if (isGeneratingReport) {
    console.log('Report generation already in progress');
    return;
  }

  // Start quick report generation operation
  startOperation('generateQuickReport');

  try {
    console.log('Generating quick report directly via DirectTranscriptionService...', {
      transcriptionLength: currentTranscription.length
    });

    // Update UI to show generation in progress
    isGeneratingReport = true;
    updateQuickGenerationUI(true);

    // DIRECT CALL - no background script, no content script, no web app communication
    const { DirectTranscriptionService } = await import('../lib/direct-transcription-service.js');
    const directService = new DirectTranscriptionService();

    // Set up progress callbacks
    directService.onReportGenerationProgress = (progress) => {
      updateQuickGenerationStatus(`Processing... ${Math.round(progress * 100)}%`);
    };

    directService.onReportGenerationStart = () => {
      updateQuickGenerationStatus('Starting quick report generation...');
    };

    // Get userId from content script (which can access page context)
    let userId = 'extension-user';
    try {
      if (selectedTabId) {
        const userIdResponse = await chrome.tabs.sendMessage(selectedTabId, {
          action: 'GET_USER_ID'
        });
        if (userIdResponse && userIdResponse.userId) {
          userId = userIdResponse.userId;
          console.log('📝 Got userId from content script:', userId);
        }
      }
    } catch (error) {
      console.warn('Could not get userId from content script:', error);
    }

    // Call quick report generation (mimics web app approach)
    const reportData = await directService.generateQuickReport(
      currentTranscription,
      userId
    );

    if (reportData && reportData.report) {
      // Update transcription with generated report
      currentReportContent = reportData.report;
      currentTranscription = reportData.report;

      // Expand report area for better viewing
      expandReportArea();

      // Update UI to show the generated report
      transcriptionText.value = reportData.report;
      updateUIForGeneratedReport();

      // Update HTML preview for formatted reports
      updateHtmlPreview();

      console.log('Quick report generated successfully, length:', reportData.report.length);
      showToast('Quick report generated successfully!', 2000);
    } else {
      throw new Error('No report content received from quick generation service');
    }
  } catch (error) {
    console.error('Error generating quick report:', error);
    showToast('Failed to generate quick report: ' + error.message, 3000);
  } finally {
    // Reset UI state
    isGeneratingReport = false;
    updateQuickGenerationUI(false);

    // End quick report generation operation
    endOperation('generateQuickReport');
  }
}

/**
 * Inject generated report into TextRad web app's report panel
 * Called automatically when copying a report
 */
async function injectReportToWebApp() {
  if (!currentReportContent && !currentTranscription) {
    console.warn('No report to inject');
    return false;
  }

  // Use the current report or transcription
  const reportToSend = currentReportContent || currentTranscription;

  try {
    // Ensure we have a selected tab
    if (!selectedTabId) {
      const tabs = await chrome.runtime.sendMessage({ action: 'GET_TEXTRAD_TABS' });
      if (!tabs || tabs.length === 0) {
        console.warn('No TextRad tab found for injection');
        return false;
      }
      selectedTabId = tabs[0].id;
    }

    console.log('📤 Injecting report to web app...', {
      tabId: selectedTabId,
      reportLength: reportToSend.length,
      formatName: selectedFormat?.name
    });

    // Send message to content script to inject report with auto-save enabled
    const response = await chrome.tabs.sendMessage(selectedTabId, {
      action: 'INJECT_REPORT',
      report: reportToSend,
      formatName: selectedFormat?.name || 'Quick Report',
      autoSave: true // Auto-trigger save button in web app
    });

    if (response && response.success) {
      console.log('✅ Report injected and saved in web app');
      return true;
    } else {
      console.warn('Failed to inject report:', response?.error);
      return false;
    }
  } catch (error) {
    console.error('❌ Error injecting report to web app:', error);
    return false;
  }
}

/**
 * Update generation UI state
 */
function updateGenerationUI(generating) {
  if (!generateButton || !generateButtonText || !generationProgress) return;

  if (generating) {
    generateButton.disabled = true;
    generateButtonText.textContent = 'Generating...';
    generationProgress.classList.remove('hidden');
    if (generationStatus) {
      generationStatus.textContent = 'Processing with TextRad AI...';
    }
  } else {
    generateButton.disabled = false;
    generateButtonText.textContent = 'Generate Report';
    generationProgress.classList.add('hidden');
  }
}

/**
 * Update quick generation UI state
 */
function updateQuickGenerationUI(generating) {
  if (!quickGenerateButton || !quickGenerateButtonText || !quickGenerationProgress) return;

  if (generating) {
    quickGenerateButton.disabled = true;
    quickGenerateButtonText.textContent = 'Generating...';
    quickGenerationProgress.classList.remove('hidden');
    if (quickGenerationStatus) {
      quickGenerationStatus.textContent = 'Processing with TextRad AI...';
    }
  } else {
    quickGenerateButton.disabled = false;
    quickGenerateButtonText.textContent = 'Quick Generate Report';
    quickGenerationProgress.classList.add('hidden');
  }
}

/**
 * Update generation status message
 */
function updateGenerationStatus(statusMessage) {
  if (generationStatus) {
    generationStatus.textContent = statusMessage;
    console.log('Generation status:', statusMessage);
  }
}

/**
 * Start dynamic message rotation during report generation
 */
function startMessageRotation() {
  // Clear any existing rotation
  stopMessageRotation();

  // Reset to first message
  currentMessageIndex = 0;

  // Show first message immediately
  if (generationStatus) {
    generationStatus.textContent = generationMessages[currentMessageIndex];
  }

  // Rotate messages every 2.5 seconds
  messageRotationInterval = setInterval(() => {
    currentMessageIndex = (currentMessageIndex + 1) % generationMessages.length;
    if (generationStatus) {
      generationStatus.textContent = generationMessages[currentMessageIndex];
    }
  }, 2500);
}

/**
 * Stop message rotation and clear interval
 */
function stopMessageRotation() {
  if (messageRotationInterval) {
    clearInterval(messageRotationInterval);
    messageRotationInterval = null;
  }
  currentMessageIndex = 0;
}

/**
 * Update quick generation status message
 */
function updateQuickGenerationStatus(statusMessage) {
  if (quickGenerationStatus) {
    quickGenerationStatus.textContent = statusMessage;
    console.log('Quick generation status:', statusMessage);
  }
}

/**
 * Proactively warmup report generation service
 * Called when format selector is shown to reduce generation latency
 */
async function warmupReportGenerationService() {
  try {
    console.log('🔥 Proactively warming up report generation service...');

    // Import DirectTranscriptionService to trigger warmup
    const { DirectTranscriptionService } = await import(chrome.runtime.getURL('lib/direct-transcription-service.js'));
    const directService = new DirectTranscriptionService();

    // The DirectTranscriptionService constructor automatically starts periodic warmup
    // and calls ensureWarmService() which triggers warmup if needed
    console.log('✅ Report generation service warmup initiated');
  } catch (error) {
    // Silently fail - warmup is optional optimization
    console.log('Warmup attempt failed (non-critical):', error.message);
  }
}

/**
 * Update UI after report is generated
 */
function updateUIForGeneratedReport() {
  // Update transcription panel header to indicate this is now a report
  const transcriptionHeader = transcriptionPanel.querySelector('h2');
  if (transcriptionHeader) {
    transcriptionHeader.innerHTML = '<i class="fas fa-file-medical mr-1"></i>Generated Report';
  }

  // Update copy button text
  if (copyButton) {
    copyButton.title = 'Copy & Save Report';
  }

  // Hide template selection and generate panels (report is complete)
  if (templateSelectionPanel) {
    templateSelectionPanel.classList.add('hidden');
  }
  if (generateReportPanel) {
    generateReportPanel.classList.add('hidden');
  }

  // Hide quick generate button since we now have a proper report
  if (quickGenerateContainer) {
    quickGenerateContainer.classList.add('hidden');
  }

  // Show a "New Dictation" option to reset the workflow
  showReportCompletionActions();
}

/**
 * Show actions available after report generation
 */
function showReportCompletionActions() {
  // The existing "New Dictation" button will reset everything
  // We could add additional actions here if needed
  console.log('Report generation complete - New Dictation button available for reset');
}

/**
 * Show template error message
 */
function showTemplateError(message) {
  if (templateError && templateErrorMessage) {
    templateErrorMessage.textContent = message;
    templateError.classList.remove('hidden');
  }

  console.error('Template error:', message);
}

/**
 * Reset template selection state (called by New Dictation)
 */
function resetTemplateState() {
  // Clear template-related state
  selectedFormat = null;
  currentReportContent = null;

  // Reset HTML preview state
  isShowingHtmlPreview = false;

  // Update HTML preview (will hide toggle)
  updateHtmlPreview();

  // Collapse report area if expanded
  collapseReportArea();

  // Reset UI elements
  if (formatSearchInput) {
    formatSearchInput.value = '';
    formatSearchInput.placeholder = 'Type to search formats...';
  }
  if (formatSelector) {
    formatSelector.value = '';
  }
  if (formatDropdownList) {
    formatDropdownList.innerHTML = '';
    formatDropdownList.classList.add('hidden');
  }
  availableFormats = [];
  filteredFormats = [];
  selectedFormatIndex = -1;

  // Hide template panels
  if (templateSelectionPanel) {
    templateSelectionPanel.classList.add('hidden');
  }
  if (generateReportPanel) {
    generateReportPanel.classList.add('hidden');
  }

  // Hide quick generate container
  if (quickGenerateContainer) {
    quickGenerateContainer.classList.add('hidden');
  }

  // Update container state after changes
  updateContainerState();
  if (templateError) {
    templateError.classList.add('hidden');
  }

  // Reset transcription panel header
  const transcriptionHeader = transcriptionPanel.querySelector('h2');
  if (transcriptionHeader) {
    transcriptionHeader.innerHTML = 'Transcription';
  }

  // Reset copy button
  if (copyButton) {
    copyButton.title = 'Copy Text';
  }

  console.log('Template state reset');
}

// Listen for messages from background script (mini extension triggers)
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('Popup received message:', message);

  if (message.action === 'MINI_EXTENSION_TRIGGER_RECORDING' && message.fromMiniExtension) {
    console.log('Triggering recording from mini extension...');

    // Wait for popup to be fully initialized
    setTimeout(() => {
      // Trigger the same recording function as the mini record button
      if (recordButton && !recordButton.disabled) {
        console.log('Clicking record button from mini extension trigger...');
        recordButton.click();
        sendResponse({ success: true, message: 'Recording triggered from mini extension' });
      } else if (miniRecordButton && !miniRecordButton.disabled) {
        console.log('Clicking mini record button from mini extension trigger...');
        miniRecordButton.click();
        sendResponse({ success: true, message: 'Recording triggered from mini extension via mini button' });
      } else {
        console.error('No record button available or all disabled');
        sendResponse({ success: false, message: 'No record button available' });
      }
    }, 300);

    return true; // Keep message channel open for async response
  }
});

/**
 * Toggle between markdown and HTML preview mode
 */
function toggleHtmlPreviewMode() {
  if (!currentReportContent) {
    // Only show preview for generated reports
    return;
  }

  isShowingHtmlPreview = !isShowingHtmlPreview;

  if (isShowingHtmlPreview) {
    // Switch to HTML preview
    const htmlContent = HTMLFormatter.convertMarkdownToHTML(
      transcriptionText.value,
      selectedFormat ? selectedFormat.name : null
    );
    htmlPreviewContent.innerHTML = htmlContent;

    // Preserve textarea dimensions by copying them to HTML preview container
    const textareaStyles = window.getComputedStyle(transcriptionText);
    htmlPreviewContainer.style.height = textareaStyles.height;
    htmlPreviewContainer.style.minHeight = textareaStyles.minHeight;
    htmlPreviewContainer.style.maxHeight = textareaStyles.maxHeight;

    // Show HTML preview, hide textarea
    htmlPreviewContainer.classList.remove('hidden');
    transcriptionText.classList.add('hidden');

    // Update toggle button
    previewToggleText.textContent = 'Markdown';
  } else {
    // Switch back to markdown text
    htmlPreviewContainer.classList.add('hidden');
    transcriptionText.classList.remove('hidden');

    // Update toggle button
    previewToggleText.textContent = 'HTML';
  }
}

/**
 * Update HTML preview when report content changes
 */
function updateHtmlPreview() {
  if (currentReportContent && HTMLFormatter.isMarkdownFormatted(transcriptionText.value)) {
    // Show HTML preview toggle for formatted reports
    if (toggleHtmlPreview) {
      toggleHtmlPreview.classList.remove('hidden');
    }

    // Default to HTML view for generated reports
    if (!isShowingHtmlPreview) {
      isShowingHtmlPreview = true;

      // Update toggle button text
      if (previewToggleText) {
        previewToggleText.textContent = 'Markdown';
      }
    }

    // Update HTML preview content and show it
    const htmlContent = HTMLFormatter.convertMarkdownToHTML(
      transcriptionText.value,
      selectedFormat ? selectedFormat.name : null
    );
    if (htmlPreviewContent) {
      htmlPreviewContent.innerHTML = htmlContent;
    }

    // Show HTML preview, hide textarea
    if (htmlPreviewContainer) {
      htmlPreviewContainer.classList.remove('hidden');

      // Preserve textarea dimensions by copying them to HTML preview container
      const textareaStyles = window.getComputedStyle(transcriptionText);
      htmlPreviewContainer.style.height = textareaStyles.height;
      htmlPreviewContainer.style.minHeight = textareaStyles.minHeight;
      htmlPreviewContainer.style.maxHeight = textareaStyles.maxHeight;
    }
    if (transcriptionText) {
      transcriptionText.classList.add('hidden');
    }
  } else {
    // Hide HTML preview toggle for regular transcriptions
    if (toggleHtmlPreview) {
      toggleHtmlPreview.classList.add('hidden');
    }

    // Switch back to markdown view if needed
    if (isShowingHtmlPreview) {
      isShowingHtmlPreview = false;
      if (htmlPreviewContainer) {
        htmlPreviewContainer.classList.add('hidden');
      }
      if (transcriptionText) {
        transcriptionText.classList.remove('hidden');
      }
      if (previewToggleText) {
        previewToggleText.textContent = 'HTML';
      }
    }
  }
}

/**
 * Auto-resize textarea based on content
 */
function autoResizeTextarea(textarea) {
  if (!textarea) return;

  // Reset height to auto to get the correct scrollHeight
  textarea.style.height = 'auto';

  // Calculate new height based on content
  const minHeight = 80; // matches CSS min-height
  const maxHeight = textarea.classList.contains('report-mode') ? 300 : 250; // Updated limits
  const newHeight = Math.min(Math.max(textarea.scrollHeight, minHeight), maxHeight);

  // Apply new height with smooth transition
  textarea.style.height = newHeight + 'px';

  // Update container if needed
  updateContainerForContent();
}

/**
 * Update container sizing based on content
 */
function updateContainerForContent() {
  const container = document.getElementById('expanded-container');
  const textarea = document.getElementById('transcription-text');

  if (!container || !textarea) return;

  // Determine if we have significant content
  const hasSignificantContent = textarea.value.length > 100;
  const isReportMode = container.classList.contains('has-report');

  if (hasSignificantContent && !isReportMode) {
    container.classList.add('has-transcription');
  } else if (!hasSignificantContent && !isReportMode) {
    container.classList.remove('has-transcription');
  }
}

// Initialize popup when DOM is loaded
document.addEventListener('DOMContentLoaded', initialize);